(function () {
    'use strict';

    var objectId = window.floorplanData.objectId || 0,
        objectInformationCache = {},
        $selectedObjectId = $('selectedObjectId').setValue(0),
        $selectedObjectPolygon = $('selectedObjectPolygon').setValue(''),
        selectedObjectId = 0,
        selectedObjectIsFloorplan = false,
        selectedObjectIsNested = false,
        floorplanObjectImages = [],
        state,
        editMode = false,
        $content = $('content'),
        $contentWrapper = $('contentWrapper'),
        $visualizationTop = $('floorplan-visualization-top'),
        $visualizationLeft = $('floorplan-visualization-left'),
        $visualizationLeftTabs = $('floorplan-visualization-left-tabs'),
        $visualizationLeftHeader = $visualizationLeft.down('.header'),
        $visualizationLeftContent = $('floorplan-visualization-left-content'),
        $visualizationLeftContentObjectInformation = $('floorplan-visualization-left-content-object-information'),
        $visualizationLeftFunctions = $('floorplan-visualization-left-functions'),
        $visualizationLeftLegend = $('floorplan-visualization-left-legend'),
        // Top buttons.
        $deleteButton = $('floorplanDeleteButton'),
        $editButton = $('floorplanEditButton'),
        $saveButton = $('floorplanSaveButton'),
        $cancelButton = $('floorplanCancelButton'),
        $profileSelect = $('floorplanProfile'),
        $changeLayoutButton = $('floorplan-layout-editor'),
        $changeBackgroundButton = $visualizationTop.down('[data-action="edit-background"]'),
        $setScaleButton = $visualizationTop.down('[data-action="set-scale"]'),
        $layerBrowserButton = $visualizationTop.down('[data-action="layer-browser"]'),
        $refreshButton = $visualizationTop.down('[data-action="refresh"]'),
        $gridSelect = $('floorplanGrid'),
        $gridDisplayToggle = $('floorplanGridDisplayToggle'),
        $gridSnapToggle = $('floorplanGridSnapToggle'),
        $radiusDisplayToggle = $('floorplanRadiusDisplayToggle'),
        $scaleToggle = $('floorplanScaleDisplayToggle'),
        userScaleDistance = 100,
        // Object specific formfields.
        $objectOrientationSelect = $('floorplan_object_orientation'),
        $objectWidthInput = $('floorplan_object_formfactor__width'),
        $objectHeightInput = $('floorplan_object_formfactor__height'),
        $objectDepthInput = $('floorplan_object_formfactor__depth'),
        $objectWHDUnitSelect = $('floorplan_object_formfactor__unit'),
        $objectWeightInput = $('floorplan_object_formfactor__weight'),
        $objectWeightUnitSelect = $('floorplan_object_formfactor__weight_unit'),
        // Function buttons
        $orientationContainer = $('floorplan-visualization-left-functions-orientation-container'),
        $polygonEditorPopupButton = $('floorplan_polygon_editor_popup'),
        $functionCenterObject = $visualizationLeftFunctions.down('[data-action="center-object"]'),
        $functionRotateClockwise = $visualizationLeftFunctions.down('[data-action="rotate-clockwise"]'),
        $functionRotateAntiClockwise = $visualizationLeftFunctions.down('[data-action="rotate-anticlockwise"]'),
        $functionUnpositionObject = $visualizationLeftFunctions.down('[data-action="unposition-object"]'),
        $functionDisplayRadius = $visualizationLeftFunctions.down('[data-action="edit-object-radius"]'),
        $functionInheritFormfactorData = $visualizationLeftFunctions.down('[data-action="inherit-formfactor-data"]'),
        $functionToggleObjectType = $visualizationLeftFunctions.down('[data-action="toggle-object-type"]'),
        // Location tree
        $locationTreeOptions = $('floorplan-visualization-left-locationtree-options'),
        $locationTreeContent = $('floorplan-visualization-left-locationtree-content'),
        // Canvas and overlay
        $canvas = $('floorplan-visualization-canvas'),
        $overlay = $('floorplan-visualization-overlay'),
        $objectFormfactorContainer = $('floorplan_object_formfactor_container'),
        $measurementOverlay = $('floorplan-visualization-measurement'),
        $backgroundImageUploadOverlay = $('floorplan-visualization-background-popup'),
        $backgroundImageGallery = $('floorplan-visualization-background-gallery'),
        $layerBrowser = $('floorplan-visualization-layer-browser'),
        // Floorplan instance and data.
        floorplan,
        floorplanLayer,
        floorplanTree,
        objectTypeData = window.floorplanData.objectTypeData,
        authEdit = window.floorplanData.authEdit,
        authDelete = window.floorplanData.authDelete;

    // @see  FP-45
    if ($functionDisplayRadius) {
        $functionDisplayRadius.on('click', function () {
            if (selectedObjectId > 0) {
                var objectData = floorplan.getData(selectedObjectId),
                    popupData  = {
                        objectTypeColor: FloorplanHelper.getObjectType(objectData.objType).color,
                        polygon:         objectData.polygon || null,
                        radius:          objectData.radius || {}
                    };

                get_popup('floorplan_object_radius', '', 700, 500, {popupData: btoa(JSON.stringify(popupData))});
            }
        });
    }

    if ($visualizationTop) {
        // We need to listen to "window resizing" and dragging the navbar.
        Event.observe(window, 'resize', responsive_options);
        idoit.callbackManager
            .registerCallback('idoit-dragbar-update', responsive_options)
            .triggerCallback('idoit-dragbar-update');

        if ($changeBackgroundButton) {
            $backgroundImageUploadOverlay.on('update:galleryItems', function () {
                var $selectedImage = $backgroundImageGallery.down('img[data-image-id="' + floorplan.getOption('backgroundImageID') + '"]');

                if (!$selectedImage) {
                    $selectedImage = $backgroundImageGallery.down('img[src="' + floorplan.getOption('backgroundImage') + '"]');
                }

                // Here we'd like to preselect the currently active background image.
                if ($selectedImage) {
                    $selectedImage.up().addClassName('selected');
                }

                // Now we set the margins, so our thumb will float in the center.
                $backgroundImageGallery.select('img.background-image').each(function ($img) {
                    $img.setStyle({
                        marginLeft:((144 - $img.getWidth()) /2) + 'px',
                        marginTop:((144 - $img.getHeight()) /2) + 'px'
                    });
                });
            });

            $backgroundImageUploadOverlay.on('click', 'img.background-image', function (ev) {
                var $item = ev.findElement('img');

                $backgroundImageGallery.select('.selected').invoke('removeClassName', 'selected');

                $item.up('div').addClassName('selected');
            });

            $backgroundImageUploadOverlay.on('click', 'img.delete-background', function (ev) {
                var i,
                    $imageContainer = ev.findElement('div'),
                    imageId = $imageContainer.down('img.background-image').readAttribute('data-image-id');

                if (confirm(idoit.Translate.get('LC__MODULE__FLOORPLAN__VISUALIZATION__BACKGROUND_IMAGE_REMOVE_CONFIRM'))) {
                    for (i in floorplanObjectImages) {
                        if (floorplanObjectImages.hasOwnProperty(i) && floorplanObjectImages[i] == imageId) {
                            delete floorplanObjectImages[i];
                        }
                    }

                    new Ajax.Request(window.www_dir + 'floorplan/ajax/removeBackgroundImage/' + imageId, {
                        parameters: {},
                        onComplete: function (response) {
                            var json = response.responseJSON;

                            if (!json) {
                                idoit.Notify.error(response.responseText, {sticky: true});
                                return;
                            }

                            if (json && !json.success) {
                                idoit.Notify.error(json.message, {sticky: true});
                                return;
                            }

                            $imageContainer.remove();
                        }
                    });
                }
            });

            $backgroundImageUploadOverlay.on('click', '[data-action="accept"]', function () {
                var $img = $backgroundImageGallery.down('.selected img.background-image');

                $overlay.addClassName('hide');
                $backgroundImageUploadOverlay.addClassName('hide');

                if ($img) {
                    floorplan
                        .setOption('backgroundImage', null)
                        .setOption('backgroundImageID', $img.readAttribute('data-image-id'))
                        .setOption('backgroundImageSize', [$img.naturalWidth, $img.naturalHeight])
                        .resetBackgroundImage();
                } else {
                    floorplan
                        .setOption('backgroundImage', null)
                        .setOption('backgroundImageID', null)
                        .setOption('backgroundImageSize', null)
                        .resetBackgroundImage();
                }
            });

            $backgroundImageUploadOverlay.on('click', '[data-action="remove"]', function () {
                floorplan
                    .setOption('backgroundImage', null)
                    .setOption('backgroundImageID', null)
                    .setOption('backgroundImageSize', null)
                    .resetBackgroundImage();

                $overlay.addClassName('hide');
                $backgroundImageUploadOverlay.addClassName('hide');
            });

            $backgroundImageUploadOverlay.on('click', '[data-action="abort"]', function () {
                $overlay.addClassName('hide');
                $backgroundImageUploadOverlay.addClassName('hide');
            });

            $changeBackgroundButton.on('click', function () {
                var i, $img;

                if ($backgroundImageGallery.select('div').length === 0 && floorplanObjectImages.length !== 0) {
                    // There are no image-DIVs but the floorplan object has a filled "images" category. Refresh!
                    for (i in floorplanObjectImages) {
                        if (floorplanObjectImages.hasOwnProperty(i)) {
                            $img = new Element('img', {className:'background-image'});

                            $img.on('load', function (ev) {
                                var $image = ev.findElement('img');
                                $image.setStyle({
                                    marginLeft:((144 - $image.getWidth()) /2) + 'px',
                                    marginTop:((144 - $image.getHeight()) /2) + 'px'
                                });
                            });

                            $backgroundImageGallery
                                .insert(new Element('div')
                                    .update(new Element('img', {src:window.dir_images + 'axialis/basic/symbol-cancel.svg', className:'delete-background', title:idoit.Translate.get('LC_UNIVERSAL__DELETE')}))
                                    .insert($img.writeAttribute({
                                        src: window.www_dir + 'floorplan/visualization/backgroundImage/' + floorplanObjectImages[i],
                                        'data-image-id': floorplanObjectImages[i]
                                    })));
                        }
                    }
                }

                // Because this is a big modal popup, we hide all the other ones.
                $measurementOverlay.down('[data-action="abort"]').simulate('click');

                // Then we display the overlay and our upload-popup.
                $overlay.removeClassName('hide');
                $backgroundImageUploadOverlay.removeClassName('hide');

                // After we display the popup we update all the gallery images (margins etc).
                $backgroundImageUploadOverlay.fire('update:galleryItems');
            });
        }

        if ($setScaleButton && $measurementOverlay) {
            $measurementOverlay.down('[data-action="accept"]').on('click', function () {
                const userInputAmount = parseFloat($F('floorplan_measurement_count'));
                const userInputUnit = parseFloat($F('floorplan_measurement_unit'));

                // Here set the user scale - this defines how many pixels equal 1mm.
                floorplan.setUserScale(userScaleDistance / (userInputAmount * userInputUnit));
                floorplan.setOption('scalePosition', $F('floorplan_measurement_display'));
                floorplan.drawScale();

                floorplan.cancelScaleMeasurement();
                $measurementOverlay.addClassName('hide');
            });

            $measurementOverlay.down('[data-action="abort"]').on('click', function () {
                floorplan.cancelScaleMeasurement();
                $measurementOverlay.addClassName('hide');
            });

            $setScaleButton.on('click', function () {
                if (confirm(idoit.Translate.get('LC__MODULE__FLOORPLAN__VISUALIZATION__SET_FLOORPLAN_SCALE_CONFIRM'))) {
                    floorplan.initializeScaleMeasurement(function(data) {
                        userScaleDistance = FloorplanHelper.distanceBetweenPoints([data[0].x,data[0].y], [data[1].x,data[1].y]);
                    });

                    // Set default values (scale: top left, 1000mm (equals 1m)):
                    $('floorplan_measurement_display').setValue(floorplan.getOptions().scalePosition || 'nw');
                    $('floorplan_measurement_count').setValue(1000);
                    $('floorplan_measurement_unit').setValue(1);

                    $measurementOverlay.removeClassName('hide');
                }
            });
        }

        if ($refreshButton) {
            $refreshButton.on('click', function () {
                $refreshButton
                    .down('img')
                    .writeAttribute('src', window.dir_images + 'ajax-loading.gif');

                initializeFloorplan(objectId);
            })
        }
    }

    if ($visualizationLeftFunctions) {
        const rotateObject = function (object, degrees) {
            const data = floorplan.getData(object);

            data.angle = (data.angle + degrees) % 360;

            if (data.angle < 0) {
                data.angle += 360;
            }

            floorplan.updateObjectPositions(object, true);
        };

        $functionRotateClockwise.on('click', function () {
            rotateObject(selectedObjectId, 45);
        });

        $functionRotateAntiClockwise.on('click', function () {
            rotateObject(selectedObjectId, -45);
        });

        $functionCenterObject.on('click', function () {
            floorplan.center(selectedObjectId);
        });

        $functionUnpositionObject.on('click', function () {
            if (selectedObjectId > 0)
            {
                floorplan.removeObject({objectId: selectedObjectId});

                floorplanTree
                    .setPositionedObjects(floorplanTree
                        .getPositionedObjects()
                        .filter(function (d) {
                            return selectedObjectId != d;
                        }))
                    .process();
            }
        });

        $functionInheritFormfactorData.on('click', function () {
            if (confirm(idoit.Translate.get('LC__MODULE__FLOORPLAN__VISUALIZATION__INHERIT_FORMFACTOR_DATA_CONFIRM'))) {
                const data = floorplan.getData(selectedObjectId);
                const scaleFactor = floorplan.getUserScale(false);
                const formfactor = objectInformationCache['o' + selectedObjectId].information.formfactor;

                if (!formfactor.width || !formfactor.depth || isNaN(+formfactor.width) || isNaN(+formfactor.depth)) {
                    idoit.Notify.warning(idoit.Translate.get('LC__MODULE__FLOORPLAN__VISUALIZATION__FORMFACTOR_VALUES_INCORRECT'), {life: 10});
                    return;
                }

                if (data.orientation === 'n' || data.orientation === 's') {
                    data.width = formfactor.width * formfactor.depthFactor * scaleFactor;
                    data.height = formfactor.depth * formfactor.depthFactor * scaleFactor;
                } else {
                    data.height = formfactor.width * formfactor.depthFactor * scaleFactor;
                    data.width = formfactor.depth * formfactor.depthFactor * scaleFactor;
                }

                // Set a minimum width/height of 16.
                data.width = Math.max(data.width, 16);
                data.height = Math.max(data.height, 16);

                floorplan.updateObjectPositions(selectedObjectId, true);
            }
        });

        $functionToggleObjectType.on('click', function () {
            var $checkbox = $visualizationLeftLegend.down('ul input[value="' + objectInformationCache['o' + selectedObjectId].object.objTypeId + '"]');

            if ($checkbox) {
                $checkbox.setValue($checkbox.checked ? '' : '1').simulate('change');
            }
        });
    }

    if ($locationTreeOptions) {
        $locationTreeOptions.on('change', 'input', function (ev) {
            var treeMode     = ev.findElement('input').getValue(),
                rootObjectId = window.floorplanData.rootObjectId;

            if (treeMode === 'floorplan-objects') {
                rootObjectId = objectId;
            }

            floorplanTree
                .setMode(treeMode)
                .setRootNodeId(rootObjectId)
                .process();
        });
    }

    if ($visualizationLeftContent) {
        $content.select('.no-enter').invoke('on', 'keydown', function (ev) {
            var $input = ev.findElement('input');

            // Prevent the "enter" from reloading the page.
            if (ev.keyCode == Event.KEY_RETURN) {
                ev.preventDefault();
            }

            $input.store('val', $input.getValue());
        });

        $content.select('.only-numeric').invoke('on', 'keyup', function (ev) {
            var $input = ev.findElement('input'),
                val = $input.getValue().replace(',', '.').replace(/(?![\d\.])./g, '').split('.'),
                tmp = val.pop();

            if ($input.retrieve('val') != $input.getValue()) {
                if (val.length) {
                    $input.setValue(val.join('') + '.' + tmp);
                } else {
                    $input.setValue(tmp);
                }
            }
        });

        $visualizationLeftContent.down('[data-action="save-formfactor-data"]').on('click', function () {
            if (confirm((idoit.Translate.get('LC__MODULE__FLOORPLAN__VISUALIZATION__SAVE_FORMFACTOR_DATA_CONFIRM').replace('%s', $visualizationLeftContent.down('.obj-title').innerHTML)))) {
                new Ajax.Request(window.www_dir + 'floorplan/ajax/saveFormfactorData/' + selectedObjectId, {
                    parameters: {
                        width: $objectWidthInput.getValue(),
                        height: $objectHeightInput.getValue(),
                        depth: $objectDepthInput.getValue(),
                        unit: $objectWHDUnitSelect.getValue(),
                        weight: $objectWeightInput.getValue(),
                        weight_unit: $objectWeightUnitSelect.getValue()
                    },
                    onComplete: function (response) {
                        var json = response.responseJSON;

                        if (!json) {
                            idoit.Notify.error(response.responseText, {sticky: true});
                            return;
                        }

                        if (json && !json.success) {
                            idoit.Notify.error(json.message, {sticky: true});
                            return;
                        }

                        delete objectInformationCache['o' + selectedObjectId];
                        $visualizationLeftContent.fire('load:objectInformation');
                    }
                });
            }
        });

        $visualizationLeftContent.on('load:objectInformation', function () {
            var i, $objectInfoTable;

            $visualizationLeftContent.addClassName('opacity-50');
            $objectFormfactorContainer.addClassName('hide');

            if (objectInformationCache.hasOwnProperty('o' + selectedObjectId)) {
                $visualizationLeftContent.down('table')
                    .update(new Element('tr')
                        .update(new Element('td')
                            .writeAttribute('class', 'vat')
                            .writeAttribute('width', '105px')
                            .update(new Element('img')
                                .writeAttribute('src', objectInformationCache['o' + selectedObjectId].object.objImage)
                                .writeAttribute('class', 'object-image pr5')))
                        .insert(new Element('td')
                            .writeAttribute('class', 'vat')
                            .update(new Element('div')
                                .writeAttribute('class', 'cmdb-marker')
                                .setStyle({backgroundColor: objectInformationCache['o' + selectedObjectId].object.objTypeColor}))
                            .insert(new Element('strong')
                                .writeAttribute('class', 'obj-title')
                                .update(objectInformationCache['o' + selectedObjectId].object.objTypeTitle + ' &raquo; ' + objectInformationCache['o' + selectedObjectId].object.objTitle))
                            .insert(new Element('hr')
                                .writeAttribute('class', 'mt5 mb5'))
                            .insert(new Element('p')
                                .writeAttribute('class', 'mb10')
                                .update(new Element('div')
                                    .writeAttribute('class', 'cmdb-marker')
                                    .setStyle({backgroundColor: objectInformationCache['o' + selectedObjectId].object.cmdbColor}))
                                .insert(new Element('span')
                                    .update(objectInformationCache['o' + selectedObjectId].object.cmdbStatus)))));

                if (objectInformationCache['o' + selectedObjectId].information.hasOwnProperty('formfactor')) {
                    /*
                     * We CAN size an object by it's formfactor data if:
                     * 1) The width and depth is documented
                     * 2) The floorplan scale has been defined
                     */
                    if (objectInformationCache['o' + selectedObjectId].information.formfactor.width > 0 && objectInformationCache['o' + selectedObjectId].information.formfactor.depth > 0) {
                        $functionInheritFormfactorData
                            .writeAttribute('title', null)
                            .enable();
                    }

                    $objectWidthInput.setValue(objectInformationCache['o' + selectedObjectId].information.formfactor.width);
                    $objectHeightInput.setValue(objectInformationCache['o' + selectedObjectId].information.formfactor.height);
                    $objectDepthInput.setValue(objectInformationCache['o' + selectedObjectId].information.formfactor.depth);
                    $objectWHDUnitSelect.setValue(objectInformationCache['o' + selectedObjectId].information.formfactor.depthUnitId);

                    $visualizationLeftContent.down('td', 1).setStyle({position:'relative'})
                        .insert(new Element('button', { type: 'button', class:'btn btn-secondary formfactor-edit ' + (editMode ? '' : 'hide')})
                            .update(new Element('img', { src: window.dir_images + 'axialis/documents-folders/document-edit.svg'})))
                        .insert(
                        new Element('table', {width:'100%'}).update(
                            new Element('tr').update(
                                new Element('td', {className:'right'}).update(idoit.Translate.get('LC__MODULE__FLOORPLAN__WHD'))
                            ).insert(
                                new Element('td', {className:'pl10 bold'}).update(objectInformationCache['o' + selectedObjectId].information.formfactor.width + ' &times; ' + objectInformationCache['o' + selectedObjectId].information.formfactor.height + ' &times; ' + objectInformationCache['o' + selectedObjectId].information.formfactor.depth + ' ' + objectInformationCache['o' + selectedObjectId].information.formfactor.depthUnit)
                            )
                        ).insert(
                            new Element('tr').update(
                                new Element('td', {className:'right'}).update(idoit.Translate.get('LC__MODULE__FLOORPLAN__WEIGHT'))
                            ).insert(
                                new Element('td', {className:'pl10 bold'}).update(objectInformationCache['o' + selectedObjectId].information.formfactor.weight + ' ' + objectInformationCache['o' + selectedObjectId].information.formfactor.weightUnit)
                            )
                        )
                    );

                    $visualizationLeftContent.down('.formfactor-edit').on('click', function () {
                        $objectFormfactorContainer.toggleClassName('hide');
                    });
                }

                $visualizationLeftContentObjectInformation.update();

                // If we receive "object-information" data, we'll display this data in form of a nice table.
                if (objectInformationCache['o' + selectedObjectId].hasOwnProperty('properties') && Object.isArray(objectInformationCache['o' + selectedObjectId].properties) && objectInformationCache['o' + selectedObjectId].properties.length)
                {
                    $objectInfoTable = new Element('table')
                        .writeAttribute('class', 'w100 mainTable pb0 mb10 border-left border-right border-bottom');

                    for (i in objectInformationCache['o' + selectedObjectId].properties)
                    {
                        if (!objectInformationCache['o' + selectedObjectId].properties.hasOwnProperty(i))
                        {
                            continue;
                        }

                        $objectInfoTable.insert(
                            new Element('tr').update(
                                new Element('td').update(objectInformationCache['o' + selectedObjectId].properties[i][0])
                            ).insert(
                                new Element('td').update(objectInformationCache['o' + selectedObjectId].properties[i][1])
                            ));
                    }

                    $visualizationLeftContentObjectInformation
                        .insert(new Element('h5')
                            .writeAttribute('class', 'bg-neutral-200 p5 border')
                            .update(idoit.Translate.get('LC__MODULE__FLOORPLAN__VISUALIZATION__OBJECT_INFORMATION')))
                        .insert($objectInfoTable);
                }

                $visualizationLeftContent.removeClassName('opacity-50').removeClassName('hide');
            } else {
                new Ajax.Request(window.www_dir + 'floorplan/ajax/getObjectInformation/' + selectedObjectId, {
                    parameters: {
                        profileId: $profileSelect.getValue()
                    },
                    onComplete: function (response) {
                        var json = response.responseJSON;

                        if (! json) {
                            idoit.Notify.error(response.responseText, {sticky:true});
                            return;
                        }

                        if (json && !json.success) {
                            idoit.Notify.error(json.message, {sticky:true});
                            return;
                        }

                        objectInformationCache['o' + selectedObjectId] = json.data;

                        $visualizationLeftContent.fire('load:objectInformation');
                    }
                });
            }
        });
    }

    // @see FP-3  Handle grid changes
    if ($gridSelect) {
        $gridSelect.on('change', function () {
            floorplan
                .setOption('grid', window.floorplanData.scaleFactors[$gridSelect.getValue()])
                .setOption('measurementUnitName', $gridSelect.down('option:selected').innerText)
                .updateGrid()
                .drawScale();
        });
    }

    // @see FP-3  Display optional grid.
    if ($gridDisplayToggle) {
        $gridDisplayToggle.on('click', function () {
            floorplan
                .setOption('gridDisplay', $gridDisplayToggle.toggleClassName('pressed').hasClassName('pressed'))
                .updateGrid();
        });
    }

    // @see FP-3  Snap to optional grid.
    if ($gridSnapToggle) {
        $gridSnapToggle.on('click', function () {
            floorplan
                .setOption('snapToGrid', $gridSnapToggle.toggleClassName('pressed').hasClassName('pressed'));
        });
    }

    // @see FP-45  Radii can be toggled.
    if ($radiusDisplayToggle) {
        $radiusDisplayToggle.on('click', function () {
            floorplan
                .setOption('radiusDisplay', $radiusDisplayToggle.toggleClassName('pressed').hasClassName('pressed'))
                .updateObjectPositions();
        });
    }

    if ($scaleToggle) {
        $scaleToggle.on('click', function () {
            floorplan
                .setOption('scaleShow', $scaleToggle.toggleClassName('pressed').hasClassName('pressed'))
                .drawScale();
        });
    }

    if ($visualizationLeftLegend) {
        $visualizationLeftLegend.down('h3 input').on('change', function (ev) {
            var $checkboxes = $visualizationLeftLegend.down('ul').select('input[type="checkbox"]'),
                checked = ev.findElement('input').checked;

            $checkboxes.invoke('setValue', (checked ? 1 : null));

            if (! checked) {
                floorplan
                    .setOption('objectTypeFilter', $checkboxes.invoke('readAttribute', 'value').map(function(val) { return parseInt(val); }))
                    .updateObjectPositions();
            } else {
                floorplan.setOption('objectTypeFilter', []).updateObjectPositions();
            }
        });

        $visualizationLeftLegend.down('ul').on('change', 'input[type="checkbox"]', function () {
            var $checkboxes = $visualizationLeftLegend
                .down('ul')
                .select('input[type="checkbox"]')
                .filter(function ($input) { return !$input.checked; });

            floorplan
                .setOption('objectTypeFilter', $checkboxes.invoke('readAttribute', 'value').map(function(val) { return parseInt(val); }))
                .updateObjectPositions();
        });
    }

    if ($objectOrientationSelect) {
        $objectOrientationSelect.on('change', function () {
            var orientation = $objectOrientationSelect.getValue();

            if (orientation === '-1') {
                orientation = null;
            }

            floorplan.updateData(selectedObjectId, {orientation: orientation});

            // @see  FP-43 Don't save automatically
        });
    }

    function saveState () {
        // We use "JSON.parse(JSON.stringify(...))" to remove all callbacks and simultaneously creating a copy.
        state = {
            floorplan: {
                options: JSON.parse(JSON.stringify(floorplan.getOptions())),
                data: JSON.parse(JSON.stringify(floorplan.getAllData().map(function(d) {
                    const copy = Object.assign({}, d);
                    delete copy.visualizationType;
                    return copy;
                }))),
                layers: JSON.parse(JSON.stringify(floorplan.getAllLayers()))
            },
            tree: {
                positionedObjects: floorplanTree.getPositionedObjects()
            }
        };

        // Remove some leftovers...
        delete state.floorplan.options.objectTypeData;
        delete state.floorplan.options.undefinedObjectType;
        delete state.floorplan.options.authEdit;
        delete state.floorplan.options.width;
        delete state.floorplan.options.height;
        delete state.floorplan.options.minZoomLevel;
        delete state.floorplan.options.maxZoomLevel;
    }

    function restoreState () {
        floorplan
            .preProcess()
            .setLayers(state.floorplan.layers)
            .setData(state.floorplan.data)
            .extendOptions(state.floorplan.options)
            .resetBackgroundImage()
            .redrawBackground()
            .updateGrid()
            .drawScale()
            .process();

        floorplanLayer
            .setObjects(state.floorplan.data)
            .setLayers(state.floorplan.layers)
            .process()

        floorplanTree
            .setPositionedObjects(state.tree.positionedObjects)
            .process();

        for (i in window.floorplanData.scaleFactors) {
            if (! window.floorplanData.scaleFactors.hasOwnProperty(i)) {
                continue;
            }

            if (window.floorplanData.scaleFactors[i] == state.floorplan.options.grid) {
                $gridSelect.setValue(i);
            }
        }

        $gridDisplayToggle[state.floorplan.options.gridDisplay ? 'removeClassName' : 'addClassName']('pressed');
        $gridSnapToggle[state.floorplan.options.snapToGrid ? 'removeClassName' : 'addClassName']('pressed');
        $radiusDisplayToggle[state.floorplan.options.radiusDisplay ? 'removeClassName' : 'addClassName']('pressed');
        $scaleToggle[state.floorplan.options.scaleShow ? 'addClassName' : 'removeClassName']('pressed');
    }

    function enterViewMode() {
        editMode = false;

        processFloorplanControls();

        floorplanTree.setEditMode(false);
        floorplanLayer.setEditMode(false);
        floorplan.setEditMode(false);
    }

    function enterEditMode() {
        saveState();

        editMode = true;

        processFloorplanControls();

        floorplanTree.setEditMode(true);
        floorplanLayer.setEditMode(true);
        floorplan.setEditMode(true);
    }

    if ($deleteButton) {
        $deleteButton.on('click', function () {
            if (authDelete) {
                deleteFloorplan();
            } else {
                idoit.Notify.info(idoit.Translate.get('LC__MODULE__FLOORPLAN__AUTH__MISSING_DELETION_RIGHT'), {sticky: true});
            }
        });
    }

    if ($editButton) {
        $editButton.on('click', function () {
            if (authEdit) {
                enterEditMode();
            } else {
                idoit.Notify.info(idoit.Translate.get('LC__MODULE__FLOORPLAN__AUTH__MISSING_VISUALIZATION_RIGHT'), {sticky: true});
            }
        });
    }

    if ($saveButton) {
        $saveButton.on('click', function () {
            saveState();
            saveFloorplan();
            restoreState();
            enterViewMode();
        });
    }

    if ($cancelButton) {
        $cancelButton.on('click', function () {
            restoreState();
            enterViewMode();
        });
    }

    $profileSelect.on('change', function () {
        var selectedProfile = null,
            $checkboxes = $visualizationLeftLegend.select('ul input'),
            i;

        for (i in window.floorplanData.profiles) {
            if (!window.floorplanData.profiles.hasOwnProperty(i)) {
                continue;
            }

            if (window.floorplanData.profiles[i].id == $profileSelect.getValue()) {
                selectedProfile = window.floorplanData.profiles[i];
            }
        }

        floorplan.setProfile(selectedProfile);

        $checkboxes.invoke('setValue', 1);

        if (selectedProfile && selectedProfile.hasOwnProperty('config')) {
            if (selectedProfile.config.hasOwnProperty('objectTypes') && Object.isArray(selectedProfile.config.objectTypes)) {
                $checkboxes
                    .filter(function($checkbox) { return selectedProfile.config.objectTypes.in_array($checkbox.readAttribute('value')); })
                    .invoke('setValue', 0);
            }

            if (selectedProfile.config.hasOwnProperty('displayRadius')) {
                $radiusDisplayToggle[selectedProfile.config.displayRadius == 1 ? 'addClassName' : 'removeClassName']('pressed').simulate('click');
            }
        }


        floorplan
            .setOption('objectTypeFilter', $visualizationLeftLegend.select('ul input:not(:checked)').invoke('readAttribute', 'value').map(function(val) { return parseInt(val); }))
            .updateObjectPositions();

        objectInformationCache = {};

        if (selectedObjectId > 0) {
            $visualizationLeftContent.fire('load:objectInformation');
        }
    });

    $profileSelect.on('update:selection', function () {
        var previouslySelectedProfile = $profileSelect.getValue(),
            $optGroup = $profileSelect.down('optgroup'),
            i;

        $optGroup.update();

        for (i in window.floorplanData.profiles) {
            if (!window.floorplanData.profiles.hasOwnProperty(i)) {
                continue;
            }

            $optGroup.insert(new Element('option')
                .writeAttribute('value', window.floorplanData.profiles[i].id)
                .update(window.floorplanData.profiles[i].title));
        }


        $profileSelect.setValue(previouslySelectedProfile);
        $profileSelect.simulate('change');
    });

    // Create a "responsive_options" method to use as callback, when anything changes.
    function responsive_options () {
        var $svg = $canvas.down('svg'),
            canvasHeight = $canvas.getHeight();

        // One last thing: If changing the height of the browser window, we need to change the $visualizationLeftContent aswell.
        $visualizationLeftContent.setStyle({height: ($visualizationLeft.getHeight() - ($visualizationLeftTabs.getHeight() + $visualizationLeftHeader.getHeight() + $visualizationLeftFunctions.getHeight() + $visualizationLeftLegend.getHeight())) + 'px'});

        // We can not use "$svg.hasClassName(...)" because Prototype 1.7 can't handle SVG elements :(
        if ($svg) {
            $svg.setAttribute('width', $canvas.getWidth());
            $svg.setAttribute('height', canvasHeight);
        }

        if (floorplan) {
            floorplan.responsive();
        }
    }

    // @see FP-112 Unmark all menus.
    $('mainMenu').select('li').invoke('removeClassName', 'active');

    if ($('mainMenu').down('li.add-ons')) {
        // Mark the "add-on" menu, if it exists.
        $('mainMenu').down('li.add-ons').addClassName('active');
    } else if ($('mainMenu').down('li.extras')) {
        // Otherwise mark the "extras" menu, if it exists.
        $('mainMenu').down('li.extras').addClassName('active');
    }

    // Display the visualization GUI above the navbar.
    if ($contentWrapper) {
        $contentWrapper.setStyle({ top: '40px' });
    }

    window.initializeFloorplan = function initializeFloorplan (objId) {
        const $breadCrumb = $('breadcrumb');

        objectId = objId;
        $overlay.removeClassName('hide');
        floorplanObjectImages = [];
        $backgroundImageGallery.update();

        if (objId > 0) {
            var $breadCrumbItem = $breadCrumb.down('.fp-object');

            if (!$breadCrumbItem) {
                $breadCrumbItem = new Element('li', {className: 'fp-object'});

                $breadCrumb.insert($breadCrumbItem);
            }

            if ($('floorplan-startup')) {
                $('floorplan-startup').remove();
            }

            // @see  FP-66 Display edit and save when NOT on "splash screen".
            if ($editButton) {
                $editButton.removeClassName('hide');
            }

            if ($saveButton) {
                $saveButton.removeClassName('hide');
            }

            $visualizationLeftFunctions.select('select,button').invoke('enable');
            $visualizationLeftLegend.select('input').invoke('enable');
            $changeBackgroundButton.enable();
            $setScaleButton.enable();

            // Initialize the image uploader.
            new qq.FileUploader({
                element: $('floorplan-visualization-background-upload-wrapper'),
                action: window.www_dir + 'floorplan/ajax/uploadBackgroundImage/' + objId,
                multiple: true,
                autoUpload: true,
                sizeLimit: 5242880, // About 5 MB.
                allowedExtensions: ['bmp', 'png', 'jpg', 'jpeg', 'gif'],
                onUpload:function(id){
                    $('floorplan-visualization-background-gallery').insert(
                        new Element('div', {'data-qq-id': id}).update(
                            new Element('img', {src:window.dir_images + 'ajax-loading.gif', style:'margin:67px;'})
                        )
                    );
                },
                onComplete: function (id, filename, response) {
                    var $img = $('floorplan-visualization-background-gallery').down('[data-qq-id="' + id + '"] img');

                    if (response.success && response.data) {
                        $img.on('load', function () {
                            $backgroundImageUploadOverlay.fire('update:galleryItems');
                        });

                        $img
                            .addClassName('background-image')
                            .writeAttribute('src', window.www_dir + 'floorplan/visualization/backgroundImage/' + response.data.data)
                            .writeAttribute('data-image-id', response.data.data);
                    }
                },
                onError: function (id, filename, response) {
                    // This does not get triggered reliably...
                    idoit.Notify.error(response.message || idoit.Translate.get('LC__MODULE__FLOORPLAN__VISUALIZATION__UPLOAD_IMAGE_ERROR') + ' "' + filename + '".', {sticky:true})
                },
                dragText: idoit.Translate.get('LC_FILEBROWSER__DROP_FILE'),
                uploadButtonText: '<img src="' + window.dir_images + 'axialis/basic/zoom.svg" alt="" /><span>' + idoit.Translate.get('LC__UNIVERSAL__FILE_ADD') + '</span>',
                cancelButtonText: '&nbsp;',
                failUploadText: idoit.Translate.get('LC__UNIVERSAL__ERROR')
            });
        } else {
            if ($('floorplan-startup')) {
                $('floorplan-startup').removeClassName('hide');
            }

            // @see  FP-66 Don't display edit and save on "splash screen".
            if ($editButton) {
                $editButton.addClassName('hide');
            }

            if ($saveButton) {
                $saveButton.addClassName('hide');
            }

            $visualizationLeftFunctions.select('select,button').invoke('disable');
            $visualizationLeftLegend.select('input').invoke('disable');
            $changeBackgroundButton.disable();
            $setScaleButton.disable();
        }

        window.pushState({}, document.title, window.www_dir + 'floorplan/visualization/' + objId);

        if (!objId) {
            $overlay.addClassName('hide');

            floorplanTree
                .setPositionedObjects([])
                .setOpenNodes([])
                .setMode($locationTreeOptions.down(':checked').getValue())
                .process();

            return;
        }

        new Ajax.Request(window.www_dir + 'floorplan/ajax/getFloorplanData', {
            parameters: {
                objectId: objId
            },
            onComplete: function (response) {
                var json = response.responseJSON, i;

                $refreshButton
                    .down('img')
                    .writeAttribute('src', window.dir_images + 'axialis/basic/symbol-update.svg');

                if (!json) {
                    idoit.Notify.error(response.responseText, {sticky: true});
                    return;
                }

                if (json && !json.success) {
                    idoit.Notify.error(json.message, {sticky: true});
                    return;
                }

                floorplanObjectImages = json.data.images;

                try {
                    $breadCrumbItem.update(json.data.rootData.objectTitle);

                    for (i in window.floorplanData.scaleFactors) {
                        if (! window.floorplanData.scaleFactors.hasOwnProperty(i)) {
                            continue;
                        }

                        if (window.floorplanData.scaleFactors[i] == json.data.floorplanOptions.grid) {
                            $gridSelect.setValue(i);
                        }
                    }

                    // @see FP-3 FP-45  Yes, this logic is reversed - because the "simulate click" part will toggle the state.
                    $gridDisplayToggle[json.data.floorplanOptions.gridDisplay ? 'removeClassName' : 'addClassName']('pressed').simulate('click');
                    $gridSnapToggle[json.data.floorplanOptions.snapToGrid ? 'removeClassName' : 'addClassName']('pressed').simulate('click');
                    $radiusDisplayToggle[json.data.floorplanOptions.radiusDisplay ? 'removeClassName' : 'addClassName']('pressed').simulate('click');
                    $scaleToggle[json.data.floorplanOptions.scaleShow ? 'removeClassName' : 'addClassName']('pressed').simulate('click');

                    var selectedProfile = null;

                    for (i in window.floorplanData.profiles) {
                        if (window.floorplanData.profiles.hasOwnProperty(i)) {
                            if (window.floorplanData.profiles[i].id == $profileSelect.getValue()) {
                                selectedProfile = window.floorplanData.profiles[i];
                            }
                        }
                    }

                    // Add a virtual "root" layer that lies beneath everything else.
                    json.data.layerData.push({ id: null, sort: 9999 });

                    // @see  FP-84  Cleaning up some (by now) unused options.
                    delete json.data.floorplanOptions.displayScale;
                    delete json.data.floorplanOptions.measurementValue;
                    delete json.data.floorplanOptions.measurementUnit;
                    delete json.data.floorplanOptions.scaleFactor;
                    delete json.data.floorplanOptions.scaleFactorMM;

                    // @see  FP-87  Set the initial floorplan to level 0.
                    floorplan
                        .preProcess()
                        .setData(json.data.objectData)
                        .setLayers(json.data.layerData)
                        .extendOptions(json.data.floorplanOptions)
                        .extendOptions({floorplanLevel: 0})
                        .setProfile(selectedProfile)
                        .resetBackgroundImage()
                        .setBackground(json.data.layout, json.data.forms)
                        .process();

                    floorplanLayer
                        .setLayers(json.data.layerData)
                        .setObjects(json.data.objectData)
                        .process();

                    if (window.floorplanData.initialHighlight) {
                        floorplan.selectObject({objId: window.floorplanData.initialHighlight}).center(window.floorplanData.initialHighlight);

                        window.floorplanData.initialHighlight = null;
                    }

                    // @see FP-38  Process the floorplan tree.
                    floorplanTree
                        .setActiveFloorplan(objId)
                        .setPositionedObjects(json.data.objectData.map(function(d) { return d.objId }))
                        .setOpenNodes(json.data.rootData.locationPath)
                        .setMode($locationTreeOptions.down(':checked').getValue())
                        .process();

                    $profileSelect.simulate('change');

                    if (window.floorplanData.authEdit && window.floorplanData.initialObjectAddition) {
                        enterEditMode();

                        floorplanTree.addPositionedObject(window.floorplanData.initialObjectAddition.objectId);

                        floorplan
                            .addObject(window.floorplanData.initialObjectAddition)
                            .selectObject({objId: window.floorplanData.initialObjectAddition.objectId})
                            .center(window.floorplanData.initialObjectAddition.objectId);

                        floorplanLayer.setObjects(floorplan.getAllData()).process();
                    } else {
                        // If we don't center a specific object, simply center the main canvas.
                        floorplan.center();
                    }

                    floorplan.responsive();

                    processFloorplanControls();
                } catch (e) {
                    idoit.Notify.error(e, {sticky:true});
                }
            }
        });
    };

    window.setObjectPolygon = function (polygon) {
        $selectedObjectPolygon.setValue(JSON.stringify(polygon));

        floorplan.getData(selectedObjectId).polygon = polygon;
        floorplan.updateObjectPositions(selectedObjectId, true);

        // @see  FP-43 Don't save automatically
    };

    window.getLayout = function () {
        return {
            layout: floorplan.getLayoutData(),
            forms: floorplan.getFormsData()
        }
    };

    window.setLayout = function (layout, forms) {
        // First we reset everything, so we don't have "update" issues.
        floorplan.setBackground(null, []);
        // Then set the actual background.
        floorplan.setBackground(layout, forms);
    };

    // @see  FP-45
    window.setObjectRadius = function (radius, unit, color, opacity) {
        var objectData = floorplan.getData(selectedObjectId);

        radius = parseInt(radius);
        opacity = parseInt(opacity);

        objectData.radius = {
            display: radius !== 0,
            radius: radius,
            unit: unit,
            color: color,
            opacity: opacity
        };

        floorplan.updateObjectPositions(selectedObjectId);
    };

    floorplan = new Floorplan($canvas, {
        objectTypeData: objectTypeData,
        onObjSelect: function (d, nestedObject) {
            // Set the "selected object".
            selectedObjectId = d.objId;
            selectedObjectIsFloorplan = d.visualizationType instanceof window.FloorplanVisualizationTypeFloorplan;
            selectedObjectIsNested = nestedObject;
            $selectedObjectId.setValue(selectedObjectId);
            $selectedObjectPolygon.setValue(JSON.stringify(d.polygon));

            // Set the "open object" link
            $visualizationLeftFunctions.down('a[data-action="object-link"]').writeAttribute('href', window.www_dir + '?objID=' + d.objId);

            // Disable the "inherit from formfactor" button. This will be enabled, if formfactor data is available.
            $functionInheritFormfactorData
                .writeAttribute('title', idoit.Translate.get('LC__MODULE__FLOORPLAN__VISUALIZATION__INHERIT_FORMFACTOR_DATA_TITLE'))
                .disable();

            // Also we want to display some additional information and options.
            $visualizationLeftContent.highlight({endcolor:'#ffffff', restorecolor:'#ffffff'});
            $visualizationLeftFunctions.highlight({endcolor:'#ffffff', restorecolor:'#ffffff'});

            if (d.hasOwnProperty('orientation') && d.orientation !== null) {
                $objectOrientationSelect.setValue(d.orientation);
            } else {
                $objectOrientationSelect.setValue('-1');
            }

            processFloorplanControls();

            $visualizationLeftContent.fire('load:objectInformation');
        },
        onObjUnselect: function (nestedObject) {
            // Set the "selected" object to 0.
            selectedObjectId = 0;
            $selectedObjectId.setValue(0);
            $selectedObjectPolygon.setValue('');

            processFloorplanControls();
        },
        onComplete: function (objectId) {
            // Once the rendering is complete we'd like to hide the "loading" overlay.
            $overlay.addClassName('hide');

            if (objectId !== undefined) {
                this.center(objectId);
            }
        },
        openObjectInformationTab: function () {
            $('floorplan-visualization-left-tabs').down('li a').simulate('click');
        }
    });

    floorplan.createOptionalDOM();

    floorplanTree = new FloorplanTree($locationTreeContent, {
        rootObjectId:        window.floorplanData.rootObjectId,
        nodeSorting: function(a, b) {
            return a.objectTitle.localeCompare(b.objectTitle);
        },
        onSelect:            function (objectId) {
            var selectedObjectData = floorplan.getData(objectId);

            // @see  FP-65 Only process existing data.
            if (selectedObjectData) {
                floorplan.selectObject(selectedObjectData).center(objectId);
            }
        },
        onOpenFloorplan:     function (objectId, inEditMode) {
            if (editMode && !confirm(idoit.Translate.get('LC__MODULE__FLOORPLAN__LEAVE_WITHOUT_SAVE_NOTIFICATION'))) {
                return;
            }

            initializeFloorplan(objectId);

            if (inEditMode) {
                enterEditMode();
            }
        },
        onAdd:               function (objectId, data) {
            floorplanTree.addPositionedObject(objectId);
            floorplan.addObject(data);

            floorplanLayer.setObjects(floorplan.getAllData()).process();
            // @see  FP-43 Don't save automatically
        },
        onAddAllChildren:    function (parentObjectId, data) {
            var i;

            for (i in data.children) {
                if (!data.children.hasOwnProperty(i)) {
                    continue;
                }

                floorplanTree.addPositionedObject(data.children[i].objectId);
                floorplan.addObject(data.children[i], i);
            }

            floorplanLayer.setObjects(floorplan.getAllData()).process();
            // @see  FP-43 Don't save automatically
        },
        onRemove:            function (objectId, data) {
            floorplanTree.removePositionedObject(objectId);
            floorplan.removeObject(data);

            // @see  FP-43 Don't save automatically
        },
        onRemoveAllChildren: function (parentObjectId, data) {
            var i;

            for (i in data.children) {
                if (!data.children.hasOwnProperty(i)) {
                    continue;
                }

                floorplanTree.removePositionedObject(data.children[i].objectId);
                floorplan.removeObject(data.children[i]);
            }

            // @see  FP-43 Don't save automatically
        }
    });

    floorplanLayer = new LayerEditor($layerBrowser, {
        onObjectUpdate: function(objectId, layerId, visibility, sort) {
            const object = floorplan.getData(objectId);

            object.layer = layerId > 0 ? layerId : null;
            object.visibility = visibility;
            object.sort = sort;

            floorplan.process();

            floorplanLayer.setObjects(floorplan.getAllData()).process();
        },
        onObjectOrderUpdate: function (objectData) {
            for (const i in objectData) {
                if (!objectData.hasOwnProperty(i)) {
                    continue;
                }

                const object = floorplan.getData(objectData[i].objectId);

                object.sort = objectData[i].sort;
                object.layer = objectData[i].layer;
            }

            floorplan.process();

            floorplanLayer.setObjects(floorplan.getAllData()).process();
        },
        onLayerOrderUpdate: function (layerData) {
            for (const i in layerData) {
                if (!layerData.hasOwnProperty(i)) {
                    continue;
                }

                floorplan.getLayer(layerData[i].id).sort = layerData[i].sort;
            }

            floorplan.process();

            floorplanLayer.setLayers(floorplan.getAllLayers()).process();
        },
        onLayerCreate: function(title) {
            new Ajax.Request(window.www_dir + 'floorplan/layer/save', {
                parameters: {
                    floorplanObjectId: objectId,
                    layerTitle: title,
                    layerColor: '#ffffff',
                    layerConfig: '{"condensed":false, "visibility":1}'
                },
                onComplete: processLayerResponse
            });
        },
        onLayerUpdate: function(id, title, visibility, condensed, color, sort) {
            new Ajax.Request(window.www_dir + 'floorplan/layer/save', {
                parameters: {
                    floorplanObjectId: objectId,
                    layerId: id,
                    layerTitle: title,
                    layerColor: color,
                    layerConfig: JSON.stringify({"condensed": condensed, "visibility": parseFloat(sort)}),
                    layerSort: parseInt(sort)
                },
                onComplete: processLayerResponse
            });
        },
        onLayerDelete: function(id) {
            new Ajax.Request(window.www_dir + 'floorplan/layer/delete', {
                parameters: {
                    layerId: id
                },
                onComplete: processLayerResponse
            });
        }
    });

    function processLayerResponse(response) {
        var json = response.responseJSON;

        if (json.success && json.data) {
            json.data;

            // Here we re-set the layer sorting to how it was before.
            for (const i in json.data) {
                if (!json.data.hasOwnProperty(i)) {
                    continue;
                }

                try {
                    json.data[i].sort = floorplan.getLayer(json.data[i].id).sort;
                } catch (e) {
                    json.data[i].sort = 0;
                };
            }

            // Add a virtual "root" layer that lies beneath everything else.
            json.data.push({ id: null, sort: 9999 });

            floorplan
                .setLayers(json.data)
                .process();

            floorplanLayer
                .setLayers(json.data)
                .process();
        } else {
            idoit.Notify.error(json.message, {sticky: true});
        }
    }

    // Initialize the floorplan
    initializeFloorplan(objectId);

    // Function for saving the floorplan.
    function saveFloorplan () {
        // @see FP-77 Make a "snapshot" of the Floorplan.
        const $floorplanCopy = window.FloorplanHelper.prepareFullSvg($canvas);

        var options = Object.clone(floorplan.getOptions());

        if (! authEdit) {
            // Do not save anything, if the user has no edit rights.
            return;
        }

        // At first we delete some data, we don't need in our floorplan configuration.
        delete options.objectTypeData;
        delete options.undefinedObjectType;
        delete options.authEdit;
        delete options.width;
        delete options.height;
        delete options.minZoomLevel;
        delete options.maxZoomLevel;

        // @see  FP-84  Delete some unused options.
        delete options.displayScale;
        delete options.measurementValue;
        delete options.measurementUnit;
        delete options.scaleFactor;
        delete options.scaleFactorMM;

        // @see  FP-87  Remove unnecessary information.
        delete options.floorplanLevel;

        if (options.hasOwnProperty('backgroundImage') && !options.backgroundImage) {
            delete options.backgroundImage;
        }

        new Ajax.Request(window.www_dir + 'floorplan/floorplan/save', {
            parameters: {
                floorplanData: JSON.stringify(options),
                floorplanObjectId: objectId,
                floorplanObjectData: JSON.stringify(floorplan.getAllData().map(function(d) {
                    const copy = Object.assign({}, d);
                    delete copy.visualizationType;
                    return copy;
                })),
                floorplanLayoutData: JSON.stringify(floorplan.getLayoutData()),
                floorplanFormsData: JSON.stringify(floorplan.getFormsData()),
                floorplanLayerData: JSON.stringify(floorplan.getAllLayers())
            },
            onComplete: function (response) {
                var json = response.responseJSON;

                if (json.success && json.data) {
                    // We extend the options, because some of them might have been processed by PHP.
                    floorplan.extendOptions(json.data);

                    idoit.Notify.success(idoit.Translate.get('LC__MODULE__FLOORPLAN__SAVED'));

                    // @see FP-55 After saving, we refresh the tree.
                    floorplanTree
                        .setMode($locationTreeOptions.down(':checked').getValue())
                        .process();

                    idoit.Require.require('svgToPng', function() {
                        svgAsPngUri($floorplanCopy, {}, function(uri) {
                            new Ajax.Request(window.www_dir + 'floorplan/floorplan/saveSnapshot', {
                                parameters: {
                                    floorplanObjectId: objectId,
                                    floorplanSnapshot: uri
                                },
                                onComplete: function (response) {
                                    var json = response.responseJSON;

                                    if (json.success) {
                                        idoit.Notify.success(idoit.Translate.get('LC__MODULE__FLOORPLAN__SNAPSHOT_SAVED'));
                                    } else {
                                        idoit.Notify.warning(idoit.Translate.get('LC__MODULE__FLOORPLAN__SNAPSHOT_COULD_NOT_BE_SAVED') + json.message, {sticky: true});
                                    }
                                }
                            });
                        });
                    });
                } else {
                    idoit.Notify.error(json.message, {sticky: true});
                }
            }
        });
    }

    // Function for deleting the floorplan.
    function deleteFloorplan () {
        if (!window.confirm(idoit.Translate.get('LC__MODULE__FLOORPLAN__POPUP__DELETE_CONFIRMATION'))) {
            return;
        }

        if (! authDelete) {
            // Do not save anything, if the user has no edit rights.
            return;
        }

        new Ajax.Request(window.www_dir + 'floorplan/floorplan/delete', {
            parameters: {
                floorplanObjectId: objectId,
            },
            onComplete: function (response) {
                var json = response.responseJSON;
                if (json.success) {
                    initializeFloorplan(objectId);
                    enterViewMode();
                    // After delete, we refresh the tree.
                    floorplanTree.process();
                    idoit.Notify.success(idoit.Translate.get('LC__MODULE__FLOORPLAN__DELETED'));
                } else {
                    idoit.Notify.error(json.message, {sticky: true});
                }
            }
        });
    }

    function processFloorplanControls() {
        var hideElements = [];
        var showElements = [];

        if (editMode) {
            showElements.push(
                // Floorplan controls
                $saveButton,
                $cancelButton,
                $deleteButton,
                $gridSelect,
                $gridDisplayToggle,
                $gridSnapToggle,
                $changeBackgroundButton,
                $changeLayoutButton,
                $setScaleButton,
            );
            hideElements.push(
                $editButton
            );

            if (selectedObjectId > 0) {
                showElements.push(
                    $functionToggleObjectType,
                    $visualizationLeftFunctions.down('div'),
                    $visualizationLeftFunctions.down('[data-action="object-link"]')
                );

                if (selectedObjectIsNested) {
                    hideElements.push(
                        $visualizationLeftContent.down('.formfactor-edit'),
                        $orientationContainer,
                        $objectOrientationSelect,
                        $polygonEditorPopupButton,
                        $functionCenterObject,
                        $functionRotateAntiClockwise,
                        $functionRotateClockwise,
                        $functionUnpositionObject,
                        $functionDisplayRadius,
                        $functionInheritFormfactorData
                    );
                } else {
                    showElements.push(
                        $visualizationLeftContent.down('.formfactor-edit'),
                        $orientationContainer,
                        $objectOrientationSelect,
                        $functionCenterObject,
                        $functionRotateAntiClockwise,
                        $functionRotateClockwise,
                        $functionUnpositionObject,
                        $functionDisplayRadius,
                        $functionInheritFormfactorData
                    );

                    if (selectedObjectIsFloorplan) {
                        hideElements.push(
                            $polygonEditorPopupButton
                        );
                    } else {
                        showElements.push(
                            $polygonEditorPopupButton
                        );
                    }
                }
            } else {
                hideElements.push(
                    $visualizationLeftContent.down('.formfactor-edit'),
                    $orientationContainer,
                    $objectOrientationSelect,
                    $polygonEditorPopupButton,
                    $functionCenterObject,
                    $functionRotateAntiClockwise,
                    $functionRotateClockwise,
                    $functionUnpositionObject,
                    $functionDisplayRadius,
                    $functionInheritFormfactorData,
                    $functionToggleObjectType,
                    $visualizationLeftFunctions.down('div'),
                    $visualizationLeftFunctions.down('[data-action="object-link"]')
                );
            }
        } else {
            showElements.push(
                $editButton
            );
            hideElements.push(
                // Floorplan controls
                $saveButton,
                $cancelButton,
                $deleteButton,
                $gridSelect,
                $gridDisplayToggle,
                $gridSnapToggle,
                $changeBackgroundButton,
                $changeLayoutButton,
                $setScaleButton,
                // Object specific functions
                $visualizationLeftContent.down('.formfactor-edit'),
                $orientationContainer,
                $polygonEditorPopupButton,
                $functionCenterObject,
                $functionRotateAntiClockwise,
                $functionRotateClockwise,
                $functionUnpositionObject,
                $functionDisplayRadius,
                $functionInheritFormfactorData
            );

            if (selectedObjectId > 0) {
                showElements.push(
                    $visualizationLeftContent,
                    $visualizationLeftFunctions.down('div'),
                    $functionToggleObjectType,
                    $visualizationLeftFunctions.down('[data-action="object-link"]')
                );
            } else {
                hideElements.push(
                    $visualizationLeftContent,
                    $visualizationLeftFunctions.down('div'),
                    $functionToggleObjectType,
                    $visualizationLeftFunctions.down('[data-action="object-link"]')
                );
            }
        }

        showElements
            .filter(function($el) { return Object.isElement($el); }).invoke('removeClassName', 'hide')
            .filter(function ($el) { return typeof $el.enable === 'function'; }).invoke('enable');

        hideElements
            .filter(function($el) { return Object.isElement($el); }).invoke('addClassName', 'hide')
            .filter(function ($el) { return typeof $el.disable === 'function'; }).invoke('disable');
    }

    $layerBrowserButton.on('click', function () {
        $layerBrowser.toggleClassName('hide');
    });

    // Notify the user about unsaved data.
    window.onbeforeunload = function () {
        if (editMode) {
            return 'Are you sure you want to leave?';
        }
    };
})();
