/**
 * i-doit layout editor javascript class.
 *
 * @author  Leonard Fischer <lfischer@i-doit.com>
 */
window.LayoutEditor = Class.create({
    svg: null,
    selectedForm: null,
    $selectedElement: null,
    $selectedPoint: null,
    options: {},
    
    initialize: function ($el, options) {
        var that = this;
        
        this.$element = $el;
        this.$selectedElement = null;
        this.selectedPointIndex = null;
        this.$selectedPoint = null;
        this.width = this.$element.getWidth();
        this.height = this.$element.getHeight();
        this.layout = {data:[], style:{}};
        this.forms = [];
        this.transform = {x: 0, y: 0, k: 1};
        
        this.options = {
            minPoints:             3,    // The number of points is not allowed to drop underneath this limit.
            maxPoints:             30,   // The number of points is not allowed to exceed above this limit.
            maxPointsNotification: '',   // This notification will be displayed, when the "maxPoints" have been reached.
            minZoom:               0.25, // Defines the minimum zoom extent.
            maxZoom:               10,   // Defines the maximum zoom extent.
            snapToGrid:            true, // Defines if dragged elements should snap to the grid.
            $sideBar:              null  // This sidebar instance will be used to communicate via events.
        };
        
        Object.extend(this.options, options || {});
        
        // Set height and viewBox to "maxZoom * 100" to prevent unfinished rendering of the patterns in full zoom.
        this.svg = d3.select(this.$element)
            .select('svg')
            .attr('width', this.width)
            .attr('height', Math.max(400, this.options.maxZoom * 100))
            .attr('viewBox', [0, 0, this.width, Math.max(400, this.options.maxZoom * 100)]);
    
        this.$canvas = this.svg.select('#canvas');
        this.$layout = this.svg.select('#layout');
        this.$canvasForms = this.svg.select('#forms');
        this.$patternGrid = this.svg.select('#grid');
        this.$patternInnerGrid = this.svg.select('#inner-grid');
        this.$controls = this.svg.select('.controls');
        this.$adder = this.svg.select('.adder');
        this.$extra = this.svg.select('.extra');
        
        // Prepare the zoom behaviour.
        this.zoom = d3.zoom()
            .scaleExtent([this.options.minZoom, this.options.maxZoom])
            .extent([[0, 0], [this.width, this.height]])
            .on('zoom', function () { that.zoomed.call(that, d3.event.transform); });
            
        // Apply the zoom behaviour, disable double click.
        this.svg
            .call(this.zoom)
            .on('dblclick.zoom', null);
    
        this.$layout.on('click', this.select.bind(this, this.$layout));
        
        this.setObserver();
        
        return this;
    },
    
    getLayoutData: function () {
        return this.layout;
    },
    
    getFormData: function () {
        let i;
        const data = [];
        
        for (i in this.forms) {
            if (!this.forms.hasOwnProperty(i)) {
                continue;
            }
    
            data.push({
                type: this.forms[i].type,
                options: this.forms[i].options,
                style: this.forms[i].style,
                transform: this.forms[i].transform
            })
        }
        
        return data;
    },
    
    getZoomTransform: function () {
        return d3.zoomTransform(this.svg.node());
    },
    
    setObserver: function () {
        const that = this;
        
        this.svg.select('.grid').on('click', function () {
            that.unselect();
        });
        
        this.svg.on('mousemove', function () {
            if (that.$selectedElement === null || that.selectedForm !== null) {
                that.$adder.classed('hide', true);
                
                return;
            }
            
            const currentTransformation = that.getZoomTransform();
            let mouse = d3.mouse(this);
    
            mouse[0] = (mouse[0] - currentTransformation.x) / currentTransformation.k;
            mouse[1] = (mouse[1] - currentTransformation.y) / currentTransformation.k;
    
            const point = FloorplanHelper.closestPointOnPath(that.$layout.node(), mouse);
            
            that.$adder
                .classed('hide', point.distance > 40)
                .attr('cx', point.x)
                .attr('cy', point.y);
        });
    
        this.$adder.on('dblclick', this.addDraggable.bind(this));
        
        this.$element.on('grid:toggle-snap-to-grid', function(ev) {
            that.options.snapToGrid = !!ev.memo.snapToGrid;
        });
        
        this.$element.on('point:position-updated', function(ev) {
            that.layout.data[ev.memo.pointIndex] = {
                x: ev.memo.x,
                y: ev.memo.y
            }
            
            that.process();
        });
    
        this.$element.on('layout:design-updated', function(ev) {
            that.layout.style = {
                fillOpacity: (ev.memo.fillOpacity / 100.0),
                fill: ev.memo.fill,
                strokeWidth: ev.memo.strokeWidth,
                stroke: ev.memo.stroke
            };
        
            that.process();
        });
        
        this.$element.on('form:design-updated', function(ev) {
            if (this.selectedForm === null) {
                return;
            }
            
            const style = {};
            const options = {};
            const transform = {};
    
            // Set the styles.
            that.selectedForm.setStyle({
                fillOpacity: (ev.memo.fillOpacity / 100.0),
                fill: ev.memo.fill,
                strokeWidth: ev.memo.strokeWidth,
                stroke: ev.memo.stroke
            });

            for (const i in ev.memo) {
                if (!ev.memo.hasOwnProperty(i)) {
                    continue;
                }
                
                if (['radius', 'width', 'height', 'size'].indexOf(i) !== -1) {
                    options[i] = FloorplanHelper.processNumber(ev.memo[i]);
                }
    
                if (['x', 'y', 'scale', 'r'].indexOf(i) !== -1) {
                    transform[i] = FloorplanHelper.processNumber(ev.memo[i]);
                }
            }
            
            // Set Options.
            if (Object.keys(options).length) {
                that.selectedForm.setOptions(options);
            }

            // Set transformation.
            if (Object.keys(transform).length) {
                Object.extend(that.selectedForm.transform, transform);
            }

            that.process();
        });
    
        this.$element.on('point:design-updated', function(ev) {
            that.layout.style = {
                fillOpacity: (ev.memo.fillOpacity / 100.0),
                fill: ev.memo.fill,
                strokeWidth: ev.memo.strokeWidth,
                stroke: ev.memo.stroke
            };
    
            that.process();
        });
        
        this.$element.on('form:add', function(ev) {
            that.addForm(ev.memo.type)
        })
    },
    
    setLayout: function(layout) {
        this.layout = layout;
        
        return this;
    },
    
    setForms: function(forms) {
        this.forms = [];
        
        // Here we need to transform the saved forms back to JS components.
        for (i in forms) {
            if (!forms.hasOwnProperty(i)) {
                continue;
            }
    
            this.forms.push(createForm(forms[i].type)
                .setOptions(forms[i].options || {})
                .setStyle(forms[i].style || {})
                .setTransform(forms[i].transform || {}));
        }
        
        return this;
    },
    
    select: function($element) {
        // We selected the same element, don't do anything.
        if (this.$selectedElement === $element) {
            return;
        }
        
        // Unselect everything, before selecting new elements.
        this.unselect();
        
        this.selectedForm = null;
        this.$selectedElement = $element;
    
        // Remove all 'selected' CSS classes, then select the current one.
        this.$canvas.selectAll('.selected').classed('selected', false);
        this.$selectedElement.classed('selected', true);
    
        this.$selectedElement.call(floorplanLayoutLayoutDragMove(this));
        
        // Remove all (possibly outdated) draggables.
        this.$controls.selectAll('circle').remove();

        this.options.$sideBar.fire('layout:selected', {form: {style: this.layout.style}});
        
        // Update the draggables.
        this.updateDraggables();
    },
    
    selectDraggable: function($point, data, index) {
        // We selected the same element, don't do anything.
        if (this.$selectedPoint === $point) {
            return;
        }
    
        this.selectedForm = null;
        this.selectedPointIndex = index;
        this.$selectedPoint = $point;
    
        // Remove all 'selected' CSS classes, then select the current one.
        this.$controls.selectAll('.selected').classed('selected', false);
        $point.classed('selected', true);
        
        this.options.$sideBar.fire('point:selected', {data: data, index: index});
    },
    
    selectForm: function(form) {
        // We selected the same element, don't do anything.
        if (this.$selectedElement === form.$form) {
            return;
        }
    
        // Unselect everything, before selecting new elements.
        this.unselect();
        
        this.selectedForm = form;
        this.$selectedElement = form.$form;
        
        // Remove all 'selected' CSS classes, then select the current one.
        this.$canvas.selectAll('.selected').classed('selected', false);
        this.$canvasForms.selectAll('.selected').classed('selected', false);
        this.$selectedElement.classed('selected', true);
    
        this.$selectedElement.call(floorplanLayoutFormDragMove(this));
        
        // Remove all (possibly outdated) draggables.
        this.$controls.selectAll('circle').remove();
    
        this.options.$sideBar.fire('form:selected', {form: this.selectedForm});
        
        // Remove the draggables.
        this.unselectDraggable();
    },
    
    updateDraggables: function () {
        var that = this;
        
        if (this.$selectedElement) {
            const data = this.$selectedElement.data();

            if (!Array.isArray(data) || !data[0] || !Array.isArray(data[0].data)) {
                return;
            }

            this.$controls
                .selectAll('circle')
                .data(data[0].data)
                .join('circle')
                .attr('r', 8 / this.transform.k)
                .attr('cx', function(d) { return d.x; })
                .attr('cy', function(d) { return d.y; })
                .style('stroke-width', Math.max(.5, 3 / this.transform.k))
                .on('click', function (d, i) { that.selectDraggable.call(that, d3.select(this), d, i); })
                .on('dblclick', that.removeDraggable.bind(this))
                .call(floorplanLayoutPointDragMove(this));
        }
    },
    
    addDraggable: function () {
        var newPoint = [this.$adder.attr('cx'), this.$adder.attr('cy')],
            smallestDistance = Infinity,
            after = 0,
            index, distance, i, indexPoint, nextPoint;
        
        if (this.layout.data.length >= this.options.maxPoints) {
            idoit.Notify.info(this.options.maxPointsNotification.replace('%s', '' + this.options.maxPoints), {life: 5});
            return;
        }
        
        for (i in this.layout.data) {
            if (!this.layout.data.hasOwnProperty(i)) {
                continue;
            }
            
            distance = FloorplanHelper.distanceBetweenPoints([this.layout.data[i].x, this.layout.data[i].y], newPoint);
            
            if (distance < smallestDistance){
                smallestDistance = distance;
                index = parseInt(i);
            }
        }
        
        // Get the next point to evaluate if we need to insert the new point before or after the found index.
        indexPoint = this.layout.data[index];
        nextPoint = this.layout.data[(index === this.layout.data.length - 1 ? 0 : index+1)];
        
        newPoint = {
            x: Math.round(newPoint[0] / 10) * 10,
            y: Math.round(newPoint[1] / 10) * 10
        };
        
        if (FloorplanHelper.checkPointInRectangle(nextPoint.x, nextPoint.y,
            indexPoint.x, nextPoint.y,
            indexPoint.x, indexPoint.y,
            nextPoint.x, indexPoint.y,
            newPoint.x, newPoint.y)) {
            after = 1;
        }
        
        this.layout.data.splice(index + after, 0, newPoint);
        
        this.process();
    },
    
    unselect: function($element) {
        this.$selectedElement = null;
        this.selectedForm = null;
        this.selectedPointIndex = null;
        
        // Remove all 'selected' CSS classes and remove the drag handlers.
        this.$canvas.selectAll('*').classed('selected', false).on('.drag', null);
        
        // Remove all (possibly outdated) draggables.
        this.$controls.selectAll('circle').remove();
        
        this.options.$sideBar.fire('form:unselected');
    },
    
    removeDraggable: function (d, i) {
        if (this.layout.data.length <= this.options.minPoints) {
            idoit.Notify.info(idoit.Translate.get('LC__MODULE__FLOORPLAN__POPUP_EDITOR__MIN_POINTS_SET').replace('%s', this.options.minPoints));
            
            return;
        }
    
        this.layout.data.splice(i, 1);
    
        this.process();
    },
    
    unselectDraggable: function () {
        this.$selectedPoint = null;
        this.selectedPointIndex = null;
    
        this.$controls.selectAll('.selected').classed('selected', false);
    },
    
    addForm: function(type) {
        this.forms.push(createForm(type)
            .setStyle({
                fill: '#ffffff',
                fillOpacity: 1,
                stroke: '#000000',
                strokeWidth: 1
            }));
        
        this.process();
    },
    
    removeSelection: function () {
        if (this.$selectedElement !== null && this.$selectedPoint !== null) {
            if (this.layout.data.length > this.options.minPoints) {
                this.layout.data.splice(this.selectedPointIndex, 1);
            } else {
                idoit.Notify.info(idoit.Translate.get('LC__MODULE__FLOORPLAN__POPUP_EDITOR__MIN_POINTS_SET').replace('%s', this.options.minPoints));
            }
        }
        
        if (this.selectedForm !== null) {
            const formIndex = this.forms.indexOf(this.selectedForm);
            
            if (formIndex >= 0) {
                this.selectedForm.$form.remove();
    
                this.forms.splice(formIndex, 1);
            }
        }
        
        this.process();
    },
    
    zoomed: function (transform) {
        this.transform = transform;

        const transform10 = transform.k * 10;
        const transform100 = transform10 * 10;
        
        // Draw the scale.
        const currentScale = this.getZoomTransform().k;
        const scaleWidth = 500;
        const linearScale = d3.scaleLinear()
            .range([0, scaleWidth])
            .domain([0, scaleWidth / (currentScale * 100)]);
        
        const axis = d3.axisBottom(linearScale)
            .ticks(5)
            .tickFormat(function (d) { return d + 'm'; });
        
        this.svg.select('.scale').call(axis);
        
        // Don't move the grid itself, simply change the pattern.
        this.$patternGrid
            .attr('x', transform.x % transform100)
            .attr('y', transform.y % transform100)
            .attr('width', transform100)
            .attr('height', transform100);
        this.$patternInnerGrid
            .attr('width', transform10)
            .attr('height', transform10);
        
        // Scale the control circles.
        this.$adder
            .attr('r', 5 / transform.k)
            .style('stroke-width', Math.max(.5, 2 / transform.k));
        
        this.$controls
            .selectAll('circle')
            .attr('r', 8 / transform.k)
            .style('stroke-width', Math.max(.5, 3 / transform.k));
        
        // Translate and scale the canvas.
        this.$canvas.attr('transform', transform);
    },
    
    process: function () {
        const that = this;
    
        // Draw the layout.
        this.$layout
            .datum(this.layout)
            .attr('points', function (d) {
                return d.data.map(function (d) { return d.x + ',' + d.y; }).join(' ');
            })
            .style('fill-opacity', function (d) {return d.style.fillOpacity;})
            .style('fill', function (d) {return d.style.fill;})
            .style('stroke-width', function (d) {return d.style.strokeWidth;})
            .style('stroke', function (d) {return d.style.stroke;});

        // Draw additional forms.
        this.$canvasForms
            .selectAll('.object')
            .data(this.forms, function (d) { return d.id; })
            .join(
                function ($enter) {
                    $enter
                        .append(function (d) { return d.render().node(); })
                        .on('click', function (d, i) { that.selectForm.call(that, d, i); });
                },
                function ($update) {
                    // Since we use a external class to render stuff, we need to do a little hack here.
                    $update
                        .attr('data-updated', function (d) {
                            // The DOM will get updated by this call.
                            d.update();
                            
                            return new Date().getTime();
                        });
                },
                function ($exit) {
                    $exit.remove();
                }
            );
    
        // Update draggables, according to possible position changes.
        this.updateDraggables();
        
        return this;
    },
    
    zoomTo: function(x, y, scale, animate) {
        const zoomIdentity = d3.zoomIdentity
            .translate(x, y)
            .scale(scale);
    
        if (animate) {
            this.svg
                .transition()
                .duration(500)
                .call(this.zoom.transform, zoomIdentity)
        } else {
            this.svg.call(this.zoom.transform, zoomIdentity);
        }
    }
});