<?php

namespace idoit\Module\Floorplan\Model;

use idoit\Model\Dao\Base;
use isys_application as Application;
use isys_helper_link as HelperLink;

/**
 * i-doit Floorplan Model
 *
 * @package     Modules
 * @subpackage  Floorplan
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class Tree extends Base
{
    /**
     * Retrieve information of a given object.
     *
     * @param  integer $objectId
     * @param  integer $levels
     *
     * @return array
     * @throws \isys_exception_dao
     * @throws \isys_exception_database
     */
    public function getLocationChildren(int $objectId, $levels = 0)
    {
        $format = new Format($this->m_db);
        $physicalChildrenResult = $this->getPhysicalChildren($objectId);
        $logicalChildrenResult = $this->getLogicalChildren($objectId);

        $result = $format->object($objectId);
        $result['hasChildren'] = (bool)($physicalChildrenResult->count() + $logicalChildrenResult->count());

        $nextLevel = $levels - 1;

        if ($levels > 0 && $physicalChildrenResult->count()) {
            while ($row = $physicalChildrenResult->get_row()) {
                $result['children'][$row['isys_obj__id']] = $this->getLocationChildren((int)$row['isys_obj__id'], $nextLevel);
            }
        }

        if ($levels > 0 && $logicalChildrenResult->count()) {
            while ($row = $logicalChildrenResult->get_row()) {
                if (isset($result['children'][$row['isys_obj__id']])) {
                    // Skip objects we already found.
                    continue;
                }

                $result['children'][$row['isys_obj__id']] = $this->getLocationChildren((int)$row['isys_obj__id'], $nextLevel);
            }
        }

        $result['children'] = array_values($result['children']);

        return $result;
    }

    /**
     * @param  integer $objectId
     *
     * @return \isys_component_dao_result
     * @throws \isys_exception_database
     */
    private function getPhysicalChildren($objectId)
    {
        $sql = 'SELECT isys_obj__id 
            FROM isys_catg_location_list
            INNER JOIN isys_obj ON isys_obj__id = isys_catg_location_list__isys_obj__id 
            WHERE isys_catg_location_list__parentid = ' . $this->convert_sql_id($objectId) . '
            AND isys_obj__status = ' . $this->convert_sql_int(C__RECORD_STATUS__NORMAL) . '
            ORDER BY isys_obj__title ASC;';

        return $this->retrieve($sql);
    }

    /**
     * @param  integer $objectId
     *
     * @return \isys_component_dao_result
     * @throws \isys_exception_database
     */
    private function getLogicalChildren($objectId)
    {
        $sql = 'SELECT isys_obj__id 
            FROM isys_catg_logical_unit_list
            INNER JOIN isys_obj ON isys_obj__id = isys_catg_logical_unit_list__isys_obj__id 
            WHERE isys_catg_logical_unit_list__isys_obj__id__parent = ' . $this->convert_sql_id($objectId) . '
            AND isys_obj__status = ' . $this->convert_sql_int(C__RECORD_STATUS__NORMAL) . '
            ORDER BY isys_obj__title ASC;';

        return $this->retrieve($sql);
    }

    /**
     * Retrieve all floorplans.
     *
     * @param  integer $limit
     *
     * @return array
     * @throws \isys_exception_database
     */
    public function getFloorplans($limit = null)
    {
        $format = new Format($this->m_db);
        $language = Application::instance()->container->get('language');

        // We create this "fake" root, as our floorplan container.
        $result = [
            'objectId'        => 1,
            'objectTitle'     => $language->get('LC__MODULE__FLOORPLAN__TAB__LOCATION__MODE_FLOORPLANS'),
            'objectTypeId'    => null,
            'objectTypeTitle' => null,
            'objectTypeColor' => '#ffffff',
            'objectTypeIcon'  => HelperLink::get_base() . 'images/axialis/construction/house-4.svg',
            'isContainer'     => true,
            'children'        => [],
            'hasChildren'     => false
        ];

        $sql = 'SELECT isys_obj__id
            FROM isys_floorplan 
            INNER JOIN isys_obj ON isys_obj__id = isys_floorplan__isys_obj__id
            INNER JOIN isys_obj_type ON isys_obj_type__id = isys_obj__isys_obj_type__id 
            WHERE isys_floorplan__status = ' . $this->convert_sql_int(C__RECORD_STATUS__NORMAL) . '
            ORDER BY isys_obj__title ASC;';

        $floorplanResult = $this->retrieve($sql);

        while ($row = $floorplanResult->get_row()) {
            // @see  FP-68  Simply skip the root location, if it has been defined.
            if ($row['isys_obj__id'] == C__OBJ__ROOT_LOCATION) {
                continue;
            }

            $result['children'][] = $format->object((int)$row['isys_obj__id']);
        }

        $result['hasChildren'] = (bool)count($result['children']);

        return $result;
    }

    /**
     * Retireve all positioned objects of the given floorplan.
     *
     * @param  int $floorplanObjectId
     *
     * @return array
     * @throws \isys_exception_database
     */
    public function getFloorplanObjects(int $floorplanObjectId)
    {
        $format = new Format($this->m_db);

        $result = $format->object($floorplanObjectId);

        $sql = 'SELECT isys_obj__id
            FROM isys_floorplan_objects 
            INNER JOIN isys_floorplan ON isys_floorplan__id = isys_floorplan_objects__isys_floorplan__id
            INNER JOIN isys_obj ON isys_obj__id = isys_floorplan_objects__isys_obj__id
            WHERE isys_floorplan__isys_obj__id = ' . $this->convert_sql_id($floorplanObjectId) . '
            AND isys_obj__status = ' . $this->convert_sql_int(C__RECORD_STATUS__NORMAL) . '
            ORDER BY isys_obj__title ASC;';

        $floorplanResult = $this->retrieve($sql);

        while ($row = $floorplanResult->get_row()) {
            $result['children'][] = $format->object((int)$row['isys_obj__id']);
        }

        $result['hasChildren'] = (bool)count($result['children']);

        return $result;
    }
}
