<?php

use idoit\Module\Floorplan\Model\LayoutDao;

/**
 * i-doit
 *
 * Floorplan layout editor popup.
 *
 * @package     Modules
 * @subpackage  Floorplan
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @version     1.0
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class isys_popup_floorplan_layout_editor extends isys_component_popup
{
    private const DEFAULT_LAYOUT = [
        'data'  => [['x' => -200, 'y' => -200], ['x' => 200, 'y' => -200], ['x' => 200, 'y' => 200], ['x' => -200, 'y' => 200]],
        'style' => ['fill' => '#ffffff', 'fillOpacity' => 0.5, 'stroke' => '#000000', 'strokeWidth' => 1]
    ];

    /**
     * @var isys_component_template
     */
    protected $template;

    /**
     * @var isys_component_database
     */
    protected $database;

    /**
     * @var isys_component_template_language_manager
     */
    protected $language;

    /**
     * Constructor method.
     *
     * @throws Exception
     */
    public function __construct()
    {
        parent::__construct();

        $this->database = isys_application::instance()->container->get('database');
        $this->template = isys_application::instance()->container->get('template');
        $this->language = isys_application::instance()->container->get('language');
    }

    /**
     * Handles SMARTY request for dialog plus lists and builds the list base on the specified table.
     *
     * @param isys_component_template &$template
     * @param array                   $parameters
     *
     * @return string
     */
    public function handle_smarty_include(isys_component_template &$template, $parameters)
    {
        $onclick = $this->process_overlay('', '75%', '75%', $parameters, null, 700, 400);

        return '<button id="' . $parameters['name'] . '" type="button" onclick="' . $onclick . '" class="btn fr mr5 hide">' .
            '<img src="' . isys_application::instance()->www_path . 'images/axialis/cad/floor-plan.svg" />' .
            '<span>' . $this->language->get('LC__MODULE__FLOORPLAN__POPUP_LAYOUT__BUTTON') . '</span>' .
            '</button>';
    }

    /**
     * Handle specific ajax requests.
     *
     * @param isys_module_request $request
     *
     * @return  mixed
     * @author  Leonard Fischer <lfischer@i-doit.org>
     */
    public function &handle_ajax_request(isys_module_request $request)
    {
        header('Content-Type: application/json');

        $response = [
            'success' => true,
            'data'    => null,
            'message' => null
        ];

        try {
            switch ($_POST['action']) {
                case 'saveLayout':
                    $response['data'] = $this->saveLayout((int)$_POST['layoutId'], $_POST['name'], $_POST['layout'], $_POST['forms']);
                    break;

                case 'deleteLayout':
                    $response['data'] = $this->removeLayout((int)$_POST['layoutId']);
                    break;
            }
        } catch (Exception $e) {
            $response['success'] = false;
            $response['message'] = $e->getMessage();
        }

        echo \isys_format_json::encode($response);
        die;
    }

    /**
     * Method for handling the module request.
     *
     * @param isys_module_request $request
     *
     * @throws isys_exception_database
     */
    public function &handle_module_request(isys_module_request $request)
    {
        $rules = [
            'floorplan-layout-selection'               => [
                'p_arData'          => [],
                'p_bSort'           => false,
                'p_bDbFieldNN'      => true,
                'p_strSelectedID'   => 'rectangle',
                'p_bInfoIconSpacer' => 0,
                'disableInputGroup' => true
            ],
            'floorplan-layout-popup-selfdefined-title' => [
                'p_strPlaceholder' => 'LC__MODULE__FLOORPLAN__POPUP_LAYOUT__FORM_NAME',
                'p_strClass'       => 'input-small',
            ],
            'design-selection-x'                       => [
                'placeholder'       => '0.0',
                'p_bDisabled'       => true,
                'p_bInfoIconSpacer' => 0,
                'disableInputGroup' => true,
                'p_strClass'        => 'input-size-block',
                'p_dataIdentifier'  => 'x'
            ],
            'design-selection-y'                       => [
                'placeholder'       => '0.0',
                'p_bDisabled'       => true,
                'p_bInfoIconSpacer' => 0,
                'disableInputGroup' => true,
                'p_strClass'        => 'input-size-block',
                'p_dataIdentifier'  => 'y'
            ],
            'design-selection-rotation'                => [
                'placeholder'       => '0.0',
                'p_bDisabled'       => true,
                'p_bInfoIconSpacer' => 0,
                'disableInputGroup' => true,
                'p_strClass'        => 'input-size-block',
                'p_dataIdentifier'  => 'rotation'
            ],
            'design-background-transparency'           => [
                'p_arData'          => [
                    0   => '0%',
                    25  => '25%',
                    50  => '50%',
                    75  => '75%',
                    100 => '100%'
                ],
                'p_bSort'           => false,
                'p_bDbFieldNN'      => true,
                'p_strSelectedID'   => 50,
                'p_bDisabled'       => true,
                'p_bInfoIconSpacer' => 0,
                'disableInputGroup' => true,
                'p_strClass'        => 'input-size-block',
                'p_dataIdentifier'  => 'fillOpacity'
            ],
            'design-background-color'                  => [
                'p_strValue'     => '#ffffff',
                'containerClass' => '',
                'disabled'       => true,
                'size'           => 'block',
                'data'           => ['identifier' => 'fill'],
                'parent'         => '#floorplan-layout-popup .popup-content'
            ],
            'design-border-size'                       => [
                'p_arData'          => [
                    // 0 => ' - ',
                    1 => 'LC__MODULE__FLOORPLAN__POPUP_LAYOUT__TAB_DESIGN__SMALL',
                    3 => 'LC__MODULE__FLOORPLAN__POPUP_LAYOUT__TAB_DESIGN__MEDIUM',
                    5 => 'LC__MODULE__FLOORPLAN__POPUP_LAYOUT__TAB_DESIGN__LARGE'
                ],
                'p_bSort'           => false,
                'p_bDbFieldNN'      => true,
                'p_strSelectedID'   => 'm',
                'p_bDisabled'       => true,
                'p_bInfoIconSpacer' => 0,
                'disableInputGroup' => true,
                'p_strClass'        => 'input-size-block',
                'p_dataIdentifier'  => 'strokeWidth'
            ],
            'design-border-color'                      => [
                'p_strValue'     => '#000000',
                'containerClass' => '',
                'disabled'       => true,
                'size'           => 'block',
                'data'           => ['identifier' => 'stroke'],
                'parent'         => '#floorplan-layout-popup .popup-content'
            ]
        ];

        $ajaxUrl = isys_helper_link::create_url([
            C__CMDB__GET__POPUP  => 'floorplan_layout_editor',
            C__GET__AJAX_REQUEST => 'handle_ajax_request',
        ], true);

        $this->template->activate_editmode()
            ->assign('ajax_url', $ajaxUrl)
            ->assign('layouts', $this->getLayouts())
            ->assign('preselectedLayout', self::DEFAULT_LAYOUT)
            ->smarty_tom_add_rules('tom.popup.floorplan', $rules)
            ->display(isys_module_floorplan::getPath() . 'templates/popup/layout-editor.tpl');
        die;
    }

    /**
     * @return array
     * @throws isys_exception_database
     */
    private function getLayouts(): array
    {
        $layouts = [];
        $layoutDao = LayoutDao::instance($this->database);
        $polygonResult = $layoutDao->getData();

        while ($layoutRow = $polygonResult->get_row()) {
            try {
                $layouts[$layoutRow['id']] = [
                    'title'     => $this->language->get($layoutRow['title']),
                    'layout'    => \isys_format_json::decode($layoutRow['layout']),
                    'forms'     => \isys_format_json::decode($layoutRow['forms']),
                    'deletable' => !!$layoutRow['deletable']
                ];
            } catch (Exception $e) {
                isys_notify::error('Your polygon "' . $layoutRow['title'] . '" is defect, possibly the point-data is corrupted.');
            }
        }

        return $layouts;
    }

    /**
     * Method for saving a given layout + forms to the database.
     *
     * @param int    $id
     * @param string $name
     * @param string $layout
     * @param string $forms
     *
     * @return array
     * @throws isys_exception_dao
     * @throws isys_exception_database
     */
    private function saveLayout(int $id, string $name, string $layout, string $forms): array
    {
        $layoutDao = LayoutDao::instance($this->database);

        if ($id > 0 && !$layoutDao->isDeletable($id)) {
            throw new Exception($this->language->get('LC__MODULE__FLOORPLAN__POPUP_LAYOUT__OVERWRITE_FAILURE'));
        }

        $data = [
            'isys_floorplan_layouts__title'     => $name,
            'isys_floorplan_layouts__layout'    => $layout,
            'isys_floorplan_layouts__forms'     => $forms,
            'isys_floorplan_layouts__deletable' => 1
        ];

        if ($id === 0 || !empty($name)) {
            $id = null;
        } else {
            unset($data['isys_floorplan_layouts__title']);
        }

        try {
            $id = $layoutDao->saveLayout($id, $data);
        } catch (Exception $e) {
            // Code for "duplicate entry".
            if ($e->getCode() === 1062) {
                throw new Exception($this->language->get('LC__MODULE__FLOORPLAN__POPUP_LAYOUT__DUPLICATE_NAME', [$name]));
            } else {
                throw $e;
            }
        }

        return [
            'newId'   => $id,
            'layouts' => $this->getLayouts()
        ];
    }

    /**
     * Method for deleting a previously saved polygon. Only user-defined polygons can be removed!
     *
     * @param int $polygonId
     *
     * @return array
     * @throws isys_exception_dao
     * @throws isys_exception_database
     */
    private function removeLayout(int $polygonId): array
    {
        $layoutDao = new LayoutDao($this->database);

        if ($polygonId === 0 || !$layoutDao->isDeletable($polygonId)) {
            throw new Exception($this->language->get('LC__MODULE__FLOORPLAN__POPUP_LAYOUT__REMOVE_FAILURE'));
        }

        return [
            'deleted' => $layoutDao->deleteLayout($polygonId),
            'layouts' => $this->getLayouts()
        ];
    }
}
