/**
 * i-doit polygon editor javascript base class.
 *
 * @author  Leonard Fischer <lfischer@i-doit.com>
 */
window.LayerEditor = Class.create({
    options: {},
    
    initialize: function ($el, options) {
        const self = this;
        
        this.$element = $el;
        this.$layerContainer = this.$element.down('.layer-container');
        this.$newLayerButton = this.$element.down('[data-action="new-layer"]');
        this.$rootLayer = new Element('div', {className: 'root-layer layer', 'data-layer-id': 0});
        
        this.visibilityCache = {};
        this.layers = [];
        this.objects = [];
        this.options = {};
        this.editMode = false;
        
        Object.extend(this.options, options || {
            onObjectUpdate: function(objectId, layerId, visibility, sort) {},
            onObjectOrderUpdate: function(data) {},
            onLayerOrderUpdate: function(data) {},
            onLayerCreate: function(title) {},
            onLayerUpdate: function(id, title, visibility, condensed, color, sort) {},
            onLayerDelete: function(id) {}
        });
    
        this.$newLayerButton.on('click', function () {
            self.options.onLayerCreate('Layer #' + self.layers.length);
        });
    
        this.$element.on('click', 'img.toggle', function (ev) {
            const layerId = ev.findElement('.layer').readAttribute('data-layer-id');
            const layerData = self.layers.detect(function(d) { return d.id == layerId });
            
            self.options.onLayerUpdate(layerData.id, layerData.title, layerData.configuration.visibility, !layerData.configuration.condensed, layerData.color, layerData.sort);
        })
    
        this.$element.on('click', '[data-action="toggle-object-visibility"]', function (ev) {
            self.processObjectVisibility(ev.findElement('.object').readAttribute('data-object-id'));
        });
        
        this.$element.on('click', '[data-action="toggle-layer-visibility"]', function (ev) {
            const $layer = ev.findElement('.layer');
            const layerId = $layer.readAttribute('data-layer-id');
            const $objects = $layer.select('.object');
            let targetVisibility = self.visibilityCache['layer-' + layerId];
            
            switch (targetVisibility) {
                default:
                case 1:
                    targetVisibility = 0.5;
                    break;
                case 0.5:
                    targetVisibility = 0;
                    break;
                case 0:
                    targetVisibility = 1;
            }
    
            self.visibilityCache['layer-' + layerId] = targetVisibility;
            
            for (const i in $objects) {
                if (!$objects.hasOwnProperty(i)) {
                    continue;
                }
                
                self.processObjectVisibility($objects[i].readAttribute('data-object-id'), targetVisibility);
            }
        });
        
        this.$element.on('click', '[data-action="edit-layer"]', function (ev) {
            const $layer = ev.findElement('.layer');
            const $layerConfig = $layer.down('.config');
            
            // Select the value.
            $layerConfig.removeClassName('hide').down('input').activate();
        });
        
        this.$element.on('keydown', 'input', function (ev) {
            // Prevent the form from being submitted.
            if (ev.key === 'Enter') {
                ev.preventDefault();
                
                // Instead: save the layer title.
                ev.findElement('.layer').down('[data-action="update-layer"]').simulate('click');
            }
            
            // "Exit" the edit mode for this layer.
            if (ev.key === 'Escape') {
                ev.findElement('input').blur();
                ev.findElement('.config').addClassName('hide');
            }
        });
        
        this.$element.on('click', '[data-action="update-layer"]', function (ev) {
            const $layer = ev.findElement('.layer');
            const layerId = $layer.readAttribute('data-layer-id');
            const layerTitle = $layer.down('input').getValue();
            const layer = Object.extend({
                color:      '#ffffff',
                condensed:  false,
                sort:       0,
                visibility: 1
            }, self.layers.detect(function (d) { return d.id == layerId }));
            
            self.options.onLayerUpdate(parseInt(layerId), layerTitle, layer.visibility, layer.condensed, layer.color, layer.sort);
        });
        
        this.$element.on('click', '[data-action="delete-layer"]', function (ev) {
            const layerId = ev.findElement('.layer').readAttribute('data-layer-id');
            
            if (! confirm(idoit.Translate.get('LC__MODULE__FLOORPLAN__VISUALIZATION__LAYER_BROWSER__DELETE_CONFIRMATION'))) {
                return;
            }
            
            self.options.onLayerDelete(layerId);
        });
        
        return this;
    },
    
    setEditMode: function (editMode) {
        this.editMode = !!editMode;
        
        if (this.editMode) {
            this.$element.down('button').removeClassName('hide');
            this.$element.select('.handle').invoke('removeClassName', 'hide');
            this.$element.select('img.edit').invoke('removeClassName', 'hide');
        } else {
            this.$element.down('button').addClassName('hide');
            this.$element.select('.handle').invoke('addClassName', 'hide');
            this.$element.select('img.edit').invoke('addClassName', 'hide');
        }
    },
    
    setLayers: function (layers) {
        this.layers = layers
            .sort(function(a, b) {
                return a.sort - b.sort;
            })
            .map(function(d, i) {
                d.sort = i+1;
                return d;
            });
        
        return this;
    },
    
    setObjects: function (objects) {
        this.objects = objects
            .sort(function(a, b) {
                return a.sort - b.sort;
            })
            .map(function(d, i) {
                d.sort = i+1;
                return d;
            });
        
        return this;
    },
    
    getLayers: function () {
        return this.layers;
    },
    
    process: function () {
        let i, $layer;
        
        this.$layerContainer.update();
        this.$rootLayer.update();
        
        for (i in this.layers) {
            if (!this.layers.hasOwnProperty(i)) {
                continue;
            }
            
            if (this.layers[i].id === null) {
                // Skip the "virtual" root layer.
                continue;
            }
            
    
            this.$layerContainer.insert(this.renderLayer(this.layers[i]));
        }
        
        // Create the "root" layer.
        this.$layerContainer.insert(this.$rootLayer);
        
        for (i in this.objects) {
            if (!this.objects.hasOwnProperty(i)) {
                continue;
            }
    
            $layer = this.$layerContainer.down('[data-layer-id="' + this.objects[i].layer + '"] .container');
            
            if (!$layer) {
                // If the layer does not exist, throw the object in the "root".
                this.$rootLayer.insert(this.renderObject(this.objects[i]));
                continue;
            }
            
            $layer.insert(this.renderObject(this.objects[i]));
        }
        
        this.resetObservers();
        
        return this;
    },
    
    processObjectVisibility: function (objectId, forceVisibility) {
        const objectData = this.objects.detect(function(d) { return d.objId == objectId; }) || {sort: 0, visibility: 1};
        let visibility;
    
        switch (objectData.visibility) {
            case 1:
                visibility = 0.5;
                break;
            case 0.5:
                visibility = 0;
                break;
            case 0:
                visibility = 1;
        }
        
        if (forceVisibility !== undefined) {
            visibility = forceVisibility;
        }
    
        this.options.onObjectUpdate(objectId, objectData.layer, visibility, objectData.sort);
    },
    
    prepareObjectDroppables: function (drag) {
        const $objects = this.$layerContainer.select('.object');
        const $layers = this.$layerContainer.select('.layer .container');
        let $prev;
        let $next;
    
        for (const i in $objects) {
            if (!$objects.hasOwnProperty(i)) {
                continue;
            }
    
            $objects[i].insert({before: new Element('div', {className:'drop'})});
        }
        
        for (const i in $layers) {
            if (!$layers.hasOwnProperty(i)) {
                continue;
            }
            
            $layers[i].insert(new Element('div', {className:'drop'}));
        }
        
        // Remove the drop zones, directly positioned near the dragged object itself.
        if ($prev = drag.element.previous('.drop')) {
            $prev.remove();
        }
        
        if ($next = drag.element.next('.drop')) {
            $next.remove();
        }
        
        // After we created all the drop-areas, we assign the observer.
        const $droppables = this.$layerContainer.select('.drop');
    
        for (const i in $droppables)
        {
            if (!$droppables.hasOwnProperty(i))
            {
                continue;
            }
    
            Droppables.add($droppables[i], {
                hoverclass: 'hover',
                onDrop:     this.processObjectDrop.bind(this)
            });
        }
    },
    
    processObjectDrop: function ($object, $drop) {
        // We simply add the object to the "drop" because we simply need the position.
        $drop.insert($object);
    
        const $layer = $drop.up('.layer');
        const layerId = parseInt($layer.readAttribute('data-layer-id'));
        const $objects = $layer.select('.object');
        let objectSortData = [];
    
        for (const i in $objects) {
            if (!$objects.hasOwnProperty(i)) {
                continue;
            }
        
            objectSortData.push({
                objectId: $objects[i].readAttribute('data-object-id'),
                layer: layerId !== 0 ? layerId : null,
                sort: parseInt(i)
            });
        }
        
        this.options.onObjectOrderUpdate(objectSortData);
    },
    
    unprepareObjectDroppables: function () {
        this.$layerContainer.select('.drop').invoke('remove');
    },
    
    prepareLayerDroppables: function (drag) {
        const $layers = this.$layerContainer.select('.layer');
        let $prev;
        let $next;
    
        for (const i in $layers) {
            if (!$layers.hasOwnProperty(i)) {
                continue;
            }
        
            $layers[i].insert({before: new Element('div', {className:'drop'})});
        }
    
        // Remove the drop zones, directly positioned near the dragged object itself.
        if ($prev = drag.element.previous('.drop')) {
            $prev.remove();
        }
    
        if ($next = drag.element.next('.drop')) {
            $next.remove();
        }
    
        // After we created all the drop-areas, we assign the observer.
        const $droppables = this.$layerContainer.select('.drop');
    
        for (const i in $droppables)
        {
            if (!$droppables.hasOwnProperty(i))
            {
                continue;
            }
        
            Droppables.add($droppables[i], {
                hoverclass: 'hover',
                onDrop:     this.processLayerDrop.bind(this)
            });
        }
    },
    
    processLayerDrop: function ($object, $drop) {
        // We simply add the object to the "drop" because we simply need the position.
        $drop.insert($object);
    
        const $layer = this.$layerContainer.select('.layer');
        let layerSortData = [];
    
        for (const i in $layer) {
            if (!$layer.hasOwnProperty(i) || $layer[i].readAttribute('data-layer-id') == 0) {
                continue;
            }
        
            layerSortData.push({
                id: $layer[i].readAttribute('data-layer-id'),
                sort: parseInt(i)
            });
        }
    
        this.options.onLayerOrderUpdate(layerSortData);
    },
    
    unprepareLayerDroppables: function () {
        this.$layerContainer.select('.drop').invoke('remove');
    },
    
    resetObservers: function () {
        const self = this;
        const $objects = this.$layerContainer.select('.object');
        const $layers = this.$layerContainer.select('.layer');
        const $scroller = $('floorplan-visualization-layer-browser').down('.popup-content');
    
        Position.includeScrollOffsets = true;
        
        for (const i in $objects) {
            if (!$objects.hasOwnProperty(i)) {
                continue;
            }
            
            new Draggable($objects[i], {
                revert:  true,
                handle:  'handle',
                onStart: this.prepareObjectDroppables.bind(this),
                onEnd:   this.unprepareObjectDroppables.bind(this),
                snap:    function (x, y, drag) {
                    return [
                        (x + drag.offset[0]) + $scroller.scrollLeft - 13,
                        (y + drag.offset[1]) + $scroller.scrollTop - 13
                    ];
                }
            });
        }
        
        for (const i in $layers) {
            if (!$layers.hasOwnProperty(i) || $layers[i].readAttribute('data-layer-id') == 0) {
                continue;
            }
            
            new Draggable($layers[i], {
                revert:  true,
                handle:  'handle',
                onStart: this.prepareLayerDroppables.bind(this),
                onEnd:   this.unprepareLayerDroppables.bind(this),
                snap:    function (x, y, drag) {
                    return [
                        (x + drag.offset[0]) + $scroller.scrollLeft - 13,
                        (y + drag.offset[1]) + $scroller.scrollTop - 13
                    ];
                }
            });
        }
    },
    
    renderLayer: function (layerData) {
        return new Element('div', {className: 'layer', 'data-layer-id': layerData.id})
            .update(new Element('div', {className: 'title'})
                .update(new Element('div', {className: 'handle ' + (this.editMode ? '' : 'hide')}))
                .insert(new Element('img', {src: window.dir_images + 'icons/silk/bullet_arrow_' + (layerData.configuration.condensed ? 'right' : 'down') + '.png', className: 'toggle mouse-pointer mr5'}))
                .insert(new Element('span')
                    .update(layerData.title))
                .insert(new Element('img', {src: window.dir_images + 'axialis/documents-folders/document-edit.svg', 'data-action': 'edit-layer', className: 'edit mouse-pointer ' + (this.editMode ? '' : 'hide')}))
                .insert(new Element('img', {src: window.dir_images + 'axialis/basic/eye.svg', 'data-action': 'toggle-layer-visibility', className: 'visibility mouse-pointer', title: idoit.Translate.get('LC__MODULE__FLOORPLAN__VISUALIZATION__LAYER_BROWSER__TOGGLE_VISIBILITY')}))
                .insert(new Element('div', {className: 'config hide'})
                    .update(new Element('div', {className: 'input-group input-size-block'})
                        .update(new Element('input', {className: 'input', type: 'text', value: layerData.title}))
                        .insert(new Element('button', {className: 'btn ml5', type: 'button', 'data-action': 'update-layer', title: idoit.Translate.get('LC_UNIVERSAL__ACCEPT'), 'data-tooltip': 1})
                            .update(new Element('img', {src:window.dir_images + 'axialis/basic/symbol-ok.svg'})))
                        .insert(new Element('button', {className: 'btn ml5', type: 'button', 'data-action': 'delete-layer', title: idoit.Translate.get('LC_UNIVERSAL__DELETE'), 'data-tooltip': 1})
                            .update(new Element('img', {src:window.dir_images + 'axialis/basic/button-remove.svg'}))))))
            .insert(new Element('div', {className: 'container ' + (layerData.configuration.condensed ? 'condensed' : '')}));
    },
    
    renderObject: function (objectData) {
        const objectType = FloorplanHelper.getObjectType(objectData.objType);
        const visibilityIcon = objectData.visibility === 0 ? 'eye-no' : 'eye';
        const visibility = objectData.visibility === 0.5
            ? 'opacity: 0.5'
            : objectData.visibility === 0
                ? 'opacity: 1'
                : '';
        
        return new Element('div', {className: 'object', 'data-object-id': objectData.objId})
            .update(new Element('div', {className: 'handle ' + (this.editMode ? '' : 'hide')}))
            .insert(new Element('img', {src: objectType.icon, title: objectType.title}))
            .insert(new Element('span')
                .update(objectData.objTitle))
            .insert(new Element('img', {
                'data-action': 'toggle-object-visibility',
                className: 'visibility mouse-pointer',
                src: window.dir_images + 'axialis/basic/' + visibilityIcon + '.svg',
                style: visibility,
                title: idoit.Translate.get('LC__MODULE__FLOORPLAN__VISUALIZATION__LAYER_BROWSER__TOGGLE_VISIBILITY')
            }));
            
    }
});
