/**
 * i-doit floorplan visualization javascript base class.
 *
 * @see  FP-84  Small recap on how the "scale" works. The scale contains a float number (default 0.01) which determines the "mm to pixel" ratio.
 * This means one millimeter equals to 0.1 pixel and 1 meter equals to 100 pixels, by this we can calculate all sorts of measurements.
 *
 * In i-doit we always use the smallest unit of something - in case of object dimensions (formfactor) that's millimeter.
 * So if we want to calculate the dimension of an object in the floorplan, the calculation works like this:
 *
 * formfactorWidth * floorplanScale = width in pixel
 *
 * For the grid it works similar: The grid has its own unit (for example meter or foot) - this unit is represented in millimeter, so we can easily calculate :)
 *
 * gridUnit * floorplanScale = width/height of the grid.
 *
 * By solving the ticket FP-84 a floorplan ALWAYS has a scale (default 0.01, so that one meter equals 100 pixel),
 * the user can alter this by using the "Set scale" function - here the floorplanScale will be set to whatever the user defines.
 *
 * @author  Leonard Fischer <lfischer@i-doit.com>
 */
window.floorPlanDefaults = {
    polygon: [
        {x: -100, y: -100},
        {x: 100, y: -100},
        {x: 100, y: 100},
        {x: -100, y: 100}
    ]
};

window.Floorplan = Class.create({
    svg:            null,
    vis:            null,
    zoom:           null,
    editMode:       false,
    options:        {},
    profile:        {},
    defaultPolygon: window.floorPlanDefaults.polygon,

    getContainerDimensions: function () {
        const measurements = this.$element.node().getBoundingClientRect();

        return {
            width:  measurements.width || 64,
            height: (measurements.height || 64) + 16
        }
    },

    initialize: function ($el, options) {
        const that = this;

        this.$element = d3.select($el);
        this.uuid = fpUuid();

        const dimension = this.getContainerDimensions();

        this.layers = [];
        this.data = [];
        this.options = {
            minZoomLevel:             0.1,                     // Defines the minimal zoom level.
            maxZoomLevel:             8,                       // Defines the maximal zoom level.
            objectTypeData:           {},                      // JSON with all necessary object type data (name, color, ...).
            objectTypeFilter:         [],                      // Array pf object types, that shall not be displayed.
            onComplete:               Prototype.emptyFunction, // "Complete" event for when the rendering has finished.
            onObjSelect:              Prototype.emptyFunction, // Click event for a object node.
            onObjUnselect:            Prototype.emptyFunction, // Event for when a object gets unselected.
            openObjectInformationTab: Prototype.emptyFunction, // Forces the GUI to open the object information tab.
            width:                    dimension.width,         // This can be used to change the viewpoints width. The SVG element will always remain 100%x100%.
            height:                   dimension.height,        // This can be used to change the viewpoints height. The SVG element will always remain 100%x100%.
            undefinedObjectType:      {},                      // Temp object for caching "undefined object type" messages.
            includeHandlers:          true                     // Option to en- or disable handlers for interactivity (JS events).
        };

        this.resetOptions();

        this.line = d3.line()
            .x((d) => d[0])
            .y((d) => d[1])
            .curve(d3.curveLinear);

        Object.extend(this.options, options || {});

        this.zoom = Prototype.emptyFunction;

        if (this.options.includeHandlers) {
            // Prepare the different drag handlers.
            this.dragMove = floorplanDragMove(this);
            this.dragScale = floorplanDragScale(this);
            this.dragRotate = floorplanDragRotate(this);

            this.zoom = d3.zoom()
                .scaleExtent([
                    this.options.minZoomLevel,
                    this.options.maxZoomLevel
                ])
                .on('zoom', function () { that.zoomed.call(that, d3.event.transform); });
        }

        this.svg = this.$element.append('svg')
            .attr('width', this.options.width)
            .attr('height', this.options.height)
            .call(this.zoom)
            .on('click', function () {
                // Check if we clicked on a object or the canvas - if the user clicked the canvas, we "unprepare" the dragging.
                if (!Element.up(d3.event.target, '.floorplan-object')) {
                    that.unselectObject();
                }

                // @see  FP-29  If any "background" was clicked, we unselect any selected object.
                if (Element.up(d3.event.target, '.vis-background,.vis-background-image')) {
                    const $selectedFloorplanObject = d3.select(Element.up(d3.event.target, '.floorplan-object'));

                    that.unselectObject();

                    // @see  FP-29  But in case this background is inside a object, it's a nested floorplan - we select it :)
                    if ($selectedFloorplanObject.size() && $selectedFloorplanObject.classed('floorplan-object') && $selectedFloorplanObject.classed('level-' + that.uuid)) {
                        const data = $selectedFloorplanObject.data();

                        if (Array.isArray(data) && data.length) {
                            that.selectObject.call(that, data[0]);
                        }
                    }
                }
            })
            .on('dblclick.zoom', null);

        this.svg
            .append('rect')
            .attr('x', 0)
            .attr('y', 0)
            .attr('width', '100%')
            .attr('height', '100%')
            .attr('fill', 'url(#floorplan-grid)')

        // Create the base container, this will contain the pan- and zoomable contents.
        this.$container = this.svg.append('g');

        this.bgLayer = this.$container
            .append('g')
            .attr('class', 'vis-background-image exportable');

        this.background = this.$container
            .append('g')
            .attr('class', 'vis-background exportable');

        this.background
            .append('polygon');
        this.background
            .append('g')
            .attr('class', 'forms');

        this.radiusLayer = this.$container
            .append('g')
            .attr('class', 'vis-radius exportable');

        this.vis = this.$container
            .append('g')
            .attr('class', 'vis-objects exportable');

        this
            .resetBackgroundImage()
            .drawScale()
            .blingbling();

        return this;
    },

    createOptionalDOM: function () {
        // Create grid.
        this.$defs = this.svg.append('defs');

        this.gridinner = this.$defs.append('pattern')
            .attr('id', 'floorplan-grid-inner')
            .attr('patternUnits', 'userSpaceOnUse');
        this.gridinner
            .append('rect')
            .attr('width', '100%')
            .attr('height', '100%')
            .attr('fill', 'none')
            .attr('stroke', '#cccccc')
            .attr('stroke-width', '0.5');

        this.grid = this.$defs.append('pattern')
            .attr('id', 'floorplan-grid')
            .attr('patternUnits', 'userSpaceOnUse');
        this.grid
            .append('rect')
            .attr('id', 'grid')
            .attr('width', '100%')
            .attr('height', '100%')
            .attr('fill', 'url(#floorplan-grid-inner)')
            .attr('stroke', '#cccccc')
            .attr('stroke-width', '1.5');
    },

    resetOptions: function () {
        // This method will reset all the options, that can switch when loading a new floorplan.
        delete this.options.backgroundImage;     // An optional image to be displayed in the background. @todo Do not use this anymore! Use "backgroundImageID".
        delete this.options.backgroundImageID;   // An optional image ID to be displayed in the background.
        delete this.options.backgroundImageSize; // Size array of the backgroundImage as array: [x, y].
        this.options.scaleShow = false;          // Don't show the scale by default.
        this.options.scalePosition = 'nw';       // This option will define where the scale shall be displayed: nw, ne, sw, se or -1 (none)
        this.options.measurementUnitName = 'm';  // Measurement unit title.
        this.options.grid = 1000;                // Contains the grid size (grid * scale = size of the grid).
        this.options.gridDisplay = false;        // Don't show the grid by default.
        this.options.snapToGrid = false;         // Don't snap-to-grid by default.
        this.options.radiusDisplay = false;      // Don't show the object radii by default.
        this.options.scale = 0.1;                // Our default scale: 0.1 pixel equals 1 mm, so 100px equal 1 meter.

        return this;
    },

    getOptions: function () {
        return this.options;
    },

    getOption: function (option) {
        return this.options[option];
    },

    setOption: function (option, value) {
        this.options[option] = value;

        return this;
    },

    getLayoutData: function () {
        return this.layout;
    },

    getFormsData: function () {
        return this.forms;
    },

    extendOptions: function (options) {
        if (Object.isArray(options)) {
            return this;
        }

        Object.extend(this.options, options || {});

        return this;
    },

    setProfile: function (profile) {
        var $style = this.$defs.select('style');

        this.profile = null;

        if (profile.hasOwnProperty('config')) {
            this.profile = profile.config;
        }

        // Re-set the styles, after the profile has been changed.
        if ($style.size() === 0) {
            $style = this.$defs.append('style')
                .attr('type', 'text/css');
        }

        $style
            .text(this.getStyleDefinition());

        return this;
    },

    setEditMode: function (editMode) {
        this.editMode = !!editMode;

        return this;
    },

    blingbling: function () {
        var that = this;

        setTimeout(function () {
            if (that.vis) {
                that.vis.selectAll('.clicked')
                    .select('polygon')
                    .transition()
                    .duration(250)
                    .style('stroke-width', '5px')
                    .transition()
                    .delay(250)
                    .duration(250)
                    .style('stroke-width', '1px')
            }

            that.blingbling();
        }, 1100);
    },

    getStyleDefinition: function () {
        return 'svg .floorplan-object { \
                    fill: none; \
                    stroke: #444; \
                    stroke-width: 1px; \
                } \
                svg #grid line { \
                    stroke: #aaa; \
                    stroke-width: 1px; \
                    stroke-dasharray: 2, 2; \
                } \
                svg .scale-circle { \
                    fill: #5897fb; \
                    stroke: #fff; \
                    stroke-width: 1px; \
                } \
                svg text { \
                    font-family: "Helvetica Neue", "Lucida Grande", "Tahoma", Arial, serif; \
                    font-size: 11px; \
                    stroke: transparent; \
                    stroke-width: 0; \
                    fill: #000; \
                } \
                svg text.title { \
                    font-weight: bold; \
                } \
                svg path.orientation { \
                    fill: #444; \
                    stroke: none; \
                } \
                svg .white-box { \
                    fill: #fff; \
                    stroke: #aaa; \
                    stroke-width: 1px; \
                } \
                svg .drag-rotate-text { \
                    font-size: 10px; \
                } \
                svg rect.drag-overlay { \
                    fill: transparent; \
                    stroke: #fff; \
                    stroke-width: 1px; \
                    stroke-dasharray: 3, 3; \
                } \
                svg .scale-point, \
                svg .scale-path { \
                    stroke: #f00; \
                    stroke-width: 2px; \
                    fill: transparent; \
                    stroke-linecap: round; \
                } \
                svg .vis-radius .clicked { \
                    stroke: #fff; \
                    stroke-width: 3px; \
                } \
                svg .clicked > polygon { \
                    stroke: ' + ((this.profile && this.profile.highlightColor) ? this.profile.highlightColor : '#5897fb') + ';  \
                    stroke-width: 3px; \
                }';
    },

    textFormat: function (text, width) {
        var result = text;

        if (width <= 0) {
            return '';
        }

        d3.select(this).text(result);

        while (this.getComputedTextLength() > width) {
            result = result.substr(0, (result.length - 1));
            d3.select(this).text(result);
        }

        if (result != text) {
            result += '..';
        }

        return result;
    },

    getScreenCenter: function () {
        var transform = this.getCurrentCanvasTransform(),
            translate = [transform.x, transform.y],
            scale     = transform.k;

        return [
            -(translate[0] / scale - (this.options.width / 2) / scale),
            -(translate[1] / scale - (this.options.height / 2) / scale)
        ];
    },

    setBackground: function (layout, forms) {
        this.layout = layout;
        this.forms = forms;

        this.redrawBackground();

        return this;
    },

    redrawBackground: function () {
        const formData = [];

        for (const i in this.forms) {
            if (!this.forms.hasOwnProperty(i)) {
                continue;
            }

            formData.push(createForm(this.forms[i].type)
                .setOptions(this.forms[i].options || {})
                .setStyle(this.forms[i].style || {})
                .setTransform(this.forms[i].transform || {}));
        }

        if (!this.layout) {
            this.background
                .select('polygon')
                .attr('points', '0,0');
        } else {
            this.background
                .select('polygon')
                .attr('transform', 'translate(0,0)scale(' + (this.getUserScale(false) * 10.0) + ')')
                .datum(this.layout)
                .attr('points', (d) => d.data.map((d) => d.x + ',' + d.y).join(' '))
                .style('fill-opacity', (d) => d.style.fillOpacity)
                .style('fill', (d) => d.style.fill)
                .style('stroke-width', (d) => d.style.strokeWidth)
                .style('stroke', (d) => d.style.stroke);
        }

        this.background
            .select('g.forms')
            .attr('transform', 'translate(0,0)scale(' + (this.getUserScale(false) * 10.0) + ')')
            .selectAll('.object')
            .data(formData, (d) => d.id)
            .join(
                function ($enter) {
                    $enter
                        .append((d) => d.render().node());
                },
                function ($update) {
                    // Since we use a external class to render stuff, we need to do a little hack here.
                    $update
                        .attr('data-updated', function (d) {
                            // The DOM will get updated by this call.
                            d.update();

                            return new Date().getTime();
                        });
                },
                function ($exit) {
                    $exit.remove();
                }
            );

        return this;
    },

    resetBackgroundImage: function () {
        var $background = this.bgLayer.select('image');

        if ($background.empty()) {
            $background = this.bgLayer.append('image')
                .attr('x', 0)
                .attr('y', 0);
        }

        if (this.options.hasOwnProperty('backgroundImageID') && this.options.backgroundImageID && Object.isArray(this.options.backgroundImageSize)) {
            $background
                .attr('xlink:href', document.location.href.replace(/\/visualization(.*)/, '') + '/visualization/backgroundImage/' + this.options.backgroundImageID)
                .attr('width', this.options.backgroundImageSize[0])
                .attr('height', this.options.backgroundImageSize[1]);
        } else if (this.options.hasOwnProperty('backgroundImage') && this.options.backgroundImage && Object.isArray(this.options.backgroundImageSize)) {
            $background
                .attr('xlink:href', this.options.backgroundImage)
                .attr('width', this.options.backgroundImageSize[0])
                .attr('height', this.options.backgroundImageSize[1]);
        } else {
            $background
                .attr('xlink:href', '')
                .attr('width', 0)
                .attr('height', 0);
        }

        return this;
    },

    responsive: function () {
        const dimensions = this.getContainerDimensions();

        this.options.width = dimensions.width;
        this.options.height = dimensions.height;
        this.drawScale();

        // Set a certain size to prevent non-looping patterns.
        this.svg
            .attr('width', Math.max(this.options.width, this.getUserScale(true) * this.getGridSize()))
            .attr('height', Math.max(this.options.height, this.getUserScale(true) * this.getGridSize()));

        return this;
    },

    initializeScaleMeasurement: function (callback) {
        this.cancelScaleMeasurement();

        var that            = this,
            center          = this.getScreenCenter(),
            circles, path, data,
            measurementDrag = d3.drag()
                .on('start', function (d) {
                    that.tmp = {
                        $obj:    d3.select(this),
                        clientX: d3.event.sourceEvent.clientX,
                        clientY: d3.event.sourceEvent.clientY,
                        objX:    d.x,
                        objY:    d.y,
                        scale:   that.getCurrentCanvasZoom()
                    };

                    that.vis.classed('currently-dragging', true);
                })
                .on('drag', function (d) {
                    var data;

                    d.x = that.tmp.objX + (d3.event.sourceEvent.clientX - that.tmp.clientX) / that.tmp.scale;
                    d.y = that.tmp.objY + (d3.event.sourceEvent.clientY - that.tmp.clientY) / that.tmp.scale;

                    that.tmp.$obj.attr('transform', (d) => 'translate(' + d.x + ',' + d.y + ')');

                    data = that.vis.selectAll('circle.scale-point').data();

                    if (Object.isFunction(callback)) {
                        callback(data);
                    }

                    that.vis.select('path.scale-path')
                        .attr('d', that.line([
                            [data[0].x, data[0].y],
                            [data[1].x, data[1].y]
                        ]));
                })
                .on('end', function () {
                    that.vis.classed('currently-dragging', false);

                    delete that.tmp;
                });

        // Select the measurement-circles.
        circles = this.vis.selectAll('circle.scale-point');
        path = this.vis.select('path.scale-path');

        // Calculate the current screen-center.
        data = [
            {x: center[0] - 50, y: center[1]},
            {x: center[0] + 50, y: center[1]}
        ];

        if (path.empty()) {
            path = this.vis.append('path').attr('class', 'scale-path');
        }

        if (circles.size() === 0) {
            circles = circles.data(data)
                .enter()
                .append('circle')
                .attr('class', 'scale-point mouse-move')
                .attr('r', 5)
                .call(measurementDrag);
        } else {
            circles.data(data);
        }

        circles
            .attr('transform', (d) => 'translate(' + d.x + ',' + d.y + ')')
            .classed('hide', false);

        path
            .attr('d', this.line([
                [data[0].x, data[0].y],
                [data[1].x, data[1].y]
            ]))
            .classed('hide', false);

        // Set the scalefactor initially, so we can simply "re-save" the setting without breaking the scale.
        if (Object.isFunction(callback)) {
            callback(that.vis.selectAll('circle.scale-point').data());
        }

        return this;
    },

    cancelScaleMeasurement: function () {
        this.vis.selectAll('circle.scale-point').classed('hide', true);
        this.vis.select('path.scale-path').classed('hide', true);
        this.drawScale();

        this.redrawBackground();
        this.updateGrid();

        return this;
    },

    zoomed: function (transform) {
        if (this.vis.classed('currently-dragging')) {
            // Stop moving the canvas, if we are currently dragging something.
            return null;
        }

        // Move the map by translating the container.
        this.$container.attr('transform', transform);

        // Also resize all "rotation" images.
        this.vis.selectAll('.rotate-image,.rotate-text,.toggle-floorplan').dispatch('update:position');

        // Update the grid and responsive state.
        this.updateGrid()
            .responsive()
            .drawScale();
    },

    updateGrid: function () {
        if (!this.options.gridDisplay) {
            this.grid
                .attr('x', null)
                .attr('y', null)
                .attr('width', null)
                .attr('height', null);
            this.gridinner
                .attr('width', null)
                .attr('height', null);

            return this;
        }

        const transform = this.getCurrentCanvasTransform();
        const gridSize = this.getUserScale(true) * this.getGridSize();
        const innerGridSize = gridSize / 10;

        this.grid
            .attr('x', transform.x % gridSize)
            .attr('y', transform.y % gridSize)
            .attr('width', gridSize)
            .attr('height', gridSize);
        this.gridinner
            .attr('width', innerGridSize)
            .attr('height', innerGridSize);

        return this;
    },

    preProcess: function () {
        this.vis.selectAll('g').remove();

        this
            .resetBackgroundImage()
            .drawScale()
            .resetOptions();

        return this;
    },

    renderObject: function ($selection, that) {
        $selection
            .on('click', null)
            .on('dblclick', null)
            .on('click', function (d) {
                if (!that.options.includeHandlers) {
                    // @see  FP-29  Don't handle clicks on nested objects.
                    return;
                }

                that.selectObject.call(that, d);
            })
            .on('dblclick', function (d) {
                if (!that.options.includeHandlers) {
                    // @see  FP-29  Don't handle clicks on nested objects.
                    return;
                }

                if (that.profile && that.profile.hasOwnProperty('dblClickToOpenObject') && that.profile.dblClickToOpenObject == 1) {
                    window.open(window.www_dir + '?objID=' + d.objId);
                }
            })
            .call(function ($subSelection) {
                $subSelection.each(function (d, i, selection) {
                    if (! d.visualizationType) {
                        d.visualizationType = createVisualizationType(selection[i], d, {
                            objectTypeFilter:            that.options.objectTypeFilter,
                            floorplanLevel:              that.options.floorplanLevel,
                            onObjSelect:                 function (d, nested) {
                                // Unselect everything (visually) before setting new 'clicked' classes.
                                that.vis.selectAll('.clicked').classed('clicked', false);
                                that.options.onObjSelect(d, nested);
                            },
                            onObjUnselect:               function (nested) {
                                // Unselect everything (visually).
                                that.vis.selectAll('.clicked').classed('clicked', false);
                                that.options.onObjUnselect(nested);
                            },
                            triggerObjectInformationTab: that.triggerObjectInformationTab.bind(that)
                        });

                        d.visualizationType.render();
                    } else {
                        d.visualizationType.update({
                            scale:        that.options.scale,
                            currentScale: that.getCurrentCanvasZoom(),
                            transition:   false
                        });
                    }
                });
            });
    },

    process: function (transition, centerObjectId) {
        const that = this;

        // At the beginning we create the layers.
        this.vis
            .selectAll('g.layer.level-' + this.uuid)
            .data(this.layers, (d) => 'layer-' + that.uuid + '_' + d.id)
            .join('g')
            .sort((a, b) => b.sort - a.sort)
            .attr('class', 'layer level-' + this.uuid)
            .attr('data-layer-id', (d) => d.id )
            .selectAll('g.floorplan-object.level-' + this.uuid)
            .data(
                function (d) {
                    // Filter by the current layer.
                    return that.data.filter(function (object) {return object.layer == d.id || (object.layer === null && d.id === 0)});
                },
                function (d) {
                    // Use floorplan layer and object id as reference.
                    return 'object-' + that.uuid + '_' + d.objId;
                }
            )
            .join(
                function (enter) {
                    return enter
                        .append('g')
                        .attr('class', 'floorplan-object level-' + that.uuid)
                        .attr('data-id', function (d) { return d.objId; })
                        .call(that.renderObject, that);
                },
                function (update) {
                    return update;
                },
                function (exit) {
                    return exit.remove();
                }
            )
            .sort((a, b) => b.sort - a.sort)
            .attr('data-sort', (d) => d.sort);

        this.radiusLayer
            .selectAll('circle')
            .data(this.data, (d) => d.objId)
            .join('circle')
            .attr('data-id', (d) => d.objId);

        this.updateObjectPositions(undefined, transition);

        this.options.onComplete.call(this, centerObjectId);
    },

    // @see FP-31 Improved the scale by using D3's own axis.
    drawScale: function () {
        var $scale         = this.svg.select('#floorplan-scale'),
            scaleTransform = [20, 20],
            axis,
            linearScale,
            scaleWidth     = this.options.width / 4,
            scaleUnit      = this.options.measurementUnitName || ' m',
            scaleDomain    = scaleWidth / ((this.options.scale * this.getGridSize()) * this.getCurrentCanvasZoom());

        // Check if the scale should be shown.
        $scale.classed('hide', !this.options.scaleShow)

        if (!this.options.scaleShow) {
            return this;
        }

        // Default the scale position to 'nw' (= north-west (= top-left)).
        if (!this.options.hasOwnProperty('scalePosition') || this.options.scalePosition == -1) {
            this.options.scalePosition = 'nw';
        }

        if ($scale.empty()) {
            $scale = this.svg.append('g')
                .attr('id', 'floorplan-scale')
                .attr('class', 'axis');
        }

        linearScale = d3.scaleLinear()
            .range([0, scaleWidth]);

        if (this.options.scalePosition.substr(1, 1) === 'w') {
            linearScale.domain([0, scaleDomain]);
        } else {
            linearScale.domain([scaleDomain, 0]);
            scaleTransform[0] = (this.options.width - scaleWidth - 20);
        }

        if (this.options.scalePosition.substr(0, 1) === 'n') {
            axis = d3.axisBottom(linearScale);
        } else {
            axis = d3.axisTop(linearScale);
            scaleTransform[1] = (this.options.height - 30);
        }

        axis.ticks(5).tickFormat((d) => d + ' ' + scaleUnit);

        $scale
            .attr('transform', 'translate(' + scaleTransform + ')')
            .call(axis);

        return this;
    },

    setLayers: function (layers) {
        this.layers = layers;

        return this;
    },

    getAllLayers: function () {
        return this.layers;
    },

    getLayer: function (id) {
        return this.layers.detect((d) => d.id == id);
    },

    setData: function (data) {
        this.data = data;

        return this;
    },

    getAllData: function () {
        this.sanitizeData();

        return this.data;
    },

    getData: function (id) {
        this.sanitizeData();

        return this.data.detect((d) => d.objId == id);
    },

    sanitizeData: function () {
        var i;

        // We "normalize" the data (round some values etc...).
        for (i in this.data) {
            if (this.data.hasOwnProperty(i)) {
                this.data[i].x = Math.round(this.data[i].x * 100) / 100;
                this.data[i].y = Math.round(this.data[i].y * 100) / 100;
                this.data[i].width = Math.round(this.data[i].width * 100) / 100;
                this.data[i].height = Math.round(this.data[i].height * 100) / 100;
            }
        }
    },

    updateData: function (id, data) {
        var sourceData = this.data.filter((d) => d.objId == id);

        Object.extend(sourceData[0], data || {});

        this.updateObjectPositions(id);
        return this;
    },

    triggerObjectInformationTab: function () {
        if (this.profile && this.profile.hasOwnProperty('changeTabOnObjectSelect') && this.profile.changeTabOnObjectSelect == 1) {
            this.options.openObjectInformationTab();
        }
    },

    selectObject: function (data) {
        var $object = this.vis.select('.level-' + this.uuid + '[data-id="' + data.objId + '"]');

        this.triggerObjectInformationTab();

        if ($object.size() && !$object.classed('clicked')) {
            this.unselectObject();
            this.vis.selectAll('.clicked').classed('clicked', false);

            $object.classed('clicked', true);

            // Also unselect all radii.
            this.radiusLayer.select('[data-id="' + data.objId + '"]').classed('clicked', true).moveToFront();

            this.options.onObjSelect.call(this, data, false);
        }

        if (this.editMode) {
            this.prepareDrag.call(this, data);
        }

        return this;
    },

    unselectObject: function () {
        this.svg.selectAll('.clicked').classed('clicked', false);

        this.unprepareDrag();

        this.options.onObjUnselect.call(this, false);

        return this;
    },

    addObject: function (data, index) {
        var center = this.getScreenCenter();

        // Check if the object has already been assigned.
        if (this.getData(data.objectId)) {
            return;
        }

        if (!index) {
            index = 0;
        }

        this.data.push({
            angle:                 0,
            objId:                 data.objectId,
            objTitle:              data.objectTitle,
            objType:               data.objectTypeId,
            orientation:           null,
            polygon:               this.defaultPolygon,
            width:                 128,
            height:                48,
            x:                     center[0] + (index * 64) - 64,
            y:                     center[1] + (index * 24) - 24,
            sort:                  this.data.length,
            hasOwnBackgroundImage: data.hasOwnBackgroundImage,
            hasOwnFloorplan:       data.hasOwnFloorplan,
            hasOwnLayout:          data.hasOwnLayout
        });

        this.process();

        return this;
    },

    removeObject: function (data) {
        this.data = this.data.filter((d) => d.objId != data.objectId);

        this.process();
    },

    center: function (id) {
        var scale = this.getCurrentCanvasZoom(),
            x     = 0,
            y     = 0,
            data,
            zoomIdentity;

        if (Object.isUndefined(id)) {
            if (this.layout === null) {
                // In case no object was selected, center the background image (if one exists).
                if (Object.isArray(this.options.backgroundImageSize)) {
                    x = (this.options.backgroundImageSize[0] * scale - this.options.width) * -.5;
                    y = (this.options.backgroundImageSize[1] * scale - this.options.height) * -.5;
                }
            } else {
                x = (scale - this.options.width) * -.5;
                y = (scale - this.options.height) * -.5;
            }
        } else {
            data = this.getData(id);

            x = -(data.x * scale) + (this.options.width / 2 - data.width / 2);
            y = -(data.y * scale) + (this.options.height / 2 - data.height / 2);

            this.selectObject(data);
        }

        zoomIdentity = d3
            .zoomIdentity
            .translate(x, y)
            .scale(scale);

        this.svg
            .transition()
            .duration(500)
            .call(this.zoom.transform, zoomIdentity);

        this.drawScale();
    },

    prepareDrag: function (data) {
        this.unprepareDrag();

        var that       = this,
            directions = ['n', 'ne', 'e', 'se', 's', 'sw', 'w', 'nw'],
            $obj       = this.vis.select('.floorplan-object.level-' + this.uuid + '[data-id="' + data.objId + '"]'),
            $innerContent, i;

        // Create the drag-elements for the object.
        $obj.append('rect')
            .attr('class', 'drag-overlay mouse-move')
            .attr('data-role', 'drag-object')
            .call(this.dragMove)
            .on('click', this.selectObject.bind(this));

        for (i in directions) {
            if (!directions.hasOwnProperty(i)) {
                continue;
            }

            $obj.append('circle')
                .attr('r', 3)
                .attr('class', 'scale-circle mouse-move-' + directions[i])
                .attr('data-role', 'scale-' + directions[i])
                .call(this.dragScale);
        }

        $obj.append('image')
            .attr('class', 'rotate-image mouse-crosshair')
            .attr('data-role', 'drag-rotate')
            .attr('width', 16)
            .attr('height', 16)
            .attr('xlink:href', window.dir_images + 'axialis/cad/modify-rotate.svg')
            .on('update:position', function (d) {
                const scale = (1 / that.getCurrentCanvasZoom());

                d3.select(this)
                    .attr('x', 0)
                    .attr('y', -20)
                    .attr('transform', 'translate(-' + (8 * scale) + ',-' + (d.height / 2 + 8 * scale) + ')scale(' + scale + ')');
            })
            .dispatch('update:position')
            .call(this.dragRotate);

        $obj.append('image')
            .attr('class', 'rotate-text mouse-pointer')
            .attr('width', 16)
            .attr('height', 16)
            .attr('xlink:href', window.dir_images + 'axialis/development/debug-step-out.svg')
            .on('update:position', function (d) {
                const scale = (1 / that.getCurrentCanvasZoom());

                d3.select(this)
                    .attr('x', -20)
                    .attr('y', -20)
                    .attr('transform', 'translate(-' + (8 * scale) + ',-' + (d.height / 2 + 8 * scale) + ')scale(' + scale + ')');
            })
            .on('click', (d) => { d.rotateText = !d.rotateText; })
            .dispatch('update:position');

        // @see  FP-29  Add the 'toggle' button.
        if (data.hasOwnFloorplan && (data.hasOwnLayout || data.hasOwnBackgroundImage)) {
            $obj.append('image')
                .attr('class', 'toggle-floorplan mouse-pointer')
                .attr('width', 16)
                .attr('height', 16)
                .attr('title', 'Hallo')
                .attr('xlink:href', window.floorplanWwwPath + 'assets/add-on-icon.svg')
                .on('click', function (d) {
                    d.visualizeAsFloorplan = !d.visualizeAsFloorplan;
                    d.visualizationType = null;

                    $obj.select('.inner-content').remove();

                    that.renderObject($obj, that);
                })
                .on('update:position', function (d) {
                    const scale = (1 / that.getCurrentCanvasZoom());

                    d3.select(this)
                        .attr('x', 20)
                        .attr('y', -20)
                        .attr('transform', 'translate(-' + (8 * scale) + ',-' + (d.height / 2 + 8 * scale) + ')scale(' + scale + ')');
                })
                .dispatch('update:position');
        }

        this.updateObjectPositions(data.objId);

        return this;
    },

    unprepareDrag: function () {
        this.vis
            .selectAll('[data-role^="drag"],[data-role^="rotate"],[data-role^="scale"],.rotate-text,.toggle-floorplan')
            .remove();

        return this;
    },

    updateObjectPositions: function (object, transition) {
        var $selection, $radii,
            that = this;

        if (!Object.isUndefined(object)) {
            $selection = this.vis.select('.level-' + this.uuid + '[data-id="' + object + '"]');
            $radii = this.radiusLayer.select('[data-id="' + object + '"]')
        } else {
            $selection = this.vis.selectAll('g.floorplan-object.level-' + this.uuid);
            $radii = this.radiusLayer.selectAll('circle')
        }

        $radii
            .classed('hide', function (d) {
                return !that.options.radiusDisplay ||
                       that.options.objectTypeFilter.in_array(d.objType) ||
                       !(d.hasOwnProperty('radius') && d.radius.hasOwnProperty('display') && d.radius.display);
            })
            .filter(function (d) {
                return that.options.radiusDisplay &&
                       !that.options.objectTypeFilter.in_array(d.objType) &&
                       d.hasOwnProperty('radius') &&
                       d.radius.hasOwnProperty('display') &&
                       d.radius.display;
            })
            .interrupt()
            .transition()
            .duration(transition ? 400 : 0)
            .attr('r', (d) => that.getUserScale(false) * (d.radius.radius * window.floorplanData.scaleFactors[d.radius.unit]))
            .style('fill', (d) => d.radius.color)
            .style('fill-opacity', (d) => d.radius.opacity / 100)
            .attr('transform', (d) => 'translate(' + d.x + ',' + d.y + ')');

        $selection
            .classed('hide', (d) => that.options.objectTypeFilter.includes(d.objType))
            .each(function (d) {
                // @see FP-140 FP-141 Only update the visualization, if it is not hidden.
                if (d.visualizationType.ready && !that.options.objectTypeFilter.includes(d.objType)) {
                    // @see FP-29 Use the visualization type to update the objects 'inner content'.
                    d.visualizationType.update({
                        scale:        that.options.scale,
                        currentScale: that.getCurrentCanvasZoom(),
                        transition:   transition
                    });
                }
            });

        return this;
    },

    getCurrentCanvasTransform: function () {
        return d3.zoomTransform(this.svg.node());
    },

    getCurrentCanvasZoom: function () {
        return this.getCurrentCanvasTransform().k;
    },

    /**
     * The 1000 (mm) default equals 1 meter
     *
     * @returns {number|boolean}
     */
    getGridSize: function () {
        if (!isNaN(parseFloat(this.options.grid)) && isFinite(this.options.grid)) {
            return this.options.grid;
        }

        return 1000;
    },

    getUserScale: function (includeZoom) {
        const zoom = includeZoom ? this.getCurrentCanvasZoom() : 1;

        return parseFloat(this.options.scale * zoom);
    },

    setUserScale: function (newScale) {
        this.options.scale = parseFloat(newScale);

        this.responsive();
        this.process();

        this.redrawBackground();
        this.updateGrid();
    }
});
