<?php

namespace idoit\Module\Floorplan\Controller;

use idoit\Exception\Exception;
use idoit\Module\Floorplan\Model\Floorplan;
use idoit\Module\Floorplan\Model\Layer;
use isys_application as Application;
use isys_cmdb_dao_category_g_formfactor as DaoFormfactor;
use isys_cmdb_dao_category_g_images as DaoImages;
use isys_cmdb_dao_category_g_location as DaoLocation;
use isys_convert as Convert;
use isys_format_json as JSON;
use isys_helper as Helper;
use isys_register as Register;

/**
 * i-doit cmdb controller
 *
 * @package     Modules
 * @subpackage  Floorplan
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class Ajax extends Main
{
    /**
     * @var array
     */
    protected $response;

    /**
     * Overwriting the "handle" method.
     *
     * @param Register    $request
     * @param Application $application
     *
     * @return  null
     */
    public function handle(Register $request, Application $application)
    {
        return null;
    }

    /**
     * Pre method gets called by the framework.
     *
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function pre()
    {
        header('Content-Type: application/json');

        $this->response = [
            'success' => true,
            'data'    => null,
            'message' => null
        ];
    }

    /**
     * Method for retrieving the necessary floorplan (and object-) data.
     *
     * @param Register $request
     */
    public function getFloorplanData(Register $request)
    {
        try {
            $language = $this->getDi()->get('language');
            $objectId = $request->get('POST')->get('objectId');

            if ($objectId > 0) {
                $dao = $this->dao(Application::instance());
                $floorplanLayout = null;
                $backgroundImages = $objects = $floorplanConfiguration = $floorplanForms = [];

                $resImages = DaoImages::instance($this->getDi()->get('database'))->get_data(null, $objectId);

                if (count($resImages)) {
                    while ($imageRow = $resImages->get_row()) {
                        $backgroundImages[] = $imageRow['isys_catg_images_list__id'];
                    }
                }

                $objects = [];
                $layers = [];
                $floorplan = $dao->getFloorplan($objectId)->get_row();

                // Check if there's a saved floorplan.
                if (is_array($floorplan) && !empty($floorplan)) {
                    $floorplanLayout = JSON::decode($floorplan['isys_floorplan__layout']);
                    $floorplanForms = (array)JSON::decode($floorplan['isys_floorplan__forms']);
                    $floorplanConfiguration = JSON::decode($floorplan['isys_floorplan__configuration']);

                    // @see  FP-84  Reformat some options to fit the new format.
                    if (isset($floorplanConfiguration['scaleFactor'], $floorplanConfiguration['scaleFactorMM'])) {
                        $floorplanConfiguration['scale'] = $floorplanConfiguration['scaleFactor'] / $floorplanConfiguration['scaleFactorMM'];
                    }

                    if (empty($floorplanConfiguration['backgroundImage'])) {
                        unset($floorplanConfiguration['backgroundImage']);
                    }

                    unset(
                        $floorplanConfiguration['displayScale'],
                        $floorplanConfiguration['measurementValue'],
                        $floorplanConfiguration['measurementUnit'],
                        $floorplanConfiguration['scaleFactor'],
                        $floorplanConfiguration['scaleFactorMM']
                    );

                    $layerRes = Layer::instance($this->getDi()->get('database'))
                        ->getData('AND isys_floorplan_layers__isys_floorplan__id = ' . $dao->convert_sql_id($floorplan['isys_floorplan__id']));

                    while ($row = $layerRes->get_row()) {
                        $layers[$row['id']] = $row;
                        $layers[$row['id']]['configuration'] = JSON::decode($row['configuration']);
                    }

                    $objectRes = $dao->getObjectsByFloorplan($objectId);

                    while ($row = $objectRes->get_row()) {
                        $objects[$row['isys_obj__id']] = JSON::decode($row['isys_floorplan_objects__configuration']);
                        $configuration = [
                            'backgroundImageSize' => null
                        ];

                        if ($row['configuration'] !== null && JSON::is_json_array($row['configuration'])) {
                            $configuration = JSON::decode($row['configuration']);
                        }

                        // This is necessary for "old" objects without angle.
                        $objects[$row['isys_obj__id']]['angle'] = (int)$objects[$row['isys_obj__id']]['angle'];
                        $objects[$row['isys_obj__id']]['objId'] = (int)$row['isys_obj__id'];
                        $objects[$row['isys_obj__id']]['objTitle'] = $language->get($row['isys_obj__title']);
                        $objects[$row['isys_obj__id']]['objType'] = (int)$row['isys_obj__isys_obj_type__id'];
                        $objects[$row['isys_obj__id']]['x'] = $row['lat'] ?: 0;
                        $objects[$row['isys_obj__id']]['y'] = $row['lng'] ?: 0;
                        $objects[$row['isys_obj__id']]['layer'] = isset($layers[$row['isys_floorplan_objects__isys_floorplan_layers']])
                            ? (int)$row['isys_floorplan_objects__isys_floorplan_layers']
                            : null;
                        $objects[$row['isys_obj__id']]['sort'] = (int)$row['isys_floorplan_objects__sort'];
                        $objects[$row['isys_obj__id']]['visibility'] = (float)($objects[$row['isys_obj__id']]['visibility'] ?? 1);
                        $objects[$row['isys_obj__id']]['hasOwnFloorplan'] = (bool)$row['hasOwnFloorplan'];
                        $objects[$row['isys_obj__id']]['hasOwnLayout'] = (bool)$row['hasOwnLayout'];
                        $objects[$row['isys_obj__id']]['hasOwnBackgroundImage'] = $configuration['backgroundImageSize'];

                        if (!isset($objects[$row['isys_obj__id']]['visualizeAsFloorplan'])) {
                            $objects[$row['isys_obj__id']]['visualizeAsFloorplan'] = false;
                        }
                    }
                }

                $DaoLocation = DaoLocation::instance($this->getDi()->get('database'));

                $rootObjectData = $DaoLocation->get_data(null, $objectId)->get_row();

                // Get the location path - this is necessary to open the location tree in the frontend.
                $locationPath = $DaoLocation->get_location_path($objectId);
                $locationPath[] = C__OBJ__ROOT_LOCATION;
                $locationPath = array_reverse($locationPath);
                $locationPath[] = $objectId;

                // Set only necessary values in a "modern" form.
                $rootObjectData = [
                    'objectId'        => $rootObjectData['isys_obj__id'],
                    'objectTitle'     => $rootObjectData['isys_obj__title'],
                    'objectTypeId'    => $rootObjectData['isys_obj_type__id'],
                    'objectTypeTitle' => $language->get($rootObjectData['isys_obj_type__title']),
                    'locationPath'    => $locationPath
                ];

                $this->response['data'] = [
                    'rootData'         => $rootObjectData,
                    'layout'           => $floorplanLayout,
                    'forms'            => $floorplanForms,
                    'objectData'       => array_values($objects),
                    'layerData'        => array_values($layers),
                    'floorplanOptions' => $floorplanConfiguration,
                    'images'           => $backgroundImages
                ];
            } else {
                $this->response['data'] = [
                    'objectData'       => [],
                    'floorplanOptions' => [],
                    'images'           => [],
                ];
            }
        } catch (\Exception $e) {
            $this->response['success'] = false;
            $this->response['message'] = $e->getMessage();
        }
    }

    /**
     * Method for retrieving object information by a given profile.
     *
     * @param Register $request
     */
    public function getObjectInformation(Register $request)
    {
        try {
            $objectId = (int)$request->get('id');
            $profileId = (int)$request->get('POST')->get('profileId');

            $this->response['data'] = $this->dao(Application::instance())->getObjectInformation($objectId, $profileId);
        } catch (\Exception $e) {
            $this->response['success'] = false;
            $this->response['message'] = $e->getMessage();
        }
    }

    /**
     * Method for uploading a new floorplan background Image.
     *
     * @param Register $request
     */
    public function uploadBackgroundImage(Register $request)
    {
        try {
            $this->response['data'] = (new DaoImages($this->getDi()->get('database')))->handle_upload($request->get('id'));
        } catch (\Exception $e) {
            $this->response['success'] = false;
            $this->response['message'] = $e->getMessage();
        }
    }

    /**
     * Method for uploading a new floorplan background Image.
     *
     * @param Register $request
     */
    public function removeBackgroundImage(Register $request)
    {
        try {
            $this->response['data'] = (new DaoImages($this->getDi()->get('database')))->delete_image($request->get('id'));
        } catch (\Exception $e) {
            $this->response['success'] = false;
            $this->response['message'] = $e->getMessage();
        }
    }

    /**
     * Method for creating/saving formfactor data of an object.
     *
     * @param Register $request
     */
    public function saveFormfactorData(Register $request)
    {
        try {
            $post = $request->get('POST');
            $objectId = $request->get('id');
            $DaoFormfactor = new DaoFormfactor($this->getDi()->get('database'));

            $this->response['data'] = $DaoFormfactor->save_single_value($objectId, [
                'width'       => Helper::filter_number($post->get('width', 0)),
                'height'      => Helper::filter_number($post->get('height', 0)),
                'depth'       => Helper::filter_number($post->get('depth', 0)),
                'unit'        => $post->get('unit', 0),
                'weight'      => Helper::filter_number($post->get('weight', 0)),
                'weight_unit' => $post->get('weight_unit', 0),
            ]);
        } catch (\Exception $e) {
            $this->response['success'] = false;
            $this->response['message'] = $e->getMessage();
        }
    }

    /**
     * Post method gets called by the framework.
     *
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function post()
    {
        echo JSON::encode($this->response);
        die;
    }
}
