<?php

namespace idoit\Module\SyneticsFlows\Automation\Trigger\ConditionBuilder;

use idoit\Module\SyneticsFlows\Automation\Trigger\Condition\AndCondition;
use idoit\Module\SyneticsFlows\Automation\Trigger\Condition\CmdbCondition;
use idoit\Module\SyneticsFlows\Automation\Trigger\Condition\Condition;
use idoit\Module\SyneticsFlows\Automation\Trigger\Condition\OrCondition;
use idoit\Module\SyneticsFlows\Dto\Criteria;
use idoit\Module\SyneticsFlows\Dto\Criteria\AndCriteria;
use idoit\Module\SyneticsFlows\Dto\Criteria\ArrayCriteria;
use idoit\Module\SyneticsFlows\Dto\Criteria\InCriteria;
use idoit\Module\SyneticsFlows\Dto\Criteria\NotInCriteria;
use idoit\Module\SyneticsFlows\Dto\Criteria\OrCriteria;
use idoit\Module\SyneticsFlows\Dto\Criteria\ValueCriteria;
use isys_array;
use isys_cmdb_dao_category_property;
use isys_component_database;
use Throwable;

class CmdbConditionBuilder implements ConditionBuilder
{
    public function __construct(
        private isys_component_database $database
    )
    {
    }

    /**
     * @param Condition $condition
     *
     * @return array|null
     */
    private function buildCondition(Condition $condition)
    {
        $result = [];
        if ($condition instanceof AndCondition || $condition instanceof OrCondition) {
            $items = $condition->getItems();
            $length = count($items);
            $operator = $condition instanceof AndCondition ? 'AND' : 'OR';
            foreach ($items as $item) {
                $length--;
                if (!$item instanceof Condition && !$item instanceof Criteria) {
                    continue;
                }

                if ($item instanceof Condition) {
                    $result[] = $this->buildCondition($item);
                }

                if ($item instanceof Criteria) {
                    $result[] = $this->buildCriteria($item);
                }

                if ($length > 0) {
                    $result[] = $operator;
                }
            }
            return $result;
        }

        if ($condition instanceof CmdbCondition) {
            $criteria = $condition->getCriteria();
            return $this->buildCriteria($criteria, null);
        }

        return null;
    }

    /**
     * @param Criteria    $criteria
     * @param string|null $operator
     *
     * @return array|null
     */
    private function buildCriteria(Criteria $criteria, ?string $operator = null)
    {
        if ($criteria instanceof OrCriteria || $criteria instanceof AndCriteria) {
            $items = $criteria->getItems();
            $length = count($items);
            $operator = $criteria instanceof AndCriteria ? 'AND' : 'OR';
            $result = [];
            foreach ($items as $item) {
                $length--;
                if (!$item instanceof Criteria) {
                    continue;
                }

                if ($length === 0) {
                    $operator = null;
                }
                $result[] = $this->buildCriteria($item, $operator);
            }

            return $result;
        }

        $condition = [];
        [$category] = explode('.', $criteria->getField());

        if ($criteria instanceof ValueCriteria || $criteria instanceof ArrayCriteria) {
            $condition = [
                'category' => $category,
                'property' => str_replace('.', '-', $criteria->getField()),
                'comparison' => $criteria instanceof InCriteria ? '=' : ($criteria instanceof NotInCriteria ? '!=' : $criteria->getComparisonOperator()),
                'value' => !empty($criteria->getValue()) || $criteria->getValue() === '' ? $criteria->getValue() : null,
            ];

            if ($operator !== null) {
                $condition['operator'] = $operator;
            }

            return $condition;
        }

        return null;
    }

    /**
     * @param Condition $condition
     *
     * @return string
     * @throws ConditionBuildException
     */
    public function build(Condition $condition): string
    {
        $conditions = $this->buildCondition($condition);

        try {
            $queryBuilder = new isys_cmdb_dao_category_property($this->database);
            $queryBuilder->set_query_as_report(true)
                ->reset();

            $queryConditions = $queryBuilder->create_property_query_condition([$conditions]);

            $joins = array_filter((new isys_array($queryConditions['joins']))->flatten());

            $pattern = "isys_obj__id IN (SELECT obj_main.isys_obj__id FROM isys_obj as obj_main %s WHERE TRUE %s)";

            return sprintf($pattern, implode(' ', $joins), $queryConditions['conditions']);
        } catch (Throwable $e) {
            throw ConditionBuildException::buildError('CMDB', $e->getMessage());
        }
    }
}
