#!/bin/bash

cd "$(dirname "$0")" || exit 1;

while getopts u:p:i: flag
do
    case "${flag}" in
        u) IDOIT_CONFIG_USERNAME=${OPTARG};;
        p) IDOIT_CONFIG_PASSWORD=${OPTARG};;
        i) IDOIT_CONFIG_TENANT=${OPTARG};;
        *) exit 1;;
    esac
done

if [ -z "${IDOIT_CONFIG_USERNAME}" ] && [ -z "${IDOIT_CONFIG_PASSWORD}" ] && [ -z "${IDOIT_CONFIG_TENANT}" ]; then
    echo "i-doit flows daemon creation"
    echo "============================"
    echo "This script creates an .service file to execute all necessary console commands to perform flows."
    echo "============================"
    echo "USAGE"
    echo "Please specify credentials of an i-doit user and also the tenant id the daemon should use in flow executions."
    echo ""
    echo "-u: Username of i-doit user"
    echo "-p: Password of i-doit user"
    echo "-i: Id of tenant the user is placed in"
    echo "============================"
    echo "EXAMPLE"
    echo "sudo ./create-daemon.sh -u admin -p admin -i 1"
    echo "============================"
    echo "!!! PLEASE RUN AS SUDO !!!"
    exit 1;
fi

if [ -z "${IDOIT_CONFIG_USERNAME}" ]; then
    echo "Please specify -u to set i-doit username."
    exit 1;
fi

if [ -z "${IDOIT_CONFIG_PASSWORD}" ]; then
    echo "Please specify -p to set i-doit password."
    exit 1;
fi

if [ -z "${IDOIT_CONFIG_TENANT}" ]; then
    echo "Please specify -i to set i-doit tenant id."
    exit 1;
fi

SERVICE_USER=""
if id -u "wwwrun" >/dev/null 2>&1; then
    SERVICE_USER="wwwrun"
fi

if id -u "www-data" >/dev/null 2>&1; then
    SERVICE_USER="www-data"
fi

if id -u "apache" >/dev/null 2>&1; then
    SERVICE_USER="apache"
fi

if [ -z "${SERVICE_USER}" ]; then
    echo "Unable to identify webserver user which is required for daemon definition."
    exit 1;
fi

IDOIT_FLOWS_UNIT="i-doit-flows-${IDOIT_CONFIG_TENANT}"
IDOIT_FLOWS_DIR=$(pwd)
IDOIT_ROOT_DIR=${IDOIT_FLOWS_DIR/"/src/classes/modules/synetics_flows"/""}

TARGET_UNIT_FILE="/lib/systemd/system/${IDOIT_FLOWS_UNIT}.service"

if [ -f "$TARGET_UNIT_FILE" ]; then
   echo "The unit file for tenant with Id #${IDOIT_CONFIG_TENANT} already exists. Please delete it first and try again."
   echo "---------------------------"
   echo "$TARGET_UNIT_FILE"
   exit 1;
fi

cat <<EOT > "$TARGET_UNIT_FILE"
[Unit]
Description=Flow Daemon for Tenant #${IDOIT_CONFIG_TENANT}
After=network.target
StartLimitIntervalSec=0

[Service]
Type=simple
Restart=always
RestartSec=5
SyslogIdentifier=${IDOIT_FLOWS_UNIT}
User=${SERVICE_USER}
Environment="IDOIT_USERNAME=${IDOIT_CONFIG_USERNAME}"
Environment="IDOIT_PASSWORD=${IDOIT_CONFIG_PASSWORD}"
Environment="IDOIT_TENANT_ID=${IDOIT_CONFIG_TENANT}"
WorkingDirectory=${IDOIT_ROOT_DIR}
ExecStartPre=php ./console.php flows:time-trigger -u\${IDOIT_USERNAME} -p\${IDOIT_PASSWORD} -i \${IDOIT_TENANT_ID}
ExecStart=php ./console.php flows:perform -u\${IDOIT_USERNAME} -p\${IDOIT_PASSWORD} -i \${IDOIT_TENANT_ID}

[Install]
WantedBy=multi-user.target

EOT

systemctl daemon-reload
systemctl enable "${IDOIT_FLOWS_UNIT}"
systemctl start "${IDOIT_FLOWS_UNIT}"
systemctl status "${IDOIT_FLOWS_UNIT}"

exit 0;
