<?php declare(strict_types = 1);

namespace idoit\Module\Forms\Exceptions;

use Exception;
use Symfony\Component\HttpFoundation\Response;

class DataByAttributeException extends Exception
{
    /**
     * @return DataByAttributeException
     */
    public static function NoAttributeInRequest()
    {
        return new self("There is no attribute id specified in your request!", Response::HTTP_BAD_REQUEST);
    }

    /**
     * @return DataByAttributeException
     */
    public static function NoCategoryConstantDefined()
    {
        return new self("Category constant is not defined in the attribute id", Response::HTTP_BAD_REQUEST);
    }

    /**
     * @return DataByAttributeException
     */
    public static function NoPropertyKeyDefined()
    {
        return new self("Property key is not defined.", Response::HTTP_BAD_REQUEST);
    }

    /**
     * @param string $categoryConst
     *
     * @return DataByAttributeException
     */
    public static function CategoryDaoDoesNotExist(string $categoryConst)
    {
        return new self("The category dao for '{$categoryConst}' does not exist.", Response::HTTP_NOT_FOUND);
    }

    /**
     * @param string $categoryConst
     *
     * @return DataByAttributeException
     */
    public static function CategoryIsBlocked(string $categoryConst)
    {
        return new self("The category for '{$categoryConst}' is blocked.", Response::HTTP_BAD_REQUEST);
    }

    /**
     * @param string $categoryConst
     * @param string $propertyKey
     *
     * @return DataByAttributeException
     */
    public static function CategoryPropertyIsBlocked(string $categoryConst, string $propertyKey)
    {
        return new self(
          "The property '{$propertyKey}' of category '{$categoryConst}' is blocked.",
          Response::HTTP_BAD_REQUEST
        );
    }

    /**
     * @param string $propertyKey
     * @param string $categoryConst
     *
     * @return DataByAttributeException
     */
    public static function NoCollectorFound(string $propertyKey, string $categoryConst)
    {
        return new self(
          "Attribute data collector for Property '{$propertyKey}' in category '{$categoryConst}' is not applicable.",
          Response::HTTP_NOT_FOUND
        );
    }

    /**
     * @param string $key
     * @param string $message
     *
     * @return DataByAttributeException
     */
    public static function FailedToCollectData(string $key, string $message)
    {
        return new self(
          "Could not collect Data for property {$key} with message: {$message}",
          Response::HTTP_NOT_FOUND
        );
    }

    /**
     * @param string $propertyKey
     * @param string $categoryConst
     *
     * @return DataByAttributeException
     */
    public static function PropertyDoesNotExistInCategory(string $propertyKey, string $categoryConst)
    {
        return new self(
          "Property '{$propertyKey}' does not exist in category '{$categoryConst}'.",
          Response::HTTP_NOT_FOUND
        );
    }
}
