<?php

namespace idoit\Module\Syneticsgmbh_homeoffice\Processor;

use idoit\Module\Cmdb\Model\CiTypeCategoryAssigner;
use idoit\Module\Syneticsgmbh_homeoffice\Processor;
use isys_application as Application;
use isys_cmdb_dao as CmdbDao;
use isys_cmdb_dao_dialog as DialogDao;
use isys_component_database;
use isys_tenantsettings as TenantSetting;

/**
 * Class Update
 *
 * @package   idoit\Module\Syneticsgmbh_homeoffice\Processor
 * @copyright synetics GmbH
 * @license   https://www.i-doit.com
 */
class Update extends Processor
{
    /**
     * @var int
     */
    private $tenantId;

    /**
     * Activate constructor.
     *
     * @param isys_component_database $tenantDatabase
     * @param int                     $tenantId
     */
    public function __construct(isys_component_database $tenantDatabase, int $tenantId)
    {
        parent::__construct($tenantDatabase);

        $this->tenantId = $tenantId;
    }

    /**
     * @throws \idoit\Exception\JsonException
     * @throws \isys_exception_dao
     * @throws \isys_exception_database
     */
    public function process()
    {
        if (!\defined('C__IMPORT__DIRECTORY')) {
            $importUploadDirectory = TenantSetting::get('system.dir.import-uploads', rtrim(BASE_DIR, '/') . '/imports/');

            \define('C__IMPORT__DIRECTORY', rtrim($importUploadDirectory, '/') . '/' . Application::instance()->tenant->id . '/');
        }

        $objectTypeAssignmentData = $this->getDataFileContent('object-type-assignments.json');

        if (\count($objectTypeAssignmentData)) {
            $this->globalCategoryAssignments($objectTypeAssignmentData);
        }

        $customCategoryAssignmentData = $this->getDataFileContent('custom-categories.json');

        if (\count($customCategoryAssignmentData)) {
            $this->customCategoryAssignments($customCategoryAssignmentData);
        }

        $reportData = $this->getDataFileContent('reports.json');

        if (\count($reportData)) {
            $this->setReportTenants($reportData);
        }

        $dialogPlusData = $this->getDataFileContent('dialog-plus.json');

        if (\count($dialogPlusData)) {
            $this->synchronizeDialogPlusData($dialogPlusData);
        }

        $this->copyDataFiles('csv', C__IMPORT__DIRECTORY);
        $this->copyDataFiles('xml', C__IMPORT__DIRECTORY);
    }

    /**
     * @param array $objectTypeAssignmentData
     *
     * @return void
     * @throws \isys_exception_dao
     * @throws \isys_exception_database
     */
    private function globalCategoryAssignments(array $objectTypeAssignmentData)
    {
        $dao = CmdbDao::instance($this->database);

        foreach ($objectTypeAssignmentData as $objectType) {
            // Since the `CiTypeCategoryAssigner` seems to remove previous assignments, we do it here with plain SQL.
            $objectTypeConstant = $dao->convert_sql_text($objectType['constant']);

            foreach ($objectType['global'] as $globalCategory) {
                $categoryConstant = $dao->convert_sql_text($globalCategory);

                $checkSql = "SELECT isys_obj_type_2_isysgui_catg__id
                    FROM isys_obj_type_2_isysgui_catg
                    INNER JOIN isys_obj_type ON isys_obj_type__id = isys_obj_type_2_isysgui_catg__isys_obj_type__id
                    INNER JOIN isysgui_catg ON isysgui_catg__id = isys_obj_type_2_isysgui_catg__isysgui_catg__id
                    WHERE isys_obj_type__const = {$objectTypeConstant}
                    AND isysgui_catg__const = {$categoryConstant} 
                    LIMIT 1;";

                // Only do the assignment, if it does not exist yet.
                if (!\count($dao->retrieve($checkSql))) {
                    // Retrieve the object type ID.
                    $objectTypeId = (int)$dao
                        ->retrieve("SELECT isys_obj_type__id AS id FROM isys_obj_type WHERE isys_obj_type__const = {$objectTypeConstant} LIMIT 1")
                        ->get_row_value('id');

                    if ($objectTypeId === 0) {
                        continue;
                    }

                    // Retrieve the category ID.
                    $globalCategoryId = (int)$dao
                        ->retrieve("SELECT isysgui_catg__id AS id FROM isysgui_catg WHERE isysgui_catg__const = {$categoryConstant} LIMIT 1")
                        ->get_row_value('id');

                    if ($globalCategoryId === 0) {
                        continue;
                    }

                    $assignSql = "INSERT INTO isys_obj_type_2_isysgui_catg
                        SET isys_obj_type_2_isysgui_catg__isys_obj_type__id = {$objectTypeId},
                        isys_obj_type_2_isysgui_catg__isysgui_catg__id = {$globalCategoryId};";

                    $dao->update($assignSql) && $dao->apply_update();
                }
            }
        }
    }

    /**
     * @param array $customCategoryData
     *
     * @return void
     * @throws \isys_exception_dao
     * @throws \isys_exception_database
     */
    private function customCategoryAssignments(array $customCategoryData)
    {
        $dao = CmdbDao::instance($this->database);

        foreach ($customCategoryData as $customCategory) {
            // Since the `CiTypeCategoryAssigner` seems to remove previous assignments, we do it here with plain SQL.
            $categoryConstant = $dao->convert_sql_text($customCategory['constant']);

            // @see  PACKAGER-18  Process referenced reports via constant, not via ID.
            $configurationChanged = false;
            $categoryConfiguration = unserialize($customCategory['configuration'], ['allowed_classes' => false]);

            foreach ($categoryConfiguration as &$fieldConfiguration) {
                if ($fieldConfiguration['type'] === 'f_popup' && $fieldConfiguration['popup'] === 'report_browser' && !empty($fieldConfiguration['identifier'])) {
                    // Find the report with the given constant.
                    $reportSql = 'SELECT isys_report__id AS id
                        FROM isys_report
                        WHERE isys_report__const = ' . $dao->convert_sql_text($fieldConfiguration['identifier']) . '
                        LIMIT 1;';

                    $fieldConfiguration['identifier'] = (int)$dao->retrieve($reportSql)->get_row_value('id');
                    $configurationChanged = true;
                }
            }

            // Update the configuration.
            if ($configurationChanged) {
                $configuration = $dao->convert_sql_text(serialize($categoryConfiguration));

                $categoryUpdateQuery = "UPDATE isysgui_catg_custom 
                    SET isysgui_catg_custom__config = {$configuration}
                    WHERE isysgui_catg_custom__const = {$categoryConstant}
                    LIMIT 1;";

                $dao->update($categoryUpdateQuery) && $dao->apply_update();
            }

            foreach ($customCategory['objectTypeAssignment'] as $objectType) {
                $objectTypeConstant = $dao->convert_sql_text($objectType);

                $checkSql = "SELECT isys_obj_type_2_isysgui_catg_custom__id
                    FROM isys_obj_type_2_isysgui_catg_custom
                    INNER JOIN isys_obj_type ON isys_obj_type__id = isys_obj_type_2_isysgui_catg_custom__isys_obj_type__id
                    INNER JOIN isysgui_catg_custom ON isysgui_catg_custom__id = isys_obj_type_2_isysgui_catg_custom__isysgui_catg_custom__id
                    WHERE isys_obj_type__const = {$objectTypeConstant}
                    AND isysgui_catg_custom__const = {$categoryConstant} 
                    LIMIT 1;";

                // Only do the assignment, if it does not exist yet.
                if (!\count($dao->retrieve($checkSql))) {
                    // Retrieve the object type ID.
                    $objectTypeId = (int)$dao
                        ->retrieve("SELECT isys_obj_type__id AS id FROM isys_obj_type WHERE isys_obj_type__const = {$objectTypeConstant} LIMIT 1")
                        ->get_row_value('id');

                    if ($objectTypeId === 0) {
                        continue;
                    }

                    // Retrieve the category ID.
                    $customCategoryId = (int)$dao
                        ->retrieve("SELECT isysgui_catg_custom__id AS id FROM isysgui_catg_custom WHERE isysgui_catg_custom__const = {$categoryConstant} LIMIT 1")
                        ->get_row_value('id');

                    if ($customCategoryId === 0) {
                        continue;
                    }

                    $assignSql = "INSERT INTO isys_obj_type_2_isysgui_catg_custom
                        SET isys_obj_type_2_isysgui_catg_custom__isys_obj_type__id = {$objectTypeId},
                        isys_obj_type_2_isysgui_catg_custom__isysgui_catg_custom__id = {$customCategoryId};";

                    $dao->update($assignSql) && $dao->apply_update();
                }
            }
        }
    }

    /**
     * @param array $reports
     *
     * @return void
     * @throws \isys_exception_dao
     */
    private function setReportTenants(array $reports)
    {
        $dao = CmdbDao::instance($this->database);

        if (!$dao->table_exists('isys_report')) {
            return;
        }

        foreach ($reports as $report) {
            /*
             * While moving the reports from the system to the tenant DB, the field 'isys_report__mandator' remains intact.
             * This means we have to set the correct tenant ID in order to see the reports in the GUI.
             */
            $sql = 'UPDATE isys_report
                SET isys_report__mandator = ' . $dao->convert_sql_id($this->tenantId) . '
                WHERE isys_report__const = ' . $dao->convert_sql_text($report['constant']) . '
                LIMIT 1;';

            $dao->update($sql) && $dao->apply_update();
        }
    }

    /**
     * @param array $dialogPlusData
     *
     * @return void
     * @throws \Exception
     * @throws \isys_exception_dao
     * @throws \isys_exception_database
     */
    private function synchronizeDialogPlusData(array $dialogPlusData)
    {
        $iteratedTables = [];

        foreach ($dialogPlusData as $table => $entries) {
            // Check if we already iterated this table.
            if (\in_array($table, $iteratedTables, true)) {
                continue;
            }

            $referenceTable = null;

            // Check if we have a reference.
            foreach ($entries[0] as $key => $value) {
                if (isset($dialogPlusData[$key]) && strpos($key, 'isys_') === 0 && !\in_array($key, $iteratedTables, true)) {
                    $referenceTable = $key;

                    // The "referenced" table needs to be processed first!
                    $this->processDialogTable($referenceTable, $dialogPlusData[$referenceTable]);

                    // Add the table as "iterated".
                    $iteratedTables[] = $referenceTable;
                }
            }

            // Iterate the current table.
            $this->processDialogTable($table, $entries, $referenceTable);

            // Add the table as "iterated".
            $iteratedTables[] = $table;
        }
    }

    /**
     * @param string      $table
     * @param array       $entries
     * @param string|null $parentTable
     *
     * @return void
     * @throws \Exception
     * @throws \isys_exception_dao
     * @throws \isys_exception_database
     */
    private function processDialogTable(string $table, array $entries, string $parentTable = null)
    {
        $language = Application::instance()->container->get('language');
        $dao = CmdbDao::instance($this->database);

        $dialogDao = null;
        $identifier = null;
        $identifierCondition = 'TRUE';

        if ($parentTable) {
            $dialogDao = DialogDao::instance($this->database)
                ->set_table($parentTable)
                ->load();
        }

        if (strpos($table, 'isys_dialog_plus_custom:') === 0) {
            list($table, $identifier) = explode(':', $table);

            $identifierCondition = 'isys_dialog_plus_custom__identifier = ' . $dao->convert_sql_text($identifier);
        }

        foreach ($entries as $entry) {
            $parentData = null;
            $parentReferenceId = null;
            $constant = $dao->convert_sql_text($entry['constant']);
            $rawTitle = $dao->convert_sql_text($entry['title']);
            $translatedTitle = $dao->convert_sql_text($language->get($entry['title']));

            if ($dialogDao) {
                $parentData = $dialogDao->get_data(null, $entry[$parentTable]);

                if ($parentData[$parentTable . '__id'] !== null) {
                    $parentReferenceId = $parentData[$parentTable . '__id'];
                }
            }

            $findEntrySql = "SELECT {$table}__id AS id
                FROM {$table} 
                WHERE {$identifierCondition}
                AND (
                    ({$constant} <> '' AND {$table}__const LIKE {$constant})
                    OR {$table}__title LIKE {$rawTitle} 
                    OR {$table}__title LIKE {$translatedTitle}
                )
                LIMIT 1;";

            $foundId = (int)$dao->retrieve($findEntrySql)
                ->get_row_value('id');

            $dialogData = [
                $table . '__const = ' . $constant,
                $table . '__title = ' . $rawTitle,
                $table . '__sort = ' . $dao->convert_sql_int($entry['sort']),
                $table . '__status = ' . $dao->convert_sql_int(C__RECORD_STATUS__NORMAL)
            ];

            if ($identifier !== null) {
                $dialogData[] = 'isys_dialog_plus_custom__identifier = ' . $dao->convert_sql_text($identifier);
            }

            if ($parentReferenceId !== null) {
                $dialogData[] = $table . '__' . $parentTable . '__id = ' . $dao->convert_sql_id($parentReferenceId);
            }

            $data = implode(', ', $dialogData);

            if ($foundId) {
                $upsertSql = "UPDATE {$table} SET {$data} WHERE {$table}__id = {$foundId};";
            } else {
                $upsertSql = "INSERT INTO {$table} SET {$data};";
            }

            $dao->update($upsertSql) && $dao->apply_update();
        }
    }
}
