<?php

namespace idoit\Module\Maintenance\Model;

use DateTime;
use isys_application;
use isys_cmdb_dao;
use isys_cmdb_dao_category_g_image as imageDao;
use isys_component_database;
use isys_maintenance_dao as maintenanceDao;

/**
 * i-doit
 *
 * Maintenance model for "overview" page.
 *
 * @package     idoit\Module\Maintenance\Model
 * @package     maintenance
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @version     1.0
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       Maintenance 1.1
 */
class Overview extends isys_cmdb_dao
{
    /**
     * @var \isys_locale
     */
    private $locale;

    /**
     * @var \isys_component_template_language_manager
     */
    private $lang;

    /**
     * @var maintenanceDao
     */
    private $maintenanceDao;

    /**
     * @var imageDao
     */
    private $imageDao;

    private $maintenances = [];

    private $objects = [];

    /**
     * Overview constructor.
     *
     * @param isys_component_database $p_db
     */
    public function __construct(isys_component_database $p_db)
    {
        global $g_comp_template_language_manager;

        parent::__construct($p_db);

        $this->locale         = isys_application::instance()->container->get('locales');
        $this->lang           = $g_comp_template_language_manager;
        $this->maintenanceDao = maintenanceDao::instance($this->m_db);
        $this->imageDao       = imageDao::instance($this->m_db);
    }

    /**
     * @param DateTime $from
     * @param DateTime $to
     * @param integer  $typeId
     *
     * @return array
     * @throws \Exception
     * @throws \isys_exception_database
     */
    public function getDataByTimespan(DateTime $from, DateTime $to, $typeId = null)
    {
        $maintenances = $this->maintenanceDao->getMaintenancesBetween($from, $to, $typeId);

        foreach ($maintenances as $maintenance)
        {
            if (!isset($this->maintenances[$maintenance['isys_maintenance__id']]))
            {
                $tupel                            = $maintenance;
                $tupel['isys_contact_tag__title'] = $this->lang->get($tupel['isys_contact_tag__title']);
                $tupel['objects']                 = [];
                $tupel['contacts']                = [];

                unset(
                    $tupel['isys_maintenance__interval'],
                    $tupel['isys_maintenance__isys_maintenance_mailtemplate__id'],
                    $tupel['isys_maintenance__mail_dispatched'],
                    $tupel['isys_maintenance__date_from'],
                    $tupel['isys_maintenance__date_to'],
                    $tupel['isys_maintenance__status']
                );

                $l_contact_res = Maintenance::instance($this->m_db)->getMaintenanceContacts($tupel['isys_maintenance__id']);

                if (count($l_contact_res)) {
                    while ($l_contact_row = $l_contact_res->get_row()) {
                        $tupel['contacts'][] = [
                            'isys_obj__id'         => $l_contact_row['isys_obj__id'],
                            'isys_obj__title'      => $l_contact_row['isys_obj__title'],
                            'isys_obj_type__title' => $this->lang->get($l_contact_row['isys_obj_type__title']),
                            'isys_obj_type__icon'  => $l_contact_row['isys_obj_type__icon']
                        ];
                    }
                }

                $this->maintenances[$tupel['isys_maintenance__id']] = $tupel;

                // Get the objects, connected with this maintenance.
                $objectResult = Maintenance::instance($this->m_db)->getMaintenanceObjects($tupel['isys_maintenance__id']);

                while ($objectRow = $objectResult->get_row()) {
                    $this->maintenances[$maintenance['isys_maintenance__id']]['objects'][] = $objectRow['isys_obj__id'];

                    if (!isset($this->objects[$objectRow['isys_obj__id']])) {
                        $objectRow['maintenances']         = [];
                        $objectRow['roles']                = [];
                        $objectRow['isys_obj_type__title'] = $this->lang->get($objectRow['isys_obj_type__title']);
                        $objectRow['image']                = $this->getObjectImage($objectRow['isys_obj__id']);
                        $objectRow['isys_obj_type__icon']  = $this->getObjectTypeIcon($objectRow['isys_obj_type__icon']);

                        $l_names       = [];
                        $l_contact_res = Contact::instance($this->m_db)->getByObjectAndTag($objectRow['isys_obj__id'], $maintenance['isys_contact_tag__id']);

                        if (count($l_contact_res)) {
                            while ($l_contact_row = $l_contact_res->get_row()) {
                                $l_names[] = $this->lang->get($this->maintenanceDao->get_obj_type_name_by_obj_id($l_contact_row['isys_obj__id'])) .
                                    ' > ' . $l_contact_row['isys_obj__title'] .
                                    (empty($l_contact_row['isys_contact_tag__title']) ? '' : ' (' . $this->lang->get($l_contact_row['isys_contact_tag__title']) . ')');
                            }
                        }

                        $l_names = array_filter($l_names);

                        if (count($l_names)) {
                            $objectRow['roles'][$maintenance['isys_contact_tag__id']] = $l_names;
                        }

                        $this->objects[$objectRow['isys_obj__id']] = $objectRow;
                    }
                }
            }

            $this->maintenances[$maintenance['isys_maintenance__id']]['dates'][] = [
                'from'          => strtotime($maintenance['isys_maintenance__date_from']),
                'fromFormatted' => $this->locale->fmt_datetime($maintenance['isys_maintenance__date_from']),
                'to'            => strtotime($maintenance['isys_maintenance__date_to']),
                'toFormatted'   => $this->locale->fmt_datetime($maintenance['isys_maintenance__date_to']),
            ];

            // And finally we add a relation between objects and maintenances.
            //$this->objects[$maintenance['isys_obj__id']]['maintenances'][] = $maintenance['isys_maintenance__id'];
        }

        return [
            'objects'      => $this->objects,
            'maintenances' => $this->maintenances
        ];
    }

    /**
     * Retrieve the object image of the given object.
     *
     * @param int $objectId
     *
     * @return string
     */
    private function getObjectImage($objectId)
    {
        global $g_config, $g_dirs;

        $l_object['image'] = $this->imageDao->get_image_name_by_object_id($objectId);

        if ($l_object['image'])
        {
            return $g_config['www_dir'] . 'upload/images/' . $l_object['image'];
        }
        else
        {
            return $g_dirs['images'] . 'objecttypes/' . $this->maintenanceDao->get_type_by_object_id($objectId)
                    ->get_row_value('isys_obj_type__obj_img_name');
        }
    }

    /**
     * Retrieve the object image of the given object.
     *
     * @param string $objectTypeIcon
     *
     * @return string
     */
    private function getObjectTypeIcon($objectTypeIcon)
    {
        global $g_dirs;

        if (empty($objectTypeIcon))
        {
            return $g_dirs['images'] . 'empty.gif';
        }
        else if (strpos($objectTypeIcon, '/') === false)
        {
            return $g_dirs['images'] . 'tree/' . $objectTypeIcon;
        }

        return $objectTypeIcon;
    }
}
