<?php

use idoit\AddOn\AuthableInterface;
use idoit\Component\ContainerFacade;
use idoit\Component\Interval\Config;
use idoit\Component\Interval\Interval;
use idoit\Module\Maintenance\Model\Mailtemplate;
use idoit\Module\Maintenance\Model\Maintenance;

/**
 * i-doit
 *
 * Maintenance module class.
 *
 * @package     modules
 * @subpackage  maintenance
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @version     1.0.1
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       i-doit 1.5.0
 */
class isys_module_maintenance extends isys_module implements AuthableInterface
{
    // Define, if this module shall be displayed in the named menus.
    const DISPLAY_IN_MAIN_MENU = true;
    const DISPLAY_IN_SYSTEM_MENU = false;

    /**
     * This number will be used to render "x" clients in the planning (and overview) list.
     * @var  integer
     */
    const NUMBER_OF_CLIENTS_TO_RENDER = 6;

    /**
     * Variable which holds the maintenance DAO class.
     *
     * @var  isys_maintenance_dao
     */
    protected $m_dao;

    /**
     * Variable which holds the database component.
     *
     * @var  isys_component_database
     */
    private $databaseComponent;

    /**
     * Variable which the module request class.
     *
     * @var  isys_module_request
     */
    private $moduleRequest;

    /**
     * Variable which holds the template component.
     *
     * @var  isys_component_template
     */
    private $templateComponent;

    /**
     * Attach live status of object to contenttop header.
     *
     * @param array $p_catdata
     *
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public static function process_content_top($p_catdata)
    {
        try {
            $database = isys_application::instance()->container->get(
                'database',
                ContainerFacade::EXCEPTION_ON_INVALID_REFERENCE
            );

            if (isys_maintenance_dao::instance($database)->is_in_maintenance($p_catdata['isys_obj__id'])) {
                global $index_includes;

                if (!is_array($index_includes['contenttopobjectdetail'])) {
                    $index_includes['contenttopobjectdetail'] = (array)$index_includes['contenttopobjectdetail'];
                }

                $index_includes['contenttopobjectdetail'][] = __DIR__ . '/templates/contenttop/main_objectdetail.tpl';
            }
        } catch (Exception $e) {
            isys_notify::error($e->getMessage(), ['life' => 5]);
        }
    }

    /**
     * Initializes the module.
     *
     * @param isys_module_request &$p_req
     *
     * @return  isys_module_maintenance
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function init(isys_module_request $p_req)
    {
        $this->moduleRequest = $p_req;
        $this->databaseComponent = isys_application::instance()->container->get('database');
        $this->templateComponent = isys_application::instance()->container->get('template');
        $this->m_dao = new isys_maintenance_dao($this->databaseComponent);

        return $this;
    }

    /**
     * Method for building the breadcrumb navigation.
     *
     * @param array &$p_gets
     *
     * @return  array
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function breadcrumb_get(&$p_gets)
    {
        $l_return = [];
        $l_gets = $this->moduleRequest->get_gets();

        switch ($l_gets[C__GET__SETTINGS_PAGE]) {
            default:
            case C__MAINTENANCE__OVERVIEW:
                $l_return[] = [
                    $this->language->get('LC__MAINTENANCE__TREE__OVERVIEW_REPORT') => [
                        C__GET__MODULE_ID => C__MODULE__MAINTENANCE,
                        C__GET__TREE_NODE => $l_gets[C__GET__TREE_NODE],
                        C__GET__SETTINGS_PAGE => $l_gets[C__GET__SETTINGS_PAGE]
                    ]
                ];
                break;

            case C__MAINTENANCE__PLANNING:
                $l_return[] = [
                    $this->language->get('LC__MAINTENANCE__TREE__OPEN_MAINTENANCES') => [
                        C__GET__MODULE_ID => C__MODULE__MAINTENANCE,
                        C__GET__TREE_NODE => $l_gets[C__GET__TREE_NODE],
                        C__GET__SETTINGS_PAGE => $l_gets[C__GET__SETTINGS_PAGE]
                    ]
                ];
                break;

            case C__MAINTENANCE__PLANNING_ARCHIVE:
                $l_return[] = [
                    $this->language->get('LC__MAINTENANCE__TREE__FINISHED_MAINTENANCES') => [
                        C__GET__MODULE_ID => C__MODULE__MAINTENANCE,
                        C__GET__TREE_NODE => $l_gets[C__GET__TREE_NODE],
                        C__GET__SETTINGS_PAGE => $l_gets[C__GET__SETTINGS_PAGE]
                    ]
                ];
                break;

            case C__MAINTENANCE__MAILTEMPLATE:
                $l_return[] = [
                    $this->language->get('LC__MAINTENANCE__TREE__MAILTEMPLATES') => [
                        C__GET__MODULE_ID => C__MODULE__MAINTENANCE,
                        C__GET__TREE_NODE => $l_gets[C__GET__TREE_NODE],
                        C__GET__SETTINGS_PAGE => $l_gets[C__GET__SETTINGS_PAGE]
                    ]
                ];
        }

        return $l_return;
    }

    /**
     * This method builds the tree for the menu.
     *
     * @param isys_component_tree $p_tree
     * @param boolean $p_system_module
     * @param integer $p_parent
     *
     * @author  Leonard Fischer <lfischer@i-doit.com>
     * @see     isys_module_cmdb->build_tree();
     */
    public function build_tree(isys_component_tree $p_tree, $p_system_module = true, $p_parent = null)
    {
        global $g_dirs;

        if ($p_system_module === false) {
            // Don't sort the tree alphabetically!
            $p_tree->set_tree_sort(false);

            $l_auth = self::getAuth();
            $l_root = $p_tree->add_node(
                C__MODULE__MAINTENANCE . 0,
                $p_parent,
                $this->language->get('LC__MODULE__MAINTENANCE'),
                null,
                null,
                self::getWwwPath() . 'assets/add-on-icon.svg'
            );

            $p_tree->add_node(
                C__MODULE__MAINTENANCE . 2,
                $l_root,
                $this->language->get('LC__MAINTENANCE__TREE__OPEN_MAINTENANCES'),
                isys_helper_link::create_url([
                    C__GET__MODULE_ID => C__MODULE__MAINTENANCE,
                    C__GET__TREE_NODE => C__MODULE__MAINTENANCE . 2,
                    C__GET__SETTINGS_PAGE => C__MAINTENANCE__PLANNING,
                ]),
                '',
                $g_dirs['images'] . 'axialis/basic/calendar-1.svg',
                (int)($_GET[C__GET__TREE_NODE] == C__MODULE__MAINTENANCE . 2),
                '',
                '',
                $l_auth->is_allowed_to(isys_auth::VIEW, 'planning')
            );

            $p_tree->add_node(
                C__MODULE__MAINTENANCE . 3,
                $l_root,
                $this->language->get('LC__MAINTENANCE__TREE__FINISHED_MAINTENANCES'),
                isys_helper_link::create_url([
                    C__GET__MODULE_ID => C__MODULE__MAINTENANCE,
                    C__GET__TREE_NODE => C__MODULE__MAINTENANCE . 3,
                    C__GET__SETTINGS_PAGE => C__MAINTENANCE__PLANNING_ARCHIVE,
                ]),
                '',
                $g_dirs['images'] . 'axialis/basic/calendar-1.svg',
                (int)($_GET[C__GET__TREE_NODE] == C__MODULE__MAINTENANCE . 3),
                '',
                '',
                $l_auth->is_allowed_to(isys_auth::VIEW, 'planning_archive')
            );

            $p_tree->add_node(
                C__MODULE__MAINTENANCE . 6,
                $l_root,
                $this->language->get('LC__MAINTENANCE__TREE__ALL_MAINTENANCES'),
                isys_helper_link::create_url([
                    C__GET__MODULE_ID => C__MODULE__MAINTENANCE,
                    C__GET__TREE_NODE => C__MODULE__MAINTENANCE . 6,
                    C__GET__SETTINGS_PAGE => C__MAINTENANCE__PLANNING_ALL,
                ]),
                '',
                $g_dirs['images'] . 'axialis/basic/calendar-1.svg',
                (int)($_GET[C__GET__TREE_NODE] == C__MODULE__MAINTENANCE . 6),
                '',
                '',
                $l_auth->is_allowed_to(isys_auth::VIEW, 'planning_all')
            );

            $p_tree->add_node(
                C__MODULE__MAINTENANCE . 1,
                $l_root,
                $this->language->get('LC__MAINTENANCE__TREE__OVERVIEW_REPORT'),
                isys_helper_link::create_url([
                    C__GET__MODULE_ID => C__MODULE__MAINTENANCE,
                    C__GET__TREE_NODE => C__MODULE__MAINTENANCE . 1,
                    C__GET__SETTINGS_PAGE => C__MAINTENANCE__OVERVIEW,
                ]),
                '',
                $g_dirs['images'] . 'axialis/database/data.svg',
                (int)(!$_GET[C__GET__TREE_NODE] || $_GET[C__GET__TREE_NODE] == C__MODULE__MAINTENANCE . 1),
                '',
                '',
                $l_auth->is_allowed_to(isys_auth::VIEW, 'overview')
            );

            $p_tree->add_node(
                C__MODULE__MAINTENANCE . 5,
                $l_root,
                $this->language->get('LC__MAINTENANCE__TREE__ANNUAL_REPORT_REPORT'),
                isys_helper_link::create_url([
                    C__GET__MODULE_ID => C__MODULE__REPORT,
                    C__GET__TREE_NODE => C__MODULE__REPORT . '5',
                    C__GET__REPORT_PAGE => C__REPORT_PAGE__VIEWS,
                    C__GET__REPORT_REPORT_ID => 'isys_maintenance_reportview_maintenance_export'
                ]),
                '',
                $g_dirs['images'] . 'axialis/database/data.svg',
                (int)($_GET[C__GET__TREE_NODE] == C__MODULE__MAINTENANCE . 5),
                '',
                '',
                isys_auth_report::instance()->has("views")
            );

            $p_tree->add_node(
                C__MODULE__MAINTENANCE . 4,
                $l_root,
                $this->language->get('LC__MAINTENANCE__TREE__MAILTEMPLATES'),
                isys_helper_link::create_url([
                    C__GET__MODULE_ID => C__MODULE__MAINTENANCE,
                    C__GET__TREE_NODE => C__MODULE__MAINTENANCE . 4,
                    C__GET__SETTINGS_PAGE => C__MAINTENANCE__MAILTEMPLATE,
                ]),
                '',
                $g_dirs['images'] . 'axialis/web-email/mail-back-all.svg',
                (int)($_GET[C__GET__TREE_NODE] == C__MODULE__MAINTENANCE . 4),
                '',
                '',
                $l_auth->is_allowed_to(isys_auth::VIEW, 'mailtemplate')
            );
        }
    }

    /**
     * Start method.
     *
     * @return  isys_module_maintenance
     * @throws  isys_exception_general
     * @throws  isys_exception_licence
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function start()
    {
        // Build the module tree, but only if we are not in the system-module.
        if ($_GET[C__GET__MODULE_ID] != C__MODULE__SYSTEM) {
            $l_tree = isys_module_request::get_instance()->get_menutree();

            $this->build_tree($l_tree, false, -1);
            $this->templateComponent->assign('menu_tree', $l_tree->process($_GET[C__GET__TREE_NODE]));

            switch ($_GET[C__GET__SETTINGS_PAGE]) {
                default:
                case C__MAINTENANCE__OVERVIEW:
                    $this->process_overview();
                    break;

                case C__MAINTENANCE__PLANNING:
                    $this->process_planning();
                    break;

                case C__MAINTENANCE__PLANNING_ARCHIVE:
                    $this->process_planning(C__MAINTENANCE__PLANNING_ARCHIVE);
                    break;

                case C__MAINTENANCE__PLANNING_ALL:
                    $this->process_planning(C__MAINTENANCE__PLANNING_ALL);
                    break;

                case C__MAINTENANCE__MAILTEMPLATE:
                    $this->process_mailtemplate();
                    break;
            }
        }

        return $this;
    }

    /**
     * Modify row method for the planning list.
     *
     * @param array  &$p_row
     *
     * @throws  Exception
     * @throws  isys_exception_database
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function process_planning__list_modify_row(&$p_row)
    {
        global $g_dirs;

        $locales = isys_application::instance()->container->get('locales');

        $p_row['isys_maintenance__date_from'] = $locales->fmt_datetime($p_row['isys_maintenance__date_from']);
        $p_row['isys_maintenance__date_to'] = $locales->fmt_datetime($p_row['isys_maintenance__date_to']);

        $yes = '<img src="' . $g_dirs['images'] . 'axialis/basic/symbol-ok.svg" class="vam mr5" />' . $this->language->get('LC__UNIVERSAL__YES');
        $no = '<img src="' . $g_dirs['images'] . 'axialis/basic/symbol-cancel.svg" class="vam mr5" />' . $this->language->get('LC__UNIVERSAL__NO');

        $p_row['isys_maintenance__finished'] = '<span class="text-red" data-finished="0">' . $no . '</span>';
        $p_row['isys_maintenance__mail_dispatched'] = '<span class="text-red" data-mail-dispatched="0">' . $no . '</span>';
        $p_row['isys_maintenance__interval'] = '<span class="text-red">' . $no . '</span>';

        if ($p_row['isys_maintenance__finished'] !== null) {
            $p_row['isys_maintenance__finished'] = '<span class="text-green" data-finished="1">' . $yes . '</span>';
        }

        if ($p_row['isys_maintenance__mail_dispatched'] !== null) {
            $p_row['isys_maintenance__mail_dispatched'] = '<span class="text-green" data-mail-dispatched="1">' . $yes . '</span>';
        }

        if (!empty($p_row['isys_maintenance__interval'])) {
            $p_row['isys_maintenance__interval'] = '<span class="text-green">' . $yes . '</span>';
        }
    }

    /**
     * Process method for the "overview" page.
     */
    private function process_overview()
    {
        $l_rules = [
            'C__MAINTENANCE__OVERVIEW__FILTER__DATE_FROM' => [
                'p_strPopupType' => 'calendar',
                'p_strClass' => 'input-small',
                'p_bInfoIconSpacer' => 0,
                'p_strValue' => date('Y-m-d'),
            ],
            'C__MAINTENANCE__OVERVIEW__FILTER__DATE_TO' => [
                'p_strPopupType' => 'calendar',
                'p_strClass' => 'input-small',
                'p_bInfoIconSpacer' => 0,
            ]
        ];

        $l_day_of_week = date('N') - 1;
        $l_this_week_time = time() - ($l_day_of_week * isys_convert::DAY);

        $l_this_week = mktime(
            0,
            0,
            0,
            date('m', $l_this_week_time),
            date('d', $l_this_week_time),
            date('Y', $l_this_week_time)
        );

        $this->templateComponent->activate_editmode()
            ->assign('ajax_url', isys_helper_link::create_url([C__GET__AJAX => 1, C__GET__AJAX_CALL => 'maintenance']))
            ->assign('this_week', $l_this_week)
            ->assign('this_month', mktime(0, 0, 0, date('m'), 1, date('Y')))
            ->assign('next_week', $l_this_week + isys_convert::WEEK)
            ->assign('next_month', mktime(0, 0, 0, (date('m') + 1), 1, date('Y')))
            ->assign('next_next_month', mktime(0, 0, 0, (date('m') + 2), 1, date('Y')))
            ->assign('planning_url', isys_helper_link::create_url([
                C__GET__MODULE_ID => C__MODULE__MAINTENANCE,
                C__GET__SETTINGS_PAGE => C__MAINTENANCE__PLANNING
            ]))
            ->include_template('contentbottomcontent', __DIR__ . '/templates/overview.tpl')
            ->smarty_tom_add_rules('tom.content.bottom.content', $l_rules);
    }

    /**
     * Process method for the "planning" page.
     *
     * @param string $p_type
     */
    private function process_planning($p_type = C__MAINTENANCE__PLANNING)
    {
        $l_gets = $this->moduleRequest->get_gets();
        $l_posts = $this->moduleRequest->get_posts();

        $l_id = isys_glob_which_isset($l_gets[C__GET__ID], $l_posts[C__GET__ID]);
        $l_navmode = isys_glob_which_isset($l_gets[C__GET__NAVMODE], $l_posts[C__GET__NAVMODE]);

        // This will happen, if a user uses the checkboxes and the "edit" buttno.
        if (is_array($l_id)) {
            $l_id = $l_id[0];
        }

        if (!$l_navmode && $l_id > 0) {
            $l_navmode = C__NAVMODE__EDIT;
        }

        switch ($l_navmode) {
            default:
                $this->process_planning__list($p_type);
                break;

            case C__NAVMODE__EDIT:
                $this->process_planning__edit($p_type, $l_id);
                break;

            case C__NAVMODE__NEW:
                $this->process_planning__edit($p_type);
                break;

            case C__NAVMODE__SAVE:
                $this->process_planning__edit($p_type, $this->process_planning__save($p_type, $l_posts));
                break;

            case C__NAVMODE__DELETE:
                $this->process_planning__delete(
                    $p_type,
                    isys_glob_which_isset($l_gets[C__GET__ID], $l_posts[C__GET__ID])
                );
                $this->process_planning__list($p_type);
                break;
        }

        $this->templateComponent->smarty_tom_add_rule('tom.content.navbar.cRecStatus.p_bInvisible=1');
    }

    /**
     * Processes the planning list.
     *
     * @param string $p_type
     *
     * @throws  Exception
     * @throws  isys_exception_general
     */
    private function process_planning__list($p_type = C__MAINTENANCE__PLANNING)
    {
        $auth = self::getAuth();
        $l_gets = $this->moduleRequest->get_gets();

        if ($p_type === C__MAINTENANCE__PLANNING) {
            $l_rights = [
                isys_auth::EDIT => $auth->is_allowed_to(isys_auth::EDIT, 'planning'),
                isys_auth::DELETE => $auth->is_allowed_to(isys_auth::DELETE, 'planning')
            ];
        } elseif ($p_type === C__MAINTENANCE__PLANNING_ARCHIVE) {
            $l_rights = [
                isys_auth::EDIT => $auth->is_allowed_to(isys_auth::EDIT, 'planning_archive'),
                isys_auth::DELETE => $auth->is_allowed_to(isys_auth::DELETE, 'planning_archive')
            ];
        } else {
            $l_rights = [
                isys_auth::EDIT => $auth->is_allowed_to(isys_auth::EDIT, 'planning_all'),
                isys_auth::DELETE => $auth->is_allowed_to(isys_auth::DELETE, 'planning_all')
            ];
        }

        $l_url_params = [
            C__GET__MODULE_ID => C__MODULE__MAINTENANCE,
            C__GET__TREE_NODE => $l_gets[C__GET__TREE_NODE],
            C__GET__SETTINGS_PAGE => $p_type,
            C__GET__ID => '[{isys_maintenance__id}]'
        ];

        $l_list_headers = [
            'isys_maintenance__id' => 'ID',
            'isys_maintenance_type__title' => 'LC__MAINTENANCE__PLANNING__TYPE',
            'clients' => 'LC__MAINTENANCE__PLANNING__OBJECT_SELECTION',
            'isys_maintenance__date_from' => 'LC__MAINTENANCE__PLANNING__DATE_FROM',
            'isys_maintenance__date_to' => 'LC__MAINTENANCE__PLANNING__DATE_TO',
            'isys_maintenance__finished' => 'LC__MAINTENANCE__PLANNING__FINISHED',
            'isys_maintenance__mail_dispatched' => 'LC__MAINTENANCE__PLANNING__MAIL_DISPATCHED',
            'isys_maintenance__interval' => 'LC__MAINTENANCE__PLANNING__INTERVAL'
        ];

        if ($p_type !== C__MAINTENANCE__PLANNING_ALL) {
            unset($l_list_headers['isys_maintenance__finished']);
        }

        $listData = [];
        $l_result = $this->m_dao->get_data();
        $l_result_count = count($l_result);

        $maintenanceModel = Maintenance::instance($this->databaseComponent);

        while ($row = $l_result->get_row()) {
            if ($p_type === C__MAINTENANCE__PLANNING) {
                if (!empty($row['isys_maintenance__finished'])) {
                    continue;
                }
            } elseif ($p_type === C__MAINTENANCE__PLANNING_ARCHIVE) {
                if ($row['isys_maintenance__finished'] === null) {
                    continue;
                }
            }

            $clients = [];
            $l_object_res = $maintenanceModel->getMaintenanceObjects($row['isys_maintenance__id']);

            if (count($l_object_res)) {
                $l_obj_count = count($l_object_res);
                $l_cnt = 1;

                while ($l_row = $l_object_res->get_row()) {
                    if ($l_cnt > self::NUMBER_OF_CLIENTS_TO_RENDER) {
                        $clients[] = '... (' . $l_obj_count . ')';
                        break;
                    }

                    $clients[] = $this->language->get($l_row['isys_obj_type__title']) . ' &raquo; ' . $l_row['isys_obj__title'] . ' {' . $l_row['isys_obj__id'] . '}';

                    $l_cnt++;
                }
            }

            $listData[] = [
                'isys_maintenance__id' => (int)$row['isys_maintenance__id'],
                'isys_maintenance_type__title' => $row['isys_maintenance_type__title'],
                'clients' => implode(', ', $clients),
                'isys_maintenance__date_from' => $row['isys_maintenance__date_from'],
                'isys_maintenance__date_to' => $row['isys_maintenance__date_to'],
                'isys_maintenance_mailtemplate__title' => $row['isys_maintenance_mailtemplate__title'],
                'isys_maintenance__finished' => $row['isys_maintenance__finished'],
                'isys_maintenance__mail_dispatched' => $row['isys_maintenance__mail_dispatched'],
                'isys_maintenance__interval' => $row['isys_maintenance__interval']
            ];
        }

        /** @var  isys_component_list $l_list */
        $l_list = new isys_component_list($listData);

        $l_list->set_row_modifier($this, 'process_planning__list_modify_row')
            ->config($l_list_headers, isys_helper_link::create_url($l_url_params), '[{isys_maintenance__id}]');

        if ($l_list->createTempTable()) {
            $this->templateComponent->assign('list', $l_list->getTempTableHtml());
        }

        isys_component_template_navbar::getInstance()
            ->set_active($l_rights[isys_auth::EDIT], C__NAVBAR_BUTTON__NEW)
            ->set_active(($l_rights[isys_auth::EDIT] && $l_result_count > 0), C__NAVBAR_BUTTON__EDIT)
            ->set_active(($l_rights[isys_auth::DELETE] && $l_result_count > 0), C__NAVBAR_BUTTON__DELETE)
            ->set_visible(true, C__NAVBAR_BUTTON__NEW)
            ->set_visible(true, C__NAVBAR_BUTTON__EDIT)
            ->set_visible(true, C__NAVBAR_BUTTON__DELETE);

        if ($l_rights[isys_auth::EDIT] && $p_type === C__MAINTENANCE__PLANNING) {
            isys_component_template_navbar::getInstance()->append_button(
                'LC__MAINTENANCE__PLANNING__FINISH_MAINTENANCE',
                'maintenance-finish',
                [
                    'tooltip' => $this->language->get('LC__MAINTENANCE__POPUP__FINISH_MAINTENANCE'),
                    'icon' => 'axialis/basic/symbol-ok.svg',
                    'icon_inactive' => 'axialis/basic/symbol-ok.svg',
                    'js_onclick' => isys_factory::get_instance('isys_popup_maintenance_finish')
                        ->process_overlay('', 800, 500),
                    'accesskey' => 'f',
                    'navmode' => 'maintenance_finish'
                ]
            );
        }

        if ($auth->is_allowed_to(isys_auth::EXECUTE, 'send_mails')) {
            isys_component_template_navbar::getInstance()
                ->append_button('LC__MAINTENANCE__SEND_MAIL', 'maintenance-send-mail', [
                        'tooltip'       => $this->language->get('LC__MAINTENANCE__SEND_MAIL'),
                        'icon'          => 'axialis/web-email/mail-back-all.svg',
                        'icon_inactive' => 'axialis/web-email/mail-back-all.svg',
                        'js_onclick'    => ';',
                        'accesskey'     => 'm',
                        'navmode'       => 'maintenance_send_mail'
                    ]);
        }

        $this->templateComponent
            ->assign('ajax_url', isys_helper_link::create_url([C__GET__AJAX => 1, C__GET__AJAX_CALL => 'maintenance']))
            ->assign('content_title', $this->language->get('LC__MAINTENANCE__PLANNING'))
            ->include_template('contentbottomcontent', __DIR__ . '/templates/planning_list.tpl');
    }

    /**
     * Method for loading the "planning" form.
     *
     * @param string $p_type
     * @param integer $p_id
     *
     * @throws  Exception
     * @throws  isys_exception_database
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    private function process_planning__edit($p_type, $p_id = null)
    {
        $auth = self::getAuth();

        if ($p_type === C__MAINTENANCE__PLANNING) {
            $l_is_allowed_to_edit = $auth->is_allowed_to(isys_auth::EDIT, 'planning');
        } else {
            $l_is_allowed_to_edit = $auth->is_allowed_to(isys_auth::EDIT, 'planning_archive');
        }

        $intervalExpired = false;
        $l_contacts = $l_objects = $l_mailtemplates = [];
        $lastInterval = $nextInterval = isys_tenantsettings::get('gui.empty_value', '-');
        $locales = isys_application::instance()->container->get('locales');

        isys_component_template_navbar::getInstance()
            ->set_active($l_is_allowed_to_edit, C__NAVBAR_BUTTON__SAVE)
            ->set_active(true, C__NAVBAR_BUTTON__CANCEL)
            ->set_save_mode('formsubmit');

        if ($p_id !== null) {
            $l_planning_data = $this->m_dao->get_data($p_id)->get_row();

            $l_object_res = Maintenance::instance($this->databaseComponent)->getMaintenanceObjects($p_id);
            $l_contact_res = Maintenance::instance($this->databaseComponent)->getMaintenanceContacts($p_id);

            if (count($l_object_res)) {
                while ($l_object_row = $l_object_res->get_row()) {
                    $l_objects[] = (int)$l_object_row['isys_obj__id'];
                }
            }

            if (count($l_contact_res)) {
                while ($l_contact_row = $l_contact_res->get_row()) {
                    $l_contacts[] = (int)$l_contact_row['isys_obj__id'];
                }
            }
        } else {
            // Default selection for the maintenance type.
            $l_planning_data = [
                'isys_maintenance__finished' => null,
                'isys_maintenance__mail_dispatched' => null,
                'isys_maintenance__isys_maintenance_type__id' => $this->m_dao
                    ->retrieve('SELECT isys_maintenance_type__id FROM isys_maintenance_type WHERE isys_maintenance_type__const = "C__MAINTENANCE__TYPE__CLIENT_MAINTENANCE";')
                    ->get_row_value('isys_maintenance_type__id')
            ];
        }

        $l_mailtemplate_res = Mailtemplate::instance($this->databaseComponent)
            ->getMailtemplates();

        if (count($l_mailtemplate_res)) {
            while ($l_row = $l_mailtemplate_res->get_row()) {
                $l_mailtemplates[$l_row['isys_maintenance_mailtemplate__id']] = $l_row['isys_maintenance_mailtemplate__title'];
            }
        }

        $intervalConfig = $l_planning_data['isys_maintenance__interval'];

        if (isys_format_json::is_json_array($intervalConfig)) {
            $intervalConfig = Config::byJSON($intervalConfig);
            $interval = new Interval($intervalConfig, new DateTime($l_planning_data['isys_maintenance__date_from']));

            $intervalExpired = $interval->isIntervalExpired();

            if ($interval->hasLastIntervalDate()) {
                $lastInterval = $locales->fmt_date($interval->getLastIntervalDate()->format('Y-m-d'));
            }

            if ($interval->hasNextIntervalDate()) {
                if ($intervalConfig->getInitialDate()->getTimestamp() >= time()) {
                    $nextInterval = $locales->fmt_date($intervalConfig->getInitialDate()->format('Y-m-d'));
                } else {
                    $nextInterval = $locales->fmt_date($interval->getNextIntervalDate()->format('Y-m-d'));
                }
            }
        }

        $l_rules = [
            'C__MAINTENANCE__PLANNING__ID' => [
                'p_strValue' => (int)$p_id,
                'p_bInvisible' => true,
                'p_bInfoIconSpacer' => 0
            ],
            'C__MAINTENANCE__PLANNING__OBJECT_SELECTION' => [
                isys_popup_browser_object_ng::C__MULTISELECTION => true,
                'p_strValue' => empty($l_objects) ? '' : isys_format_json::encode($l_objects)
            ],
            'C__MAINTENANCE__PLANNING__TYPE' => [
                'p_strPopupType' => 'dialog_plus',
                'p_strTable' => 'isys_maintenance_type',
                'p_strSelectedID' => $l_planning_data['isys_maintenance__isys_maintenance_type__id'],
                'p_strClass' => 'input-small'
            ],
            'C__MAINTENANCE__PLANNING__DATE_FROM' => [
                'p_strPopupType' => 'calendar',
                'p_strClass' => 'input-small',
                'p_strStyle' => 'width:50%',
                'p_bTime' => true,
                'p_strValue' => $l_planning_data['isys_maintenance__date_from']
            ],
            'C__MAINTENANCE__PLANNING__DATE_TO' => [
                'p_strPopupType' => 'calendar',
                'p_strClass' => 'input-small',
                'p_strStyle' => 'width:50%',
                'p_bTime' => true,
                'p_strValue' => $l_planning_data['isys_maintenance__date_to'],
                'p_bInfoIconSpacer' => 0
            ],
            'C__MAINTENANCE__PLANNING__INTERVAL' => [
                'config' => (is_object($intervalConfig) ? $intervalConfig->toArray() : '')
            ],
            'C__MAINTENANCE__PLANNING__COMMENT' => [
                'p_strValue' => $l_planning_data['isys_maintenance__comment']
            ],
            'C__MAINTENANCE__PLANNING__CONTACT_ROLES' => [
                'p_strTable' => 'isys_contact_tag',
                'p_strSelectedID' => $l_planning_data['isys_maintenance__isys_contact_tag__id'],
                'p_strClass' => 'input-small'
            ],
            'C__MAINTENANCE__PLANNING__CONTACTS' => [
                isys_popup_browser_object_ng::C__MULTISELECTION => true,
                isys_popup_browser_object_ng::C__CAT_FILTER => 'C__CATS__PERSON;C__CATS__PERSON_GROUP;C__CATS__ORGANIZATION;C__CATS__PERSON_MASTER;C__CATS__PERSON_GROUP_MASTER;C__CATS__ORGANIZATION_MASTER_DATA',
                'p_strValue' => empty($l_contacts) ? '' : isys_format_json::encode($l_contacts)
            ],
            'C__MAINTENANCE__PLANNING__MAILTEMPLATE' => [
                'p_arData' => $l_mailtemplates,
                'p_strSelectedID' => $l_planning_data['isys_maintenance__isys_maintenance_mailtemplate__id']
            ]
        ];

        if ($l_is_allowed_to_edit) {
            $this->templateComponent->activate_editmode();
        }

        if ($l_is_allowed_to_edit && $p_type === C__MAINTENANCE__PLANNING && $l_planning_data['isys_maintenance__finished'] === null) {
            isys_component_template_navbar::getInstance()->append_button(
                'LC__MAINTENANCE__PLANNING__FINISH_MAINTENANCE',
                'maintenance-finish',
                [
                    'tooltip' => $this->language->get('LC__MAINTENANCE__POPUP__FINISH_MAINTENANCE'),
                    'icon' => 'axialis/basic/symbol-ok.svg',
                    'icon_inactive' => 'axialis/basic/symbol-ok.svg',
                    'js_onclick' => isys_factory::get_instance('isys_popup_maintenance_finish')
                        ->process_overlay('', 800, 500),
                    'accesskey' => 'f',
                    'navmode' => 'maintenance_finish',
                    'active' => ($l_planning_data['isys_maintenance__finished'] === null || !$intervalExpired)
                ]
            );
        }

        if ($l_planning_data['isys_maintenance__mail_dispatched'] === null && $auth->is_allowed_to(
            isys_auth::EXECUTE,
            'send_mails'
        )) {
            isys_component_template_navbar::getInstance()
                ->append_button('LC__MAINTENANCE__SEND_MAIL', 'maintenance-send-mail', [
                        'tooltip'       => $this->language->get('LC__MAINTENANCE__SEND_MAIL'),
                        'icon'          => 'axialis/web-email/mail-back-all.svg',
                        'icon_inactive' => 'axialis/web-email/mail-back-all.svg',
                        'js_onclick'    => ';',
                        'accesskey'     => 'm',
                        'navmode'       => 'maintenance_send_mail',
                        'active'        => $l_planning_data['isys_maintenance__id'] > 0
                    ]);
        }

        $locale = isys_application::instance()->container->get('locales');

        $this->templateComponent
            ->assign('ajax_url', isys_helper_link::create_url([C__GET__AJAX => 1, C__GET__AJAX_CALL => 'maintenance']))
            ->assign(
                'mail_dispatched',
                ($l_planning_data['isys_maintenance__mail_dispatched'] !== null ? $locale->fmt_datetime($l_planning_data['isys_maintenance__mail_dispatched']) : false)
            )
            ->assign(
                'finished',
                ($l_planning_data['isys_maintenance__finished'] !== null ? $locale->fmt_datetime($l_planning_data['isys_maintenance__finished']) : false)
            )
            ->assign('planning', $l_planning_data)
            ->assign('hasInterval', is_object($intervalConfig))
            ->assign('intervalExpired', $intervalExpired)
            ->assign('lastInterval', $lastInterval)
            ->assign('nextInterval', $nextInterval)
            ->smarty_tom_add_rules('tom.content.bottom.content', $l_rules)
            ->include_template('contentbottomcontent', __DIR__ . '/templates/planning.tpl');
    }

    /**
     * Method for saving the "planning" data.
     *
     * @param string $p_type
     * @param array $p_post
     *
     * @return  integer
     * @throws  isys_exception_database
     */
    private function process_planning__save($p_type, $p_post)
    {
        $auth = self::getAuth();

        if ($p_type === C__MAINTENANCE__PLANNING) {
            $auth->check(isys_auth::EDIT, 'planning');
        } else {
            $auth->check(isys_auth::EDIT, 'planning_archive');
        }

        $l_id = ($p_post['C__MAINTENANCE__PLANNING__ID'] > 0) ? $p_post['C__MAINTENANCE__PLANNING__ID'] : null;

        $l_objects = [];
        $l_contacts = [];
        $from = $p_post['C__MAINTENANCE__PLANNING__DATE_FROM__HIDDEN'];
        $to = $p_post['C__MAINTENANCE__PLANNING__DATE_TO__HIDDEN'];

        if (isys_format_json::is_json_array($p_post['C__MAINTENANCE__PLANNING__OBJECT_SELECTION__HIDDEN'])) {
            $l_objects = isys_format_json::decode($p_post['C__MAINTENANCE__PLANNING__OBJECT_SELECTION__HIDDEN']);
        }

        //without assigned objects saving stops and the exception is thrown
        if (empty($l_objects)) {
            throw new isys_exception_general($this->language->get('LC__MAINTENANCC__EXCEPTION__NO_OBJECTS_SELECTED'));
        }

        if (isys_format_json::is_json_array($p_post['C__MAINTENANCE__PLANNING__CONTACTS__HIDDEN'])) {
            $l_contacts = isys_format_json::decode($p_post['C__MAINTENANCE__PLANNING__CONTACTS__HIDDEN']);
        }

        // We need to go sure the "date from < date to".
        if (!empty($from) && !empty($to) && $from > $to) {
            $l_date_tmp = $from;

            $from = $to;

            $to = $l_date_tmp;
        }

        if (isys_format_json::is_json_array($p_post['C__MAINTENANCE__PLANNING__INTERVAL__HIDDEN'])) {
            // When saving a maintenace planning (with valid date) we need to set the initial date = the "date from".
            if (strtotime($from) !== false) {
                $config = Config::byJSON($p_post['C__MAINTENANCE__PLANNING__INTERVAL__HIDDEN'])
                    ->setInitialDate(new DateTime($from))
                    ->toArray();

                $p_post['C__MAINTENANCE__PLANNING__INTERVAL__HIDDEN'] = isys_format_json::encode($config);
            }
        } else {
            $p_post['C__MAINTENANCE__PLANNING__INTERVAL__HIDDEN'] = null;
        }

        try {
            $data = [
                'isys_maintenance__isys_maintenance_type__id' => $p_post['C__MAINTENANCE__PLANNING__TYPE'],
                'isys_maintenance__date_from' => $from,
                'isys_maintenance__date_to' => $to,
                'isys_maintenance__interval' => $p_post['C__MAINTENANCE__PLANNING__INTERVAL__HIDDEN'],
                'isys_maintenance__comment' => $p_post['C__MAINTENANCE__PLANNING__COMMENT'],
                'isys_maintenance__isys_maintenance_mailtemplate__id' => $p_post['C__MAINTENANCE__PLANNING__MAILTEMPLATE'],
                'isys_maintenance__isys_contact_tag__id' => $p_post['C__MAINTENANCE__PLANNING__CONTACT_ROLES'],
                'isys_maintenance__status' => C__RECORD_STATUS__NORMAL
            ];

            $l_id = $this->m_dao->savePlanning($l_id, $data, $l_objects, $l_contacts);

            isys_notify::success($this->language->get('LC__MAINTENANCE__NOTIFY__SAVE_SUCCESS'));
        } catch (Exception $e) {
            isys_notify::error(
                $this->language->get('LC__MAINTENANCE__NOTIFY__SAVE_FAILURE') . $e->getMessage(),
                ['sticky' => true]
            );
        }

        return $l_id;
    }

    /**
     * Method for deleting one or more plannings.
     *
     * @param string $p_type
     * @param mixed $p_id
     *
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    private function process_planning__delete($p_type, $p_id)
    {
        if ($p_type === C__MAINTENANCE__PLANNING) {
            self::getAuth()->check(isys_auth::DELETE, 'planning');
        } else {
            self::getAuth()->check(isys_auth::DELETE, 'planning_archive');
        }

        if (!is_array($p_id)) {
            $p_id = [$p_id];
        }

        $this->m_dao->delete_planning(array_filter($p_id, 'is_numeric'));
    }

    /**
     * Process method for the "mailtemplate" page.
     *
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    private function process_mailtemplate()
    {
        $l_gets = $this->moduleRequest->get_gets();
        $l_posts = $this->moduleRequest->get_posts();

        $l_id = isys_glob_which_isset($l_gets[C__GET__ID], $l_posts[C__GET__ID]);
        $l_navmode = isys_glob_which_isset($l_gets[C__GET__NAVMODE], $l_posts[C__GET__NAVMODE]);

        // This will happen, if a user uses the checkboxes and the "edit" buttno.
        if (is_array($l_id)) {
            $l_id = $l_id[0];
        }

        if (!$l_navmode && $l_id > 0) {
            $l_navmode = C__NAVMODE__EDIT;
        }

        switch ($l_navmode) {
            default:
                $this->process_mailtemplate__list();
                break;

            case C__NAVMODE__EDIT:
                $this->process_mailtemplate__edit($l_id);
                break;

            case C__NAVMODE__NEW:
                $this->process_mailtemplate__edit();
                break;

            case C__NAVMODE__SAVE:
                $this->process_mailtemplate__edit($this->process_mailtemplate__save($l_posts));
                break;

            case C__NAVMODE__DELETE:
                $this->process_mailtemplate__delete(isys_glob_which_isset($l_gets[C__GET__ID], $l_posts[C__GET__ID]));
                $this->process_mailtemplate__list();
                break;
        }

        $this->templateComponent->smarty_tom_add_rule('tom.content.navbar.cRecStatus.p_bInvisible=1');
    }

    /**
     * Method for processing the mailtemplate list.
     *
     * @throws  Exception
     * @throws  isys_exception_general
     */
    private function process_mailtemplate__list()
    {
        $l_gets = $this->moduleRequest->get_gets();

        $l_rights = [
            isys_auth::EDIT => self::getAuth()->is_allowed_to(isys_auth::EDIT, 'mailtemplate'),
            isys_auth::DELETE => self::getAuth()->is_allowed_to(isys_auth::DELETE, 'mailtemplate')
        ];

        $l_url_params = [
            C__GET__MODULE_ID => C__MODULE__MAINTENANCE,
            C__GET__TREE_NODE => $l_gets[C__GET__TREE_NODE],
            C__GET__SETTINGS_PAGE => C__MAINTENANCE__MAILTEMPLATE,
            C__GET__ID => '[{isys_maintenance_mailtemplate__id}]'
        ];

        $l_list_headers = [
            'isys_maintenance_mailtemplate__id' => 'ID',
            'isys_maintenance_mailtemplate__title' => 'LC__MAINTENANCE__MAILTEMPLATE__TITLE',
        ];

        $l_result = Mailtemplate::instance($this->databaseComponent)
            ->getMailtemplates();
        $l_result_count = count($l_result);

        /** @var  isys_component_list $l_list */
        $l_list = new isys_component_list(null, $l_result);

        $l_list->config(
            $l_list_headers,
            isys_helper_link::create_url($l_url_params),
            '[{isys_maintenance_mailtemplate__id}]'
        );

        if ($l_list->createTempTable()) {
            $this->templateComponent->assign('list', $l_list->getTempTableHtml());
        }

        isys_component_template_navbar::getInstance()
            ->set_active($l_rights[isys_auth::EDIT], C__NAVBAR_BUTTON__NEW)
            ->set_active(($l_rights[isys_auth::EDIT] && $l_result_count > 0), C__NAVBAR_BUTTON__EDIT)
            ->set_active(($l_rights[isys_auth::DELETE] && $l_result_count > 0), C__NAVBAR_BUTTON__DELETE)
            ->set_visible(true, C__NAVBAR_BUTTON__NEW)
            ->set_visible(true, C__NAVBAR_BUTTON__EDIT)
            ->set_visible(true, C__NAVBAR_BUTTON__DELETE);

        $this->templateComponent->assign('content_title', $this->language->get('LC__MAINTENANCE__MAILTEMPLATE'))
            ->include_template('contentbottomcontent', __DIR__ . '/templates/mailtemplate_list.tpl');
    }

    /**
     * Method for loading the "planning" form.
     *
     * @param integer $p_id
     *
     * @throws  Exception
     * @throws  isys_exception_database
     */
    private function process_mailtemplate__edit($p_id = null)
    {
        $l_is_allowed_to_edit = self::getAuth()->is_allowed_to(isys_auth::EDIT, 'mailtemplate');
        $l_empty_value = isys_tenantsettings::get('gui.empty_value', '-');

        isys_component_template_navbar::getInstance()
            ->set_active($l_is_allowed_to_edit, C__NAVBAR_BUTTON__SAVE)
            ->set_active(true, C__NAVBAR_BUTTON__CANCEL)
            ->set_save_mode('formsubmit');

        $l_mailtemplate_data = [];

        if ($p_id !== null) {
            $l_mailtemplate_data = Mailtemplate::instance($this->databaseComponent)
                ->getMailtemplates($p_id)
                ->get_row();
        }

        $l_rules = [
            'C__MAINTENANCE__MAILTEMPLATE__ID' => [
                'p_strValue' => (int)$p_id,
                'p_bInvisible' => true,
                'p_bInfoIconSpacer' => 0
            ],
            'C__MAINTENANCE__MAILTEMPLATE__TITLE' => [
                'p_strValue' => $l_mailtemplate_data['isys_maintenance_mailtemplate__title']
            ],
            'C__MAINTENANCE__MAILTEMPLATE__TEXT' => [
                'p_strValue' => $l_mailtemplate_data['isys_maintenance_mailtemplate__text']
            ]
        ];

        // @see WARTUNG-85 Fetch a random object, similar to the usage in 'isys_module_qrcode'.
        $statusNormal = $this->m_dao->convert_sql_int(C__RECORD_STATUS__NORMAL);
        $sql = "SELECT isys_obj__id AS id
            FROM isys_obj
            INNER JOIN isys_obj_type ON isys_obj_type__id = isys_obj__isys_obj_type__id
            WHERE isys_obj_type__const NOT IN ('C__OBJTYPE__RELATION')
            AND isys_obj__status = {$statusNormal}
            ORDER BY RAND()
            LIMIT 1;";

        $l_variables = isys_helper_link::get_url_variables((int)$this->m_dao->retrieve($sql)->get_row_value('id'));

        $l_variables['%recipients%'] = $this->language->get('LC__UNIVERSAL__MISSES') . ' Maria Mustermann, John Doe';
        $l_variables['%maintenance_from%'] = date('d.m.Y');
        $l_variables['%maintenance_to%'] = date('d.m.Y H:i:s', (time() + isys_convert::WEEK));

        foreach ($l_variables as $l_key => &$l_value) {
            $l_value = (empty($l_value) ? $l_empty_value : $l_value);
        }

        if ($l_is_allowed_to_edit) {
            $this->templateComponent->activate_editmode();
        }

        $this->templateComponent
            ->assign('variables', $l_variables)
            ->include_template('contentbottomcontent', __DIR__ . '/templates/mailtemplate.tpl')
            ->smarty_tom_add_rules('tom.content.bottom.content', $l_rules);
    }

    /**
     * Method for saving the "planning" data.
     *
     * @param array $p_post
     *
     * @return  integer
     * @throws  isys_exception_database
     */
    private function process_mailtemplate__save($p_post)
    {
        $l_id = ($p_post['C__MAINTENANCE__MAILTEMPLATE__ID'] > 0) ? $p_post['C__MAINTENANCE__MAILTEMPLATE__ID'] : null;

        try {
            $l_id = Mailtemplate::instance($this->databaseComponent)->saveMailtemplate(
                $l_id,
                [
                    'isys_maintenance_mailtemplate__title' => $p_post['C__MAINTENANCE__MAILTEMPLATE__TITLE'],
                    'isys_maintenance_mailtemplate__text' => $p_post['C__MAINTENANCE__MAILTEMPLATE__TEXT'],
                    'isys_maintenance_mailtemplate__status' => C__RECORD_STATUS__NORMAL
                ]
            );

            isys_notify::success($this->language->get('LC__MAINTENANCE__NOTIFY__SAVE_SUCCESS'));
        } catch (Exception $e) {
            isys_notify::error(
                $this->language->get('LC__MAINTENANCE__NOTIFY__SAVE_FAILURE') . $e->getMessage(),
                ['sticky' => true]
            );
        }

        return $l_id;
    }

    /**
     * Method for deleting one or more mail templates.
     *
     * @param mixed $p_id
     *
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    private function process_mailtemplate__delete($p_id)
    {
        if (!is_array($p_id)) {
            $p_id = [$p_id];
        }

        Mailtemplate::instance($this->databaseComponent)->deleteMailtemplate(array_filter($p_id, 'is_numeric'));
    }

    /**
     * @return isys_maintenance_auth
     */
    public static function getAuth()
    {
        return isys_maintenance_auth::instance();
    }
}
