<?php

use idoit\Component\Helper\Date;

/**
 * i-doit
 *
 * Maintenance PDF class.
 *
 * @package     modules
 * @subpackage  maintenance
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class isys_maintenance_reportview_pdf extends TCPDF
{
    /**
     * This variable will hold the filepath to the header logo.
     *
     * @var  string
     */
    protected $m_logo_filepath = '';

    /**
     * This variable will hold the maintenance data.
     *
     * @var  array
     */
    protected $m_maintenance_data = [];

    /**
     * This variable will hold the object data.
     *
     * @var  array
     */
    protected $m_object_data = [];

    /**
     * This variable will hold the filename of the exported PDF.
     *
     * @var  string
     */
    protected $m_output_filename = '';

    /**
     * @var isys_component_template_language_manager
     */
    protected $language;

    /**
     * Method for returning the PDFs filename.
     *
     * @return  string
     */
    public function get_pdf_filename()
    {
        return $this->m_output_filename;
    }

    /**
     * Method for setting the logo filepath.
     *
     * @param   string $p_filepath
     *
     * @return  isys_maintenance_reportview_pdf
     */
    public function set_logo_filepath($p_filepath)
    {
        $this->m_logo_filepath = $p_filepath;

        return $this;
    }

    /**
     * Method for setting the maintenance data.
     *
     * @param   array $p_maintenance_data
     *
     * @return  isys_maintenance_reportview_pdf
     */
    public function set_maintenance_data(array $p_maintenance_data)
    {
        $this->m_maintenance_data = $p_maintenance_data;

        return $this;
    }

    /**
     * Method for setting the object data.
     *
     * @param   array $p_object_data
     *
     * @return  isys_maintenance_reportview_pdf
     */
    public function set_object_data(array $p_object_data)
    {
        $this->m_object_data = $p_object_data;

        return $this;
    }

    /**
     * Method for rendering the first page (with the maintenance details).
     *
     * @return  isys_maintenance_reportview_pdf
     */
    public function first_page()
    {
        return $this->AddPage()->defaults();
    }

    /**
     * Render the colored table.
     *
     * @return  isys_maintenance_reportview_pdf
     */
    public function fill_report()
    {
        $l_empty_value = isys_tenantsettings::get('gui.empty_value', '-');
        $l_last_month = $l_last_year = $i = 0;
        $locales = isys_application::instance()->container->get('locales');

        $this->SetFillColor(128)
            ->SetDrawColor(96)
            ->SetTextColor(255)
            ->Cell(15, 6, 'ID', 'B', 0, 'C', true)
            ->Cell(50, 6, $this->language->get('LC_UNIVERSAL__OBJECT'), 'B', 0, 'C', true)
            ->Cell(35, 6, $this->language->get('LC__MAINTENANCE__PLANNING__DATE_SPAN'), 'B', 0, 'C', true)
            ->Cell(60, 6, $this->language->get('LC__MAINTENANCE__PDF__PERSON') . ' (' . $this->language->get('LC__MAINTENANCE__PDF__PERSON_ROLE') . ')', 'B', 0, 'C', true)
            ->Cell(30, 6, $this->language->get('LC__MAINTENANCE__PDF__FINISHED'), 'B', 1, 'C', true)
            ->SetFillColor(220);

        foreach ($this->m_maintenance_data as $l_maintenance_id => $l_maintenance_data) {
            $l_month = (int)date('n', $l_maintenance_data['from']);
            $l_year = (int)date('Y', $l_maintenance_data['from']);

            if ($l_last_month != $l_month || $l_last_year != $l_year) {
                $this->SetFillColor(160)
                    ->SetTextColor(255)
                    ->Cell(0, 6, Date::getMonthName($l_month) . ' ' . $l_year, 'TB', 1, 'C', true);
            }

            foreach ($l_maintenance_data['objects'] as $l_obj_id) {
                $l_obj_data = $this->m_object_data[$l_obj_id];

                $l_odd = (bool)($i % 2);

                $rowHeight = 0;

                $contacts = $l_obj_data['roles'];

                if (is_array($contacts)) {
                    $contacts = $this->flattenArray($contacts);

                    /*
                     * @see  WARTUNG-24  Because a "contact" string can be quite long, we'll add some additional vertical space.
                     * Just in case the contact strings break to the next line and makes the row higher.
                     *
                     * We assume the name will break to the next line after 60 characters.
                     */
                    $additionalLines = array_sum(array_map(function($contact) {
                        return (int)mb_strlen($contact) / 60;
                    }, $contacts));

                    $rowHeight = (count($contacts) + $additionalLines) * 4;
                    $contacts = implode(PHP_EOL, $contacts);
                }

                // Set the minimum height to 9.
                $rowHeight = max($rowHeight, 9);

                $this->SetFillColor(($l_odd ? 255 : 230))
                    ->SetTextColor(0)
                    ->MultiCell(15, $rowHeight, $l_obj_data['isys_obj__id'], 'BT', 'C', true, 0)
                    ->MultiCell(50, $rowHeight, $this->language->get($l_obj_data['isys_obj_type__title']) . ' > ' . $l_obj_data['isys_obj__title'], 'BT', 'L', true, 0)
                    ->MultiCell(35, $rowHeight, $l_maintenance_data['fromFormatted'] . PHP_EOL . $l_maintenance_data['toFormatted'], 'BT', 'C', true, 0)
                    ->MultiCell(60, $rowHeight, (empty($contacts) ? $l_empty_value : $contacts), 'BT', 'L', true, 0);

                if ($l_maintenance_data['isys_maintenance__finished'] && strtotime($l_maintenance_data['isys_maintenance__finished']) > $l_maintenance_data['from']) {
                    $this->SetTextColor(0, 99, 0)
                        ->MultiCell(30, $rowHeight, $locales->fmt_datetime($l_maintenance_data['isys_maintenance__finished']), 'BT', 'C', true);
                } else {
                    $this->SetTextColor(170, 0, 0)
                        ->MultiCell(30, $rowHeight, $this->language->get('LC__UNIVERSAL__NO'), 'BT', 'C', true);
                }

                $i++;
            }

            $l_last_month = $l_month;
            $l_last_year = $l_year;
        }

        return $this->SetTextColor(0);
    }

    /**
     * Method for saving the created PDF to the temp directory.
     *
     * @param  string $p_filename
     */
    public function save_to_temp($p_filename = null)
    {
        if ($p_filename === null) {
            $p_filename = date('Y-m-d_H-i-s') . '_idoit-maintenance-report.pdf';
        }

        $this->Output(BASE_DIR . 'temp/' . $p_filename, 'F');

        $this->m_output_filename = $p_filename;
    }

    /**
     * Overwriting the "AddPage" method for method chaining.
     *
     * @param string $orientation
     * @param string $format
     * @param bool   $keepmargins
     * @param bool   $tocpage
     *
     * @return isys_maintenance_reportview_pdf
     */
    public function AddPage($orientation = '', $format = '', $keepmargins = false, $tocpage = false)
    {
        parent::AddPage($orientation, $format);

        return $this;
    }

    /**
     * Overwriting the "Cell" method for method chaining.
     *
     * @param int    $w
     * @param int    $h
     * @param string $txt
     * @param int    $border
     * @param int    $ln
     * @param string $align
     * @param bool   $fill
     * @param string $link
     * @param int    $stretch
     * @param bool   $ignore_min_height
     * @param string $calign
     * @param string $valign
     *
     * @return isys_maintenance_reportview_pdf
     */
    public function Cell($w, $h = 0, $txt = '', $border = 0, $ln = 0, $align = '', $fill = false, $link = '', $stretch = 0, $ignore_min_height = false, $calign = 'T', $valign = 'M')
    {
        parent::Cell($w, $h, $txt, $border, $ln, $align, $fill, $link);

        return $this;
    }

    /**
     * @param int    $w
     * @param int    $h
     * @param string $txt
     * @param int    $border
     * @param string $align
     * @param bool   $fill
     * @param int    $ln
     * @param string $x
     * @param string $y
     * @param bool   $reseth
     * @param int    $stretch
     * @param bool   $ishtml
     * @param bool   $autopadding
     * @param int    $maxh
     * @param string $valign
     * @param bool   $fitcell
     *
     * @return $this
     */
    public function MultiCell($w, $h, $txt, $border = 0, $align = 'J', $fill = false, $ln = 1, $x = '', $y = '', $reseth = true, $stretch = 0, $ishtml = false, $autopadding = true, $maxh = 0, $valign = 'T', $fitcell = false)
    {
        parent::MultiCell($w, $h, $txt, $border, $align, $fill, $ln, $x, $y, $reseth, $stretch, $ishtml, $autopadding, $maxh, $valign, $fitcell);

        return $this;
    }

    /**
     * Overwriting the "Footer" method.
     */
    public function Footer()
    {
        parent::Footer();

        // Position at 15mm from bottom.
        $this->SetY(-15);
        $this->SetFont('helvetica', '', 7);
        $this->Cell(0, 5, $this->language->get('LC__MAINTENANCE__PDF__PAGE') . ' ' . $this->PageNo() . ' / {:ptp:}', 0, 0, 'C');
    }

    /**
     * Overwriting the "Header" method.
     */
    public function Header()
    {
        parent::Header();

        if (!empty($this->m_logo_filepath) && file_exists($this->m_logo_filepath)) {
            $imageWidth = 0;
            $imageSize = @getimagesize($this->m_logo_filepath);

            if (is_array($imageSize) && $imageSize[0]) {
                $imageWidth = $imageSize[0];

                if ($imageWidth > 200) {
                    $imageWidth = 200;
                }
            }

            $this->Image($this->m_logo_filepath, 0, 0, $imageWidth);
        }

        $this->SetTopMargin(5);
        $this->Ln(5);

        // Helvetica bold 15.
        $this->SetDrawColor(96)
            ->SetTextColor(96)
            ->SetFont('helvetica', 'b', 15);

        // Title.
        $this->Cell(0, 8, $this->title, 1, 2, 'C')
            ->defaults()
            ->Ln(5);

        $this->SetTopMargin(25);
    }

    /**
     * Overwriting the "SetDrawColor" method for method chaining.
     *
     * @param int    $col1
     * @param int    $col2
     * @param int    $col3
     * @param int    $col4
     * @param bool   $ret
     * @param string $name
     *
     * @return isys_maintenance_reportview_pdf
     */
    public function SetDrawColor($col1 = 0, $col2 = -1, $col3 = -1, $col4 = -1, $ret = false, $name = '')
    {
        parent::SetDrawColor($col1, $col2, $col3);

        return $this;
    }

    /**
     * Overwriting the "SetFillColor" method for method chaining.
     *
     * @param int    $col1
     * @param int    $col2
     * @param int    $col3
     * @param int    $col4
     * @param bool   $ret
     * @param string $name
     *
     * @return isys_maintenance_reportview_pdf
     */
    public function SetFillColor($col1 = 0, $col2 = -1, $col3 = -1, $col4 = -1, $ret = false, $name = '')
    {
        parent::SetFillColor($col1, $col2, $col3);

        return $this;
    }

    /**
     * Overwriting the "SetTextColor" method for method chaining.
     *
     * @param int    $col1
     * @param int    $col2
     * @param int    $col3
     * @param int    $col4
     * @param bool   $ret
     * @param string $name
     *
     * @return isys_maintenance_reportview_pdf
     */
    public function SetTextColor($col1 = 0, $col2 = -1, $col3 = -1, $col4 = -1, $ret = false, $name = '')
    {
        parent::SetTextColor($col1, $col2, $col3);

        return $this;
    }

    /**
     * Overwriting the "SetTitle" method for method chaining.
     *
     * @param   string $p_title
     *
     * @return  isys_maintenance_reportview_pdf
     */
    public function SetTitle($p_title)
    {
        parent::SetTitle($p_title);

        return $this;
    }

    /**
     * Restores the default font.
     *
     * @return  isys_maintenance_reportview_pdf
     */
    protected function defaults()
    {
        $this->SetDrawColor(0)
            ->SetTextColor(0)
            ->SetFillColor(255)
            ->SetFont('helvetica', '', 8);

        return $this;
    }

    /**
     * Helper method to flatten a multi-dimensional array to one level.
     *
     * @param $array
     *
     * @return array
     */
    private function flattenArray($array)
    {
        if (!is_array($array)) {
            return $array;
        }

        $_flattened = [];

        // Rewrite keys
        foreach ($array as $value) {
            $_flattened[] = $this->flattenArray($value);
        }

        // Flatten
        $flattened = [];
        foreach ($_flattened as $value) {
            if (is_array($value)) {
                $flattened = array_merge($flattened, $value);
            } else {
                $flattened[] = $value;
            }
        }

        return array_filter(array_unique($flattened));
    }

    /**
     * Create a new "isys_maintenance_reportview_pdf" instance.
     *
     * @param  string $p_orientation
     * @param  string $p_unit
     * @param  string $p_format
     */
    public function __construct($p_orientation = 'P', $p_unit = 'mm', $p_format = 'A4')
    {
        parent::__construct($p_orientation, $p_unit, $p_format);

        $this->language = isys_application::instance()->container->get('language');

        $this->SetTitle($this->language->get('LC__MAINTENANCE__PDF__TITLE'))->defaults();
    }
}
