<?php

use Carbon\Carbon;
use idoit\Component\Interval\Config;
use idoit\Component\Interval\Interval;
use idoit\Module\Maintenance\Model\Maintenance;

/**
 * i-doit
 *
 * Maintenance DAO.
 *
 * @package     modules
 * @subpackage  maintenance
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @version     1.0.2
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       i-doit 1.5.0
 */
class isys_maintenance_dao extends isys_cmdb_dao
{
    /**
     * This array holds all fields of our "isys_maintenance" table.
     *
     * @var  array
     */
    protected $m_planning_fields = [
        'isys_maintenance__id'                                => 'convert_sql_id',
        'isys_maintenance__isys_maintenance_type__id'         => 'convert_sql_id',
        'isys_maintenance__date_from'                         => 'convert_sql_datetime',
        'isys_maintenance__date_to'                           => 'convert_sql_datetime',
        'isys_maintenance__interval'                          => 'convert_sql_text',
        'isys_maintenance__comment'                           => 'convert_sql_text',
        'isys_maintenance__finished'                          => 'convert_sql_datetime',
        'isys_maintenance__mail_dispatched'                   => 'convert_sql_datetime',
        'isys_maintenance__isys_maintenance_mailtemplate__id' => 'convert_sql_id',
        'isys_maintenance__isys_contact_tag__id'              => 'convert_sql_id',
        'isys_maintenance__status'                            => 'convert_sql_int'
    ];

    /**
     * Method for retrieving planning data.
     *
     * @param   mixed  $p_id
     * @param   string $p_condition
     *
     * @return  isys_component_dao_result
     * @throws  Exception
     * @throws  isys_exception_database
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function get_data($p_id = null, $p_condition = '')
    {
        $l_sql = 'SELECT * FROM isys_maintenance
			LEFT JOIN isys_maintenance_type ON isys_maintenance_type__id = isys_maintenance__isys_maintenance_type__id
			LEFT JOIN isys_maintenance_mailtemplate ON isys_maintenance_mailtemplate__id = isys_maintenance__isys_maintenance_mailtemplate__id
			WHERE TRUE';

        if ($p_id !== null) {
            if (is_array($p_id)) {
                $l_sql .= ' AND isys_maintenance__id ' . $this->prepare_in_condition($p_id);
            } else {
                $l_sql .= ' AND isys_maintenance__id  = ' . $this->convert_sql_id($p_id);
            }
        }

        return $this->retrieve($l_sql . ' ' . $p_condition . ';');
    }

    /**
     * @param   integer $p_object_id
     * @param   string  $p_from
     * @param   string  $p_to
     *
     * @return  isys_component_dao_result
     * @throws  Exception
     * @throws  isys_exception_database
     */
    public function get_data_by_maintenance_object($p_object_id, $p_from = null, $p_to = null)
    {
        $l_sql = 'SELECT * FROM isys_obj
			INNER JOIN isys_maintenance_2_object ON isys_maintenance_2_object__isys_obj__id = isys_obj__id
			LEFT JOIN isys_maintenance ON isys_maintenance__id = isys_maintenance_2_object__isys_maintenance__id
			WHERE isys_obj__id = ' . $this->convert_sql_id($p_object_id);

        if ($p_from !== null && !empty($p_from)) {
            $l_sql .= ' AND isys_maintenance__date_from >= ' . $this->convert_sql_datetime($p_from);
        }

        if ($p_to !== null && !empty($p_to)) {
            $l_sql .= ' AND isys_maintenance__date_to <= ' . $this->convert_sql_datetime($p_to);
        }

        return $this->retrieve($l_sql . ';');
    }

    /**
     * Method for saving a maintenance plan.
     *
     * @param   integer $p_id
     * @param   array   $p_data
     * @param   array   $p_objects
     * @param   array   $p_contacts
     *
     * @return  integer
     * @throws  isys_exception_dao
     * @throws  isys_exception_database
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function savePlanning($p_id = null, array $p_data = null, array $p_objects = null, array $p_contacts = null)
    {
        $l_data = [];

        foreach ($p_data as $l_key => $l_value) {
            if (isset($this->m_planning_fields[$l_key])) {
                $l_data[] = $l_key . ' = ' . call_user_func([$this, $this->m_planning_fields[$l_key]], $l_value);
            }
        }

        if (is_array($p_data) && count($l_data)) {
            if ($p_id === null) {
                $l_sql = 'INSERT INTO isys_maintenance SET ' . implode(', ', $l_data);
            } else {
                $l_sql = 'UPDATE isys_maintenance SET ' . implode(', ', $l_data) . ' WHERE isys_maintenance__id = ' . $this->convert_sql_id($p_id);
            }

            if ($this->update($l_sql) && $this->apply_update()) {
                if ($p_id === null) {
                    $p_id = $this->get_last_id_from_table('isys_maintenance');
                }
            } else {
                throw new isys_exception_database($this->m_last_error);
            }
        }

        if ($p_id > 0 && is_array($p_objects)) {
            // First we remove all connected objects.
            $this->update('DELETE FROM isys_maintenance_2_object WHERE isys_maintenance_2_object__isys_maintenance__id = ' . $this->convert_sql_id($p_id) . ';');
            $this->apply_update();

            $queryItems = [];

            foreach ($p_objects as $l_object) {
                $queryItems[] = '(' . $this->convert_sql_id($p_id) . ', ' . $this->convert_sql_id($l_object) . ')';
            }

            if (count($queryItems)) {
                $assignmentQuery = 'INSERT INTO isys_maintenance_2_object 
                    (isys_maintenance_2_object__isys_maintenance__id, isys_maintenance_2_object__isys_obj__id) 
                    VALUES ' . implode(',', $queryItems) . ';';

                // Now we add all the connected objects.
                $this->update($assignmentQuery);
                $this->apply_update();
            }
        }

        if ($p_id > 0 && is_array($p_contacts)) {
            // First we remove all connected objects.
            $this->update('DELETE FROM isys_maintenance_2_contact WHERE isys_maintenance_2_contact__isys_maintenance__id = ' . $this->convert_sql_id($p_id) . ';');
            $this->apply_update();

            $queryItems = [];

            foreach ($p_contacts as $l_contact) {
                $queryItems[] = '(' . $this->convert_sql_id($p_id) . ', ' . $this->convert_sql_id($l_contact) . ')';
            }

            if (count($queryItems)) {
                $assignmentQuery = 'INSERT INTO isys_maintenance_2_contact 
                    (isys_maintenance_2_contact__isys_maintenance__id, isys_maintenance_2_contact__isys_obj__id) 
                    VALUES ' . implode(',', $queryItems) . ';';

                // Now we add all the connected objects.
                $this->update($assignmentQuery);
                $this->apply_update();
            }
        }

        return $p_id;
    }

    /**
     * Method for deleting plannings.
     *
     * @param   array $p_ids
     *
     * @return  boolean
     * @throws  isys_exception_dao
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function delete_planning(array $p_ids)
    {
        if (!count($p_ids)) {
            return true;
        } // if

        $l_sql = 'DELETE FROM isys_maintenance WHERE isys_maintenance__id ' . $this->prepare_in_condition($p_ids) . ';';
        $l_object_sql = 'DELETE FROM isys_maintenance_2_object WHERE isys_maintenance_2_object__isys_maintenance__id ' . $this->prepare_in_condition($p_ids) . ';';
        $l_contact_sql = 'DELETE FROM isys_maintenance_2_contact WHERE isys_maintenance_2_contact__isys_maintenance__id ' . $this->prepare_in_condition($p_ids) . ';';

        return ($this->update($l_sql) && $this->update($l_object_sql) && $this->update($l_contact_sql) && $this->apply_update());
    } // function

    /**
     * Method to find out if the given object is currently in maintenance.
     *
     * @param   integer $p_obj_id
     * @param   string  $p_date
     *
     * @return  boolean
     * @throws  Exception
     * @throws  isys_exception_database
     * @throws  isys_exception_general
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function is_in_maintenance($p_obj_id, $p_date = null)
    {
        if ($p_obj_id > 0) {
            if ($p_date === null) {
                // ID-1933  Because of the data type DATE the "NOW()" function will not work, so we need "CURDATE()".
                // WARTUNG-2  Now we can use "NOW()" :)
                $p_date = 'NOW()';
            } // if

            $sql = 'SELECT isys_maintenance__id FROM isys_maintenance
				INNER JOIN isys_maintenance_2_object ON isys_maintenance_2_object__isys_maintenance__id = isys_maintenance__id
				WHERE isys_maintenance_2_object__isys_obj__id = ' . $this->convert_sql_id($p_obj_id) . '
				AND ((isys_maintenance__date_from <= ' . $this->convert_sql_datetime($p_date) . ' AND isys_maintenance__date_to >= ' . $this->convert_sql_datetime($p_date) . ')
                OR (isys_maintenance__date_from IS NULL AND isys_maintenance__date_to >= ' . $this->convert_sql_datetime($p_date) . ')
                OR (isys_maintenance__date_from <= ' . $this->convert_sql_datetime($p_date) . ' AND isys_maintenance__date_to IS NULL))
				LIMIT 1;';

            if (count($this->retrieve($sql))) {
                return true;
            }

            // WARTUNG-3 If we have found no maintenance, check if we are dealing with recurring maintenances.
            $sql = 'SELECT isys_maintenance__id, isys_maintenance__date_from, isys_maintenance__date_to, isys_maintenance__interval FROM isys_maintenance
				INNER JOIN isys_maintenance_2_object ON isys_maintenance_2_object__isys_maintenance__id = isys_maintenance__id
				WHERE isys_maintenance_2_object__isys_obj__id = ' . $this->convert_sql_id($p_obj_id) . '
				AND isys_maintenance__interval <> "";';

            $res = $this->retrieve($sql);

            // If nothing was found, we have no running maintenance.
            if (!count($res)) {
                return false;
            }

            $nowCarbon = new Carbon();

            // If we found a row (or multiple ones) we need to check if we are inside a interval.
            while ($row = $res->get_row()) {
                if (!isys_format_json::is_json_array($row['isys_maintenance__interval'])) {
                    continue;
                }

                $config = Config::byJSON($row['isys_maintenance__interval']);
                $interval = new Interval($config, $nowCarbon);

                // If the interval is reached today we'll have to check the time.
                if ($interval->isIntervalReached()) {
                    $dateTimeFrom = new Carbon($row['isys_maintenance__date_from']);
                    $dateTimeFrom->setDate(date('Y'), date('m'), date('d'));

                    $dateTimeTo = new Carbon($row['isys_maintenance__date_to']);
                    $dateTimeTo->setDate(date('Y'), date('m'), date('d'));

                    // This will only work if the times are documented correctly!
                    if ($nowCarbon->between($dateTimeFrom, $dateTimeTo)) {
                        return true;
                    }
                }

                // If we have a last interval we need to check if it's the duration
                if ($interval->hasLastIntervalDate()) {
                    $lastInterval = new Carbon($interval->getLastIntervalDate());
                    $lengthOfMaintenance = (new Carbon($row['isys_maintenance__date_from']))->diffInSeconds(new Carbon($row['isys_maintenance__date_to']));

                    // We'll check if we are currently in the timespan between the last interval + the length of the maintenance.
                    if ($nowCarbon->between($lastInterval, $lastInterval->copy()
                        ->addSeconds($lengthOfMaintenance))) {
                        return true;
                    }
                }
            }
        }

        return false;
    }

    /**
     * Method for finishing a maintenance plan + adding a entry to the logbook.
     *
     * @param   integer $p_id
     * @param   string  $p_comment
     *
     * @return  boolean
     * @throws  Exception
     * @throws  \idoit\Exception\DateException
     * @throws  \idoit\Exception\IntervalException
     * @throws  isys_exception_dao
     * @throws  isys_exception_database
     * @throws  isys_exception_general
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function finish_maintenance_planning($p_id, $p_comment)
    {
        $maintenanceDao = Maintenance::instance($this->m_db);
        $session = isys_application::instance()->container->get('session');
        $languageManager = isys_application::instance()->container->get('language');

        $maintenanceDao->saveMaintenance($p_id, [
            'isys_maintenance__finished'       => 'NOW()',
            'isys_maintenance__finish_comment' => $p_comment,
            'isys_maintenance__finish_person'  => $session->get_current_username() . ' (#' . $session->get_user_id() . ')'
        ]);

        $l_object_res = $maintenanceDao->getMaintenanceObjects($p_id);

        if (!count($l_object_res)) {
            throw new isys_exception_general($languageManager->get('LC__MAINTENANCC__EXCEPTION__NO_OBJECTS_SELECTED'));
        }

        while ($l_object_row = $l_object_res->get_row()) {
            // Insert the change to the logbook as well.
            isys_component_dao_logbook::instance($this->m_db)
                ->set_entry(
                    $languageManager->get('LC__MAINTENANCE__PLANNING__FINISHED'),
                    $this->get_last_query(),
                    null,
                    C__LOGBOOK__ALERT_LEVEL__0,
                    $l_object_row['isys_obj__id'],
                    $l_object_row['isys_obj__title'],
                    $l_object_row['isys_obj_type__title'],
                    'LC__CMDB__CATG__LOCATION',
                    C__LOGBOOK_SOURCE__MAINTENANCE,
                    serialize([]),
                    strip_tags($p_comment)
                );
        }

        // @see  WARTUNG-5  Check if we are finishing a recurring maintenance - if so: duplicate it and update the configuration.
        $maintenanceData = $maintenanceDao->getMaintenances($p_id)
            ->get_row();

        $intervalConfig = $maintenanceData['isys_maintenance__interval'];

        if (!empty($intervalConfig) && isys_format_json::is_json_array($intervalConfig)) {
            // Update the interval configuration.
            $config = Config::byJSON($intervalConfig);
            $interval = new Interval($config, new DateTime($maintenanceData['isys_maintenance__date_from']));

            // If the interval has expired, we don't need any further action.
            if ($interval->isIntervalExpired()) {
                isys_notify::info('Der Wartungsinterval ist abgeschlossen', ['sticky' => true]);

                return true;
            }

            // Duplicate the maintenance and proceed.
            $newMaintenanceId = $maintenanceDao->duplicateMaintenance($p_id);
            $maintenanceData = $maintenanceDao->getMaintenances($newMaintenanceId)
                ->get_row();

            isys_notify::info('Eine neue Wartung wird angelegt mit #' . $newMaintenanceId, ['sticky' => true]);

            // Here we find out the time-delta between maintenance start and end (in seconds).
            $timeDelta = Carbon::instance(new DateTime($maintenanceData['isys_maintenance__date_from']))
                ->diffInSeconds(Carbon::instance(new DateTime($maintenanceData['isys_maintenance__date_to'])));

            $nextIntervalStart = $interval->getNextIntervalDate();

            $config->setInitialDate($nextIntervalStart);

            // If the end criteria is "after x events" we need to decrease that number.
            if ($config->getEndAfter() === Config::END_AFTER_EVENTS) {
                $config->setEndDetails(((int)$config->getEndDetails()) - 1);
            }

            unset($maintenanceData['isys_maintenance__id'], $maintenanceData['isys_maintenance__finished'], $maintenanceData['isys_maintenance__mail_dispatched']);
            $maintenanceData['isys_maintenance__date_from'] = $nextIntervalStart->format('Y-m-d H:i:s');
            $maintenanceData['isys_maintenance__date_to'] = $nextIntervalStart->copy()
                ->modify($timeDelta . ' seconds')
                ->format('Y-m-d H:i:s');
            $maintenanceData['isys_maintenance__interval'] = isys_format_json::encode($config->toArray());

            $maintenanceDao->saveMaintenance($newMaintenanceId, $maintenanceData);
        }

        return true;
    }

    /**
     * @param DateTime  $from
     * @param DateTime  $to
     * @param array|int $typeId
     *
     * @return array
     * @throws isys_exception_database
     */
    public function getMaintenancesBetween(DateTime $from, DateTime $to, $typeId = null)
    {
        $return = [];
        $sqlFrom = $this->convert_sql_datetime($from->format('Y-m-d H:i:s'));
        $sqlTo = $this->convert_sql_datetime($to->format('Y-m-d H:i:s'));
        $carbonFrom = Carbon::instance($from);
        $carbonTo = Carbon::instance($to);

        $sql = 'SELECT isys_maintenance.*, isys_contact_tag__id, isys_contact_tag__title FROM isys_maintenance
			LEFT JOIN isys_contact_tag ON isys_contact_tag__id = isys_maintenance__isys_contact_tag__id
			WHERE isys_maintenance__status = ' . $this->convert_sql_int(C__RECORD_STATUS__NORMAL) . '
			AND ((isys_maintenance__date_from >= ' . $sqlFrom . ' AND isys_maintenance__date_to <= ' . $sqlTo . ') OR TRIM(isys_maintenance__interval) <> "")';

        if ($typeId !== null) {
            if (is_numeric($typeId) && $typeId > 0) {
                $sql .= ' AND isys_maintenance__isys_maintenance_type__id = ' . $this->convert_sql_id($typeId);
            } elseif (is_array($typeId) && count($typeId)) {
                $sql .= ' AND isys_maintenance__isys_maintenance_type__id ' . $this->prepare_in_condition($typeId);
            }
        }

        $result = $this->retrieve($sql . ';');

        // We can not simply rely on MySQL - we also need to check the recurring maintenances!
        while ($row = $result->get_row()) {
            // If no interval is defined, we can skip the following logic.
            if (empty($row['isys_maintenance__interval']) || $row['isys_maintenance__finished']) {
                $return[] = $row;
                continue;
            }

            try {
                $intervalFrom = new DateTime($row['isys_maintenance__date_from']);
                $intervalTo = new DateTime($row['isys_maintenance__date_to']);

                $timeDelta = Carbon::instance($intervalFrom)
                    ->diffInSeconds(Carbon::instance($intervalTo));

                $config = Config::byJSON($row['isys_maintenance__interval']);
                $config->setExpireIfCheckDateIsBeforeInitialDate(false);

                $interval = new Interval($config, new DateTime());

                // Skip expired and "out-of-scope" intervals.
                if ($interval->isIntervalExpired() || !$interval->getCheckDate()->between($from, $to)) {
                    continue;
                }

                if ($interval->hasLastIntervalDate() && Carbon::instance($interval->getLastIntervalDate())->between($carbonFrom, $carbonTo)) {
                    $clonedInterval = clone $interval;

                    $previousIntervals = $clonedInterval->getAllPreviousIntervals($carbonFrom);

                    if (is_array($previousIntervals)) {
                        // We got a previous interval within the given time - check further!
                        foreach ($previousIntervals as $previousInterval) {
                            $row['isys_maintenance__date_from'] = $previousInterval->format('Y-m-d H:i:s');
                            $row['isys_maintenance__date_to'] = $previousInterval->modify($timeDelta . ' seconds')->format('Y-m-d H:i:s');

                            $return[] = $row;
                        }
                    }
                }

                if ($interval->isIntervalReached()) {
                    // Use the original interval time or else the current time might get used.
                    $row['isys_maintenance__date_from'] = $interval->getCheckDate()->format('Y-m-d ') . $intervalFrom->format('H:i:s');
                    $row['isys_maintenance__date_to'] = $interval->getCheckDate()->modify($timeDelta . ' seconds')->format('Y-m-d ') . $intervalTo->format('H:i:s');

                    $return[] = $row;
                }

                if ($interval->hasNextIntervalDate() && Carbon::instance($interval->getNextIntervalDate())->between($carbonFrom, $carbonTo)) {
                    $clonedInterval = clone $interval;

                    $nextIntervals = $clonedInterval->getAllUpcomingIntervals($carbonTo);

                    if (is_array($nextIntervals)) {
                        // We got a previous interval within the given time - check further!
                        foreach ($nextIntervals as $nextInterval) {
                            $row['isys_maintenance__date_from'] = $nextInterval->format('Y-m-d H:i:s');
                            $row['isys_maintenance__date_to'] = $nextInterval->modify($timeDelta . ' seconds')->format('Y-m-d H:i:s');

                            $return[] = $row;
                        }
                    }
                }
            } catch (Exception $e) {
                // Do nothing!
            }
        }

        return $return;
    }
}
