<?php

namespace idoit\Module\Maintenance\Component;

use idoit\Module\Maintenance\Model\Maintenance;
use isys_application;
use isys_cmdb_dao_category_g_contact;
use isys_cmdb_dao_category_g_mail_addresses;
use isys_cmdb_dao_category_property;
use isys_cmdb_dao_category_s_organization_person;
use isys_cmdb_dao_category_s_person_group_members;
use isys_cmdb_dao_category_s_person_master;
use isys_component_database;
use isys_exception_general;
use isys_format_json;
use isys_helper_link;
use isys_library_mail;
use isys_locale;
use isys_maintenance_dao;
use isys_module_report_pro;
use isys_tenantsettings;
use Symfony\Component\Console\Helper\Table;
use Symfony\Component\Console\Output\BufferedOutput;

/**
 * i-doit
 *
 * Maintenance component for E-Mail.
 *
 * @package     idoit\Module\Maintenance\Model
 * @package     maintenance
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @version     1.0
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       Maintenance 1.1
 */
class Mailer
{
    /**
     * @var isys_component_database
     */
    private $db;

    /**
     * @var isys_locale
     */
    private $locale;

    /**
     * @var isys_maintenance_dao
     */
    private $dao;

    /**
     * @var  array
     */
    private $mailContacts = [];

    /**
     * @var array
     */
    private $maintenanceObjects = [];

    /**
     * @var  array
     */
    private $internalContacts = [];

    /**
     * Mailer constructor.
     *
     * @param isys_component_database $database
     * @param isys_locale             $locale
     */
    public function __construct(isys_component_database $database, isys_locale $locale)
    {
        $this->db = $database;
        $this->locale = $locale;
        $this->dao = isys_maintenance_dao::instance($this->db);
    }

    /**
     * Return all collected email recipients.
     *
     * @return array
     */
    public function getCollectedRecipients()
    {
        return $this->mailContacts;
    }

    /**
     * Return all collected objects.
     *
     * @return array
     */
    public function getAffectedObjects()
    {
        return $this->maintenanceObjects;
    }

    /**
     * This method will send the mails to all necessary contacts from the the maintenance planning.
     *
     * @param integer $maintenanceId
     *
     * @return  boolean
     * @throws  isys_exception_general
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function sendMaintenancePlanningMail($maintenanceId)
    {
        $languageManager = isys_application::instance()->container->get('language');
        $contactDao = isys_cmdb_dao_category_g_contact::instance($this->db);
        $contactResult = Maintenance::instance($this->db)->getMaintenanceContacts($maintenanceId);

        // Reset the maintenance objects and recipients.
        $this->maintenanceObjects = [];
        $this->mailContacts = [];
        // @see WARTUNG-63 Empty the internal contact cache.
        $this->internalContacts = [];

        if (count($contactResult)) {
            while ($contactRow = $contactResult->get_row()) {
                $this->getEmailRecipients($contactRow);
            }
        }

        $objectResult = Maintenance::instance($this->db)->getMaintenanceObjects($maintenanceId);

        if (!count($objectResult)) {
            throw new isys_exception_general($languageManager->get('LC__MAINTENANCC__EXCEPTION__NO_OBJECTS_SELECTED'));
        }

        $maintenance = $this->dao->get_data($maintenanceId)
            ->get_row();

        if (!$maintenance['isys_maintenance__isys_maintenance_mailtemplate__id']) {
            throw new isys_exception_general($languageManager->get('LC__MAINTENANCC__EXCEPTION__NO_MAILTEMPLATE_SELECTED'));
        }

        $notificationObjects = [];
        $combineObjects = (bool) $maintenance['isys_maintenance__combine_objects'];

        while ($objectRow = $objectResult->get_row()) {
            $this->maintenanceObjects[] = $objectRow;
            $maintenance['assignedRoles'] = (string)$maintenance['assignedRoles'];
            // Add the contacts from each object via role.
            if ($maintenance['assignedRoles'] !== '') {
                $statusNormal = $contactDao->convert_sql_int(C__RECORD_STATUS__NORMAL);

                $roles = explode(',', $maintenance['assignedRoles']);

                $contactResult = $contactDao->get_contact_objects_by_tag(
                    $objectRow['isys_obj__id'],
                    $roles,
                    " AND isys_catg_contact_list__status = {$statusNormal} AND isys_obj__status = {$statusNormal} "
                );

                if (count($contactResult)) {
                    while ($contactRow = $contactResult->get_row()) {
                        $this->getEmailRecipients($contactRow);
                    }
                }
            }

            if (!count($this->mailContacts)) {
                throw new isys_exception_general($languageManager->get('LC__MAINTENANCC__EXCEPTION__NO_RECIPIENTS'));
            }

            // And finally we generate the email and send it.
            $mailer = new isys_library_mail();

            $mailer
                ->set_charset('UTF-8')
                ->set_content_type(isys_library_mail::C__CONTENT_TYPE__PLAIN)
                ->set_backend(isys_library_mail::C__BACKEND__SMTP);

            foreach ($this->mailContacts as $contact) {
                $mailer->AddAddress($contact['email'], $contact['name']);
            }

            // Replace variables in our subject and mail body.
            $subject = $this->replaceSingularPlaceholders(
                $mailer->get_subject() . ' ' . $maintenance['isys_maintenance_mailtemplate__title'],
                $maintenance,
                $objectRow['isys_obj__id']
            );

            $mailBody = $this->replaceSingularPlaceholders(
                $maintenance['isys_maintenance_mailtemplate__text'],
                $maintenance,
                $objectRow['isys_obj__id']
            );

            $notificationObjects[] = (int)$objectRow['isys_obj__id'];

            // @see WARTUNG-90 Keep the original logic, if the emails should not be combined.
            if (!$combineObjects) {
                // Also replace the report placeholder for single object mails.
                $mailBody = $this->replaceReportPlaceholder(
                    $mailBody,
                    $maintenance,
                    [$objectRow['isys_obj__id']]
                );

                // Max. line length:
                $mailer->WordWrap = max(array_map('strlen', explode(PHP_EOL, $mailBody)));

                $mailer
                    ->set_subject($subject)
                    ->set_body($mailBody)
                    ->add_default_signature();

                $mailer->send();

                $this->dao->savePlanning($maintenanceId, ['isys_maintenance__mail_dispatched' => 'NOW()']);
            }
        }

        // @see WARTUNG-90 Some small specific logic for 'combined object' emails.
        if ($combineObjects) {
            $mailBody = $this->replaceReportPlaceholder(
                $mailBody,
                $maintenance,
                $notificationObjects
            );

            // Max. line length:
            $mailer->WordWrap = max(array_map('strlen', explode(PHP_EOL, $mailBody)));

            $mailer
                ->set_subject($subject)
                ->set_body($mailBody)
                ->add_default_signature();

            $mailer->send();

            $this->dao->savePlanning($maintenanceId, ['isys_maintenance__mail_dispatched' => 'NOW()']);
        }

        return true;
    }

    /**
     * @param string $text
     * @param array  $maintenanceData
     * @param int    $objectId
     *
     * @return string
     * @throws \isys_exception_database
     */
    private function replaceSingularPlaceholders(string $text, array $maintenanceData, int $objectId): string
    {
        $text = isys_helper_link::handle_url_variables($text, $objectId);

        // These variables are used additionally.
        foreach ($this->mailContacts as $contact) {
            $contacts[] = $contact['salutation'] . $contact['name'];
        }

        $text = strtr($text, [
            '%recipients%'       => implode(', ', $contacts),
            '%maintenance_from%' => $this->locale->fmt_datetime($maintenanceData['isys_maintenance__date_from']),
            '%maintenance_to%'   => $this->locale->fmt_datetime($maintenanceData['isys_maintenance__date_to'])
        ]);

        return $text;
    }

    /**
     * @param string $text
     * @param array  $maintenanceData
     * @param array  $objectIds
     *
     * @return string
     * @throws \idoit\Exception\JsonException
     * @throws \isys_exception_database
     */
    private function replaceReportPlaceholder(string $text, array $maintenanceData, array $objectIds): string
    {
        if (strpos($text, '%maintenance_report%') === false || !isys_format_json::is_json_array($maintenanceData['isys_maintenance_mailtemplate__report'])) {
            return $text;
        }

        $reportProperties = isys_format_json::decode($maintenanceData['isys_maintenance_mailtemplate__report']);

        $reportQuery = isys_cmdb_dao_category_property::instance($this->db)
            ->reset()
            ->set_query_as_report(true)
            ->setMultivalueStatusFilter([C__RECORD_STATUS__NORMAL])
            ->create_property_query_for_report(null, $reportProperties, true, ' AND obj_main.isys_obj__id ' . $this->dao->prepare_in_condition($objectIds));

        $moduleReport = new isys_module_report_pro();
        $tableContent = $moduleReport->process_show_report($reportQuery, null, true, false, true, false);
        $tableHeader = $moduleReport->get_report_headers();

        array_unshift($tableHeader, 'Link');

        foreach ($tableContent as $i => $row) {
            foreach ($row as $j => $cell) {
                if ($j === '__id__') {
                    // Create link in first cell.
                    $cell = isys_helper_link::create_url([C__CMDB__GET__OBJECT => $cell], true);
                }

                $tableContent[$i][$j] = trim(strip_tags($cell));
            }
        }

        $output = new BufferedOutput();

        (new Table($output))
            ->setHeaders($tableHeader)
            ->setRows($tableContent)
            ->render();

        return str_replace('%maintenance_report%', $output->fetch(), $text);
    }

    /**
     * This method will collect all necessary email addresses, even recursively (if we select persongroups).
     *
     * @param array $contact
     *
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    protected function getEmailRecipients($contact)
    {
        if (!isset($this->internalContacts[$contact['isys_obj__id']])) {
            $this->internalContacts[$contact['isys_obj__id']] = true;

            // Shall the recipients be resolved (if it's a organization or person group).
            $resolveGroups = (isys_tenantsettings::get('maintenance.email.recipients', C__MAINTENANCE__RECIPIENTS__RESOLVE_CONTACT_GROUPS) == C__MAINTENANCE__RECIPIENTS__RESOLVE_CONTACT_GROUPS);

            $emailAddress = isys_cmdb_dao_category_g_mail_addresses::instance($this->db)->get_primary_mail_as_string_by_obj_id($contact['isys_obj__id']);
            $isPerson = $this->dao->objtype_is_cats_assigned($contact['isys_obj__isys_obj_type__id'], C__CATS__PERSON);
            $isOrganization = $this->dao->objtype_is_cats_assigned($contact['isys_obj__isys_obj_type__id'], C__CATS__ORGANIZATION);
            $isPersonGroup = $this->dao->objtype_is_cats_assigned($contact['isys_obj__isys_obj_type__id'], C__CATS__PERSON_GROUP);

            if (!$resolveGroups || $isPerson) {
                $personDao = isys_cmdb_dao_category_s_person_master::instance($this->db);

                $salutationValue = $personDao->get_data(null, $contact['isys_obj__id'])
                    ->get_row_value('isys_cats_person_list__salutation');
                $salutation = $personDao->callback_property_salutation();

                if (!$emailAddress) {
                    return;
                }

                $this->mailContacts[$contact['isys_obj__id']] = [
                    'email'      => $emailAddress,
                    'name'       => $contact['isys_obj__title'],
                    'salutation' => (isset($salutation[$salutationValue]) ? $salutation[$salutationValue] . ' ' : null)
                ];
            } elseif ($isPersonGroup) {

                // First check if the person group has any e-mails
                $personGroupPrimaryMail = isys_cmdb_dao_category_g_mail_addresses::instance($this->db)->get_primary_mail_as_string_by_obj_id($contact['isys_obj__id']);

                if ($personGroupPrimaryMail) {
                    // Send E-amil to Person Group
                    $this->mailContacts[$contact['isys_obj__id']] = [
                        'email' => $personGroupPrimaryMail,
                        'name' => $contact['isys_obj__title'],
                        'salutation' => null
                    ];
                    return;
                }

                // We're dealing with a person group.
                $personResult = isys_cmdb_dao_category_s_person_group_members::instance($this->db)
                    ->get_data(null, $contact['isys_obj__id'], '', null, C__RECORD_STATUS__NORMAL);

                if (count($personResult)) {
                    while ($personRow = $personResult->get_row()) {
                        $this->getEmailRecipients([
                                'isys_obj__id'                => $personRow['person_id'],
                                'isys_obj__title'             => $personRow['person_title'],
                                'isys_obj__isys_obj_type__id' => $personRow['person_type'],
                            ]);
                    }
                }
            } elseif ($isOrganization) {
                // We're dealing with a organization.
                $personResult = isys_cmdb_dao_category_s_organization_person::instance($this->db)
                    ->get_data(null, $contact['isys_obj__id'], '', null, C__RECORD_STATUS__NORMAL);

                if (count($personResult)) {
                    while ($personRow = $personResult->get_row()) {
                        $this->getEmailRecipients($personRow);
                    }
                }
            }
        }
    }
}
