<?php

namespace idoit\Module\Maintenance\Console\Command;

use DateTime;
use Exception;
use idoit\Component\Interval\Config;
use idoit\Component\Interval\Interval;
use idoit\Console\Command\AbstractCommand;
use idoit\Module\Maintenance\Component\Mailer;
use idoit\Module\Maintenance\Model\Maintenance;
use isys_format_json as JSON;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputDefinition;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;

class MaintenanceNotificationCommand extends AbstractCommand
{
    const NAME = 'maintenance';

    /**
     * Get name for command
     *
     * @return string
     */
    public function getCommandName()
    {
        return self::NAME;
    }

    /**
     * Get description for command
     *
     * @return string
     */
    public function getCommandDescription()
    {
        return 'Sends out e-mails for objects with a planned maintenance. As defined in the maintenance add-on';
    }

    /**
     * Retrieve Command InputDefinition
     *
     * @return InputDefinition
     */
    public function getCommandDefinition()
    {
        return new InputDefinition();
    }

    /**
     * Checks if a command can have a config file via --config
     *
     * @return bool
     */
    public function isConfigurable()
    {
        return false;
    }

    /**
     * Returns an array of command usages
     *
     * @return string[]
     */
    public function getCommandUsages()
    {
        return [];
    }

    /**
     * @param InputInterface  $input
     * @param OutputInterface $output
     *
     * @return int
     * @throws \isys_exception_database
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $counter = 0;
        $locale = $this->container->get('locales');
        $database = $this->container->get('database');
        $language = $this->container->get('language');

        $output->writeln("Looking for maintenances, that start today <info>{$locale->fmt_date(time())}</info>...");

        // WARTUNG-3 We additionally fetch all planned maintnances with intervals because they might still be active.
        $basicCondition = 'isys_maintenance__mail_dispatched IS NULL AND DATE(isys_maintenance__date_from) = CURDATE()';
        $intervalCondition = 'TRIM(isys_maintenance__interval) <> ""';

        $l_res = Maintenance::instance($database)->getMaintenances(null, "AND isys_maintenance__finished IS NULL AND (({$basicCondition}) OR ({$intervalCondition}))");

        if (count($l_res)) {
            $mailer = new Mailer($database, $locale);

            while ($maintenanceRow = $l_res->get_row()) {
                try {
                    $resendEmail = (bool)($maintenanceRow['isys_maintenance__resend_emails'] ?? 1);

                    $output->writeln(['', "Found maintenance <info>#{$maintenanceRow['isys_maintenance__id']}</info>..."], OutputInterface::VERBOSITY_VERBOSE);

                    if (!$resendEmail && $maintenanceRow['isys_maintenance__mail_dispatched'] !== null) {
                        $output->writeln('> Notification was already sent - resending was switched off. Skip.', OutputInterface::VERBOSITY_VERBOSE);
                        continue;
                    }

                    if (JSON::is_json_array($maintenanceRow['isys_maintenance__interval'])) {
                        $configuration = Config::byJSON($maintenanceRow['isys_maintenance__interval']);
                        $interval = new Interval($configuration, new DateTime());

                        if (!$interval->isIntervalReached()) {
                            // Skip if the interval is not reached today.
                            $output->writeln('> The interval is not reached today. Skip.', OutputInterface::VERBOSITY_VERBOSE);
                            continue;
                        }
                    }

                    $mailer->sendMaintenancePlanningMail($maintenanceRow['isys_maintenance__id']);

                    $output->writeln("> Contacts of maintenance <info>#{$maintenanceRow['isys_maintenance__id']}</info> have been notified!", OutputInterface::VERBOSITY_VERBOSE);

                    if ($output->getVerbosity() >= OutputInterface::VERBOSITY_VERY_VERBOSE) {
                        $objects = $mailer->getAffectedObjects();
                        $recipients = $mailer->getCollectedRecipients();

                        $output->writeln('> List of objects');

                        foreach ($objects as $object) {
                            $objectString = $language->get($object['isys_obj_type__title']) . ' >> ' . $object['isys_obj__title'];
                            $output->writeln("  - <comment>'{$objectString}'</comment> (<info>#{$object['isys_obj__id']}</info>)");
                        }

                        $output->writeln([
                            '',
                            '> List of recipients'
                        ]);

                        foreach ($recipients as $recipient) {
                            $output->writeln("  - <comment>'{$recipient['name']}'</comment> <<info>{$recipient['email']}</info>>");
                        }
                    }

                    $counter++;
                } catch (Exception $e) {
                    $output->writeln("<error>Maintenance #{$maintenanceRow['isys_maintenance__id']} could not be notified, an error occured: {$e->getMessage()}</error>");
                }
            }

            $output->writeln("Processed <info>{$counter} maintenance</info> entries.", OutputInterface::VERBOSITY_VERBOSE);
            $output->writeln('Finished!');
        } else {
            $output->writeln('No e-mails need to be sent!');
        }

        return Command::SUCCESS;
    }
}
