<?php

namespace idoit\Module\Maintenance\Console\Command;

use Carbon\Carbon;
use Exception;
use idoit\Console\Command\AbstractCommand;
use idoit\Module\Maintenance\Component\Mailer;
use idoit\Module\Maintenance\Model\Maintenance;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputDefinition;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

class MaintenanceNotificationCommand extends AbstractCommand
{
    const NAME = 'maintenance:send-notifications';

    public function getCommandAliases()
    {
        return [
            'maintenance'
        ];
    }

    /**
     * Get name for command
     *
     * @return string
     */
    public function getCommandName()
    {
        return self::NAME;
    }

    /**
     * Get description for command
     *
     * @return string
     */
    public function getCommandDescription()
    {
        return 'Sends out e-mails for objects with a planned maintenance. As defined in the maintenance add-on';
    }

    /**
     * Retrieve Command InputDefinition
     *
     * @return InputDefinition
     */
    public function getCommandDefinition()
    {
        return new InputDefinition([
            new InputOption('consider-past-maintenances', null, InputOption::VALUE_NONE, 'Send e-mails for unfinished maintenances set in the past.')
        ]);
    }

    /**
     * Checks if a command can have a config file via --config
     *
     * @return bool
     */
    public function isConfigurable()
    {
        return false;
    }

    /**
     * Returns an array of command usages
     *
     * @return string[]
     */
    public function getCommandUsages()
    {
        return [];
    }

    /**
     * @param InputInterface  $input
     * @param OutputInterface $output
     *
     * @return int
     * @throws \isys_exception_database
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $counter = 0;
        $locale = $this->container->get('locales');
        $database = $this->container->get('database');
        $language = $this->container->get('language');

        $considerPast = $input->getOption('consider-past-maintenances');

        $output->writeln("Looking for maintenances, that are not <info>finished</info>.");

        $result = Maintenance::instance($database)->getMaintenances(null, "AND isys_maintenance__finished IS NULL");
        $mailer = new Mailer($database, $locale);

        while ($maintenanceRow = $result->get_row()) {
            try {
                $resendEmail = (bool)($maintenanceRow['isys_maintenance__resend_emails'] ?? 1);
                $startDate = Carbon::parse($maintenanceRow['isys_maintenance__date_from'])->startOfDay();
                $startDateString = $startDate->format('Y-m-d');
                $endDate = Carbon::parse($maintenanceRow['isys_maintenance__date_to'])->endOfDay();
                $endDateString = $endDate->format('Y-m-d');

                $fromToString = "<comment>{$startDateString}</comment>";

                if ($startDateString !== $endDateString) {
                    $fromToString .= " - <comment>{$endDateString}</comment>";
                }

                $output->writeln([
                    '',
                    "Found maintenance <info>#{$maintenanceRow['isys_maintenance__id']}</info> ({$fromToString})..."], OutputInterface::VERBOSITY_VERBOSE);

                // @see WARTUNG-103 Skip if the maintenance is set in the future.
                if ($startDate->isFuture()) {
                    $output->writeln("> The maintenance will start in the future. <info>Skip</info>.", OutputInterface::VERBOSITY_VERY_VERBOSE);
                    continue;
                }

                if ($maintenanceRow['isys_maintenance__mail_dispatched'] !== null) {
                    // @see WARTUNG-103 Skip if the notification was sent.
                    if (!$resendEmail) {
                        $output->writeln('> Notification was already sent - resending is switched <comment>off</comment>. <info>Skip</info>.', OutputInterface::VERBOSITY_VERBOSE);
                        continue;
                    }

                    $output->writeln('> Notification was already sent - resending is switched <comment>on</comment>.', OutputInterface::VERBOSITY_VERBOSE);
                }

                // @see WARTUNG-103 Skip if the maintenance was set in the past (unless the option was passed).
                if ($endDate->isPast() && !$considerPast) {
                    $output->writeln("> The maintenance already ended in the past. <info>Skip</info>.", OutputInterface::VERBOSITY_VERY_VERBOSE);
                    $output->writeln('  In order to still send a notification e-mail, pass <info>--consider-past-maintenances</info> option.', OutputInterface::VERBOSITY_DEBUG);
                    continue;
                }

                $mailer->sendMaintenancePlanningMail($maintenanceRow['isys_maintenance__id']);

                $output->writeln("> Contacts of maintenance <info>#{$maintenanceRow['isys_maintenance__id']}</info> have been notified!", OutputInterface::VERBOSITY_VERBOSE);

                if ($output->getVerbosity() >= OutputInterface::VERBOSITY_VERY_VERBOSE) {
                    $objects = $mailer->getAffectedObjects();
                    $recipients = $mailer->getCollectedRecipients();

                    $output->writeln('> List of objects');

                    foreach ($objects as $object) {
                        $objectString = $language->get($object['isys_obj_type__title']) . ' >> ' . $object['isys_obj__title'];
                        $output->writeln("  - <comment>'{$objectString}'</comment> (<info>#{$object['isys_obj__id']}</info>)");
                    }

                    $output->writeln([
                        '> List of recipients'
                    ]);

                    foreach ($recipients as $recipient) {
                        $output->writeln("  - <comment>'{$recipient['name']}'</comment> <<info>{$recipient['email']}</info>>");
                    }
                }

                $counter++;
            } catch (Exception $e) {
                $output->writeln("<error>Maintenance #{$maintenanceRow['isys_maintenance__id']} could not be notified, an error occured: {$e->getMessage()}</error>");
            }
        }

        if ($counter > 0) {
            $output->writeln(['', "Processed <info>{$counter} maintenance</info> entries.", 'Finished!'], OutputInterface::VERBOSITY_VERBOSE);
        } else {
            $output->writeln(['', 'No e-mails sent!', 'Finished!']);
        }

        return Command::SUCCESS;
    }
}
