<?php

namespace idoit\Module\Maintenance\Model;

use Exception;
use isys_cmdb_dao;
use isys_component_dao_result;
use isys_exception_dao;
use isys_exception_database;

/**
 * i-doit
 *
 * Maintenance model for "mailtemplate".
 *
 * @package     idoit\Module\Maintenance\Model
 * @package     maintenance
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @version     1.0
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       Maintenance 1.1
 */
class Maintenance extends isys_cmdb_dao
{
    /**
     * @var  array
     */
    const FIELDS = [
        'isys_maintenance__id'                                => 'convert_sql_id',
        'isys_maintenance__isys_maintenance_type__id'         => 'convert_sql_id',
        'isys_maintenance__date_from'                         => 'convert_sql_datetime',
        'isys_maintenance__date_to'                           => 'convert_sql_datetime',
        'isys_maintenance__comment'                           => 'convert_sql_text',
        'isys_maintenance__finished'                          => 'convert_sql_datetime',
        'isys_maintenance__finish_comment'                    => 'convert_sql_text',
        'isys_maintenance__finish_person'                     => 'convert_sql_text',
        'isys_maintenance__mail_dispatched'                   => 'convert_sql_datetime',
        'isys_maintenance__isys_maintenance_mailtemplate__id' => 'convert_sql_id',
        'isys_maintenance__status'                            => 'convert_sql_int',
        'isys_maintenance__interval'                          => 'convert_sql_text',
        'isys_maintenance__resend_emails'                     => 'convert_sql_int',
    ];

    /**
     * Method for retrieving maintenance data.
     *
     * @param   mixed  $id
     * @param   string $condition
     *
     * @return  isys_component_dao_result
     * @throws  Exception
     * @throws  isys_exception_database
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function getMaintenances($id = null, $condition = '')
    {
        $sql = 'SELECT * FROM isys_maintenance WHERE TRUE ' . $condition;

        if ($id !== null) {
            if (is_array($id)) {
                $sql .= 'AND isys_maintenance__id ' . $this->prepare_in_condition($id) . ' ';
            } else {
                $sql .= 'AND isys_maintenance__id  = ' . $this->convert_sql_id($id) . ' ';
            }
        }

        return $this->retrieve($sql . ';');
    }

    /**
     * Method for saving a maintenance.
     *
     * @param   integer $id
     * @param   array   $data
     *
     * @return  integer
     * @throws  isys_exception_dao
     * @throws  isys_exception_database
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function saveMaintenance($id = null, array $data = [])
    {
        $queryData = [];

        foreach ($data as $key => $value) {
            if (isset(self::FIELDS[$key])) {
                $queryData[] = $key . ' = ' . call_user_func([$this, self::FIELDS[$key]], $value);
            }
        }

        if (count($queryData)) {
            if ($id === null) {
                $sql = 'INSERT INTO isys_maintenance SET ' . implode(', ', $queryData);
            } else {
                $sql = 'UPDATE isys_maintenance SET ' . implode(', ', $queryData) . ' WHERE isys_maintenance__id = ' . $this->convert_sql_id($id);
            }

            if ($this->update($sql) && $this->apply_update()) {
                if ($id === null) {
                    $id = $this->get_last_id_from_table('isys_maintenance');
                }
            } else {
                throw new isys_exception_database($this->m_last_error);
            }
        }

        return $id;
    }

    /**
     * Method for duplicating a maintenance.
     *
     * @param  integer $id
     *
     * @return integer
     * @throws Exception
     * @throws isys_exception_dao
     * @throws isys_exception_database
     */
    public function duplicateMaintenance($id)
    {
        $data = $this->getMaintenances($id)->get_row();

        unset($data['isys_maintenance__id'], $data['isys_maintenance__finished'], $data['isys_maintenance__finish_comment'], $data['isys_maintenance__finish_person'], $data['isys_maintenance__mail_dispatched']);

        // Save the maintenance as a new entry!
        $duplicateId = $this->saveMaintenance(null, $data);

        if (!($duplicateId > 0)) {
            throw new isys_exception_dao('The maintenance #' . $id . ' could not be duplicated!');
        }

        // Prepare connected object entries.
        $objectQueries = [];
        $objectResult = $this->getMaintenanceObjects($id);

        while ($objectRow = $objectResult->get_row()) {
            $objectQueries[] = '(' . $this->convert_sql_id($duplicateId) . ', ' . $this->convert_sql_id($objectRow['isys_maintenance_2_object__isys_obj__id']) . ')';
        }

        if (count($objectQueries)) {
            // Now we add all the connected objects.
            $this->update('INSERT INTO isys_maintenance_2_object (isys_maintenance_2_object__isys_maintenance__id, isys_maintenance_2_object__isys_obj__id) VALUES ' . implode(',', $objectQueries) . ';');
            $this->apply_update();
        }

        // Prepare connected contact entries.
        $contactQueries = [];
        $contactResult = $this->getMaintenanceContacts($id);

        while ($contactRow = $contactResult->get_row()) {
            $contactQueries[] = '(' . $this->convert_sql_id($duplicateId) . ', ' . $this->convert_sql_id($contactRow['isys_maintenance_2_contact__isys_obj__id']) . ')';
        }

        if (count($contactQueries)) {
            // Now we add all the connected objects.
            $this->update('INSERT INTO isys_maintenance_2_contact (isys_maintenance_2_contact__isys_maintenance__id, isys_maintenance_2_contact__isys_obj__id) VALUES ' . implode(',', $contactQueries) . ';');
            $this->apply_update();
        }

        // Prepare connected role entries
        $idPairs = [];
        $roleResult = $this->getMaintenanceRoles($id);

        while ($roleRow = $roleResult->get_row()) {
            $idPairs[] = '(' . $this->convert_sql_id($duplicateId) . ', ' . $this->convert_sql_id($roleRow['isys_contact_tag__id']) . ')';
        }

        if (count($idPairs)) {
            // Now we add all the connected roles.
            $this->update('INSERT INTO isys_maintenance_2_role (isys_maintenance__id, isys_contact_tag__id) VALUES ' . implode(',', $idPairs) . ';');
            $this->apply_update();
        }

        return $duplicateId;
    }

    /**
     * Method for retrieving all objects for a certain maintenance plan.
     *
     * @param   integer $id
     *
     * @return  isys_component_dao_result
     * @throws  Exception
     * @throws  isys_exception_database
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function getMaintenanceObjects($id)
    {
        $sql = 'SELECT * FROM isys_obj
			LEFT JOIN isys_obj_type ON isys_obj_type__id = isys_obj__isys_obj_type__id
			LEFT JOIN isys_maintenance_2_object ON isys_maintenance_2_object__isys_obj__id = isys_obj__id
			WHERE isys_maintenance_2_object__isys_maintenance__id = ' . $this->convert_sql_id($id) . ';';

        return $this->retrieve($sql);
    }

    /**
     * Method for retrieving all contacts for a certain maintenance plan.
     *
     * @param   integer $id
     *
     * @return  isys_component_dao_result
     * @throws  Exception
     * @throws  isys_exception_database
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function getMaintenanceContacts($id)
    {
        $sql = 'SELECT * FROM isys_obj
			LEFT JOIN isys_obj_type ON isys_obj_type__id = isys_obj__isys_obj_type__id
			LEFT JOIN isys_maintenance_2_contact ON isys_maintenance_2_contact__isys_obj__id = isys_obj__id
			WHERE isys_maintenance_2_contact__isys_maintenance__id = ' . $this->convert_sql_id($id) . ';';

        return $this->retrieve($sql);
    }

    /**
     * Method for deleting maintenance(s).
     *
     * @param   int|array $id
     *
     * @return  boolean
     * @throws isys_exception_dao
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function deleteMaintenance($id)
    {
        if (!is_array($id)) {
            $id = [$id];
        }

        $sql = 'DELETE FROM isys_maintenance WHERE isys_maintenance__id ' . $this->prepare_in_condition($id) . ';';

        return ($this->update($sql) && $this->apply_update());
    }

    /**
     * @param $id
     *
     * @return isys_component_dao_result
     * @throws isys_exception_database
     */
    public function getMaintenanceRoles($id)
    {
        $sql = 'SELECT * FROM isys_contact_tag ct
			LEFT JOIN isys_maintenance_2_role m2r ON m2r.isys_contact_tag__id = ct.isys_contact_tag__id
			WHERE m2r.isys_maintenance__id = ' . $this->convert_sql_id($id) . ';';

        return $this->retrieve($sql);
    }
}
