<?php

namespace idoit\Module\Maintenance\Model;

use DateTime;
use isys_application;
use isys_cmdb_dao;
use isys_cmdb_dao_dialog;
use isys_component_database;
use isys_component_template_language_manager;
use isys_locale;
use isys_maintenance_dao as MaintenanceDao;

/**
 * i-doit
 *
 * Maintenance model for "overview" page.
 *
 * @package     idoit\Module\Maintenance\Model
 * @package     maintenance
 * @version     1.0
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       Maintenance 1.1
 */
class Overview extends isys_cmdb_dao
{
    private isys_locale $locale;

    private isys_component_template_language_manager $lang;

    private MaintenanceDao $maintenanceDao;

    private array $maintenances = [];

    private array $objects = [];

    /**
     * Overview constructor.
     *
     * @param isys_component_database $p_db
     *
     * @throws \Exception
     */
    public function __construct(isys_component_database $p_db)
    {
        parent::__construct($p_db);

        $this->locale = isys_application::instance()->container->get('locales');
        $this->lang = isys_application::instance()->container->get('language');
        $this->maintenanceDao = MaintenanceDao::instance($this->m_db);
    }

    /**
     * @param DateTime  $from
     * @param DateTime  $to
     * @param int|array $typeId
     *
     * @return array
     * @throws \Exception
     * @throws \isys_exception_database
     */
    public function getDataByTimespan(DateTime $from, DateTime $to, $typeId = null): array
    {
        $routeGenerator = isys_application::instance()->container->get('route_generator');
        $maintenances = $this->maintenanceDao->getMaintenancesBetween($from, $to, $typeId);

        $roles = isys_cmdb_dao_dialog::instance($this->m_db)
            ->reset()
            ->set_table('isys_contact_tag')
            ->get_data();

        foreach ($maintenances as $maintenance) {
            $maintenance['assignedRoles'] = (string)$maintenance['assignedRoles'];
            $translatedRoles = $assignedRoles = [];

            if ($maintenance['assignedRoles'] !== '') {
                $maintenance['assignedRoles'] = explode(',', $maintenance['assignedRoles']);
                foreach ($maintenance['assignedRoles'] as $roleId) {
                    $assignedRoles[$roleId] = $roles[$roleId];
                }
                $translatedRoles = array_values(array_map(fn($roleItem) => $this->lang->get($roleItem['isys_contact_tag__title']), $assignedRoles));
            }

            if (!isset($this->maintenances[$maintenance['isys_maintenance__id']])) {
                $tupel = $maintenance;
                $tupel['isys_contact_tag__title'] = $translatedRoles;
                $tupel['objects'] = [];
                $tupel['contacts'] = [];

                unset(
                    $tupel['isys_maintenance__interval'],
                    $tupel['isys_maintenance__isys_maintenance_mailtemplate__id'],
                    $tupel['isys_maintenance__mail_dispatched'],
                    $tupel['isys_maintenance__date_from'],
                    $tupel['isys_maintenance__date_to'],
                    $tupel['isys_maintenance__status']
                );

                $l_contact_res = Maintenance::instance($this->m_db)
                    ->getMaintenanceContacts($tupel['isys_maintenance__id']);

                if (count($l_contact_res)) {
                    while ($l_contact_row = $l_contact_res->get_row()) {
                        $tupel['contacts'][] = [
                            'isys_obj__id'         => $l_contact_row['isys_obj__id'],
                            'isys_obj__title'      => $l_contact_row['isys_obj__title'],
                            'isys_obj_type__title' => $this->lang->get($l_contact_row['isys_obj_type__title']),
                            'isys_obj_type__icon'  => $routeGenerator->generate('cmdb.object-type.image', ['objectTypeId' => $l_contact_row['isys_obj_type__id']])
                        ];
                    }
                }

                $this->maintenances[$tupel['isys_maintenance__id']] = $tupel;

                // Get the objects, connected with this maintenance.
                $objectResult = Maintenance::instance($this->m_db)
                    ->getMaintenanceObjects($tupel['isys_maintenance__id']);

                while ($objectRow = $objectResult->get_row()) {
                    $this->maintenances[$maintenance['isys_maintenance__id']]['objects'][] = $objectRow['isys_obj__id'];

                    if (!isset($this->objects[$objectRow['isys_obj__id']])) {
                        $objectRow['maintenances'] = [];
                        $objectRow['roles'] = [];
                        $objectRow['isys_obj_type__title'] = $this->lang->get($objectRow['isys_obj_type__title']);
                        $objectRow['image'] = $routeGenerator->generate('cmdb.object.image', ['objectId' => $objectRow['isys_obj__id']]);
                        $objectRow['isys_obj_type__icon'] = $routeGenerator->generate('cmdb.object-type.image', ['objectTypeId' => $objectRow['isys_obj_type__id']]);

                        $l_names = [];
                        $l_contact_res = Contact::instance($this->m_db)->getByObjectAndTag($objectRow['isys_obj__id'], $maintenance['assignedRoles']);

                        if (count($l_contact_res)) {
                            while ($l_contact_row = $l_contact_res->get_row()) {
                                $l_names[] = $this->lang->get($this->maintenanceDao->get_obj_type_name_by_obj_id($l_contact_row['isys_obj__id'])) . ' > ' .
                                    $l_contact_row['isys_obj__title'] .
                                    (empty($l_contact_row['isys_contact_tag__title']) ? '' : ' (' . $this->lang->get($l_contact_row['isys_contact_tag__title']) . ')');
                            }
                        }

                        $l_names = array_filter($l_names);

                        if (count($l_names)) {
                            $objectRow['roles'][$maintenance['isys_contact_tag__id']] = $l_names;
                        }

                        $this->objects[$objectRow['isys_obj__id']] = $objectRow;
                    }
                }
            }

            $this->maintenances[$maintenance['isys_maintenance__id']]['dates'][] = [
                'from'          => strtotime($maintenance['isys_maintenance__date_from']),
                'fromFormatted' => $this->locale->fmt_datetime($maintenance['isys_maintenance__date_from']),
                'to'            => strtotime($maintenance['isys_maintenance__date_to']),
                'toFormatted'   => $this->locale->fmt_datetime($maintenance['isys_maintenance__date_to']),
            ];
        }

        return [
            'objects'      => $this->objects,
            'maintenances' => $this->maintenances
        ];
    }
}
