<?php

use idoit\AddOn\ActivatableInterface;
use idoit\AddOn\AuthableInterface;
use idoit\AddOn\InstallableInterface;

/**
 * i-doit
 *
 * Nagios Settings.
 *
 * @package     i-doit
 * @subpackage  Modules
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class isys_module_nagios extends isys_module implements AuthableInterface, ActivatableInterface, InstallableInterface, isys_module_authable
{
    const DISPLAY_IN_MAIN_MENU = true;
    const DISPLAY_IN_SYSTEM_MENU = true;

    /**
     * Static configuration array.
     */
    protected static $m_config = [];

    /**
     * Attach live status of object to contenttop header
     *
     * @param  $p_catdata
     */
    public static function process_content_top($p_catdata)
    {
        $database = isys_application::instance()->container->get('database');

        if (defined('C__MONITORING__TYPE_NDO') && count(isys_monitoring_dao_hosts::instance($database)->get_data(null, C__MONITORING__TYPE_NDO))) {
            global $index_includes;

            if (!is_array($index_includes['contenttopobjectdetail'])) {
                $index_includes['contenttopobjectdetail'] = (array)$index_includes['contenttopobjectdetail'];
            }

            $index_includes['contenttopobjectdetail'][] = self::getPath() . 'templates/contenttop/main_objectdetail_ndo_status.tpl';
        }
    }

    /**
     * @param int $p_obj_id
     *
     * @return string
     * @throws Exception
     */
    protected static function get_correct_contact_name($p_obj_id)
    {
        $database = isys_application::instance()->container->get('database');

        try {
            if (self::$m_config['PERSON_NAME_OPTION'] == C__NAGIOS__PERSON_OPTION__USERNAME) {
                return isys_cmdb_dao_category_s_person_login::instance($database)
                    ->get_data(null, $p_obj_id)
                    ->get_row_value('isys_cats_person_list__title');
            } else {
                return isys_cmdb_dao::instance($database)
                    ->get_obj_name_by_id_as_string($p_obj_id);
            }
        } catch (Exception $e) {
            return 'ERROR: ' . $e->getMessage();
        }
    }

    /**
     * Method for initialization.
     *
     * @param   isys_module_request $p_req
     *
     * @return  isys_module_nagios
     */
    public function init(isys_module_request $p_req)
    {
        return $this;
    }

    /**
     * This method builds the tree for the menu.
     *
     * @param   isys_component_tree $p_tree
     * @param   boolean             $p_system_module
     * @param   integer             $p_parent
     *
     * @author  Leonard Fischer <lfischer@i-doit.org>
     * @since   0.9.9-7
     * @see     isys_module::build_tree()
     */
    public function build_tree(isys_component_tree $p_tree, $p_system_module = true, $p_parent = null)
    {
        global $g_dirs;

        if ($p_system_module) {
            return;
        }

        $auth = self::getAuth();

        $p_parent = $p_tree->add_node(C__MODULE__NAGIOS . '0', -1, 'Nagios', '', '', '', 0, '', '', $auth->has_any_rights_in_module());

        $p_tree->add_node(C__MODULE__NAGIOS . '3', $p_parent, 'Config', isys_helper_link::create_url([
            C__GET__MODULE_ID                => $_GET[C__GET__MODULE_ID],
            C__GET__TREE_NODE                => C__MODULE__NAGIOS . '3',
            C__GET__NAGIOS_PAGE              => C__NAGIOS_PAGE__CONFIG,
            C__GET__MAIN_MENU__NAVIGATION_ID => $_GET[C__GET__MAIN_MENU__NAVIGATION_ID]
        ]), '', $g_dirs['images'] . 'axialis/basic/tool-wrench.svg', 0, '', '', $auth->is_allowed_to(isys_auth::SUPERVISOR, 'NAGIOS_CONFIG'));

        $p_tree->add_node(C__MODULE__NAGIOS . '4', $p_parent, 'Timeperiods', isys_helper_link::create_url([
            C__GET__MODULE_ID                => $_GET[C__GET__MODULE_ID],
            C__GET__TREE_NODE                => C__MODULE__NAGIOS . '4',
            C__GET__NAGIOS_PAGE              => C__NAGIOS_PAGE__TIMEPERIODS,
            C__GET__MAIN_MENU__NAVIGATION_ID => $_GET[C__GET__MAIN_MENU__NAVIGATION_ID]
        ]), '', $g_dirs['images'] . 'axialis/basic/clock.svg', 0, '', '', $auth->is_allowed_to(isys_auth::SUPERVISOR, 'NAGIOS_TIMEPERIODS'));

        $p_tree->add_node(C__MODULE__NAGIOS . '5', $p_parent, 'Commands', isys_helper_link::create_url([
            C__GET__MODULE_ID                => $_GET[C__GET__MODULE_ID],
            C__GET__TREE_NODE                => C__MODULE__NAGIOS . '5',
            C__GET__NAGIOS_PAGE              => C__NAGIOS_PAGE__COMMANDS,
            C__GET__MAIN_MENU__NAVIGATION_ID => $_GET[C__GET__MAIN_MENU__NAVIGATION_ID]
        ]), '', $g_dirs['images'] . 'axialis/database/data.svg', 0, '', '', $auth->is_allowed_to(isys_auth::SUPERVISOR, 'NAGIOS_COMMANDS'));

        $p_tree->add_node(C__MODULE__NAGIOS . '6', $p_parent, 'Host escalations', isys_helper_link::create_url([
            C__GET__MODULE_ID                => $_GET[C__GET__MODULE_ID],
            C__GET__TREE_NODE                => C__MODULE__NAGIOS . '6',
            C__GET__NAGIOS_PAGE              => C__NAGIOS_PAGE__HOST_ESCALATIONS,
            C__GET__MAIN_MENU__NAVIGATION_ID => $_GET[C__GET__MAIN_MENU__NAVIGATION_ID]
        ]), '', $g_dirs['images'] . 'axialis/database/data.svg', 0, '', '', $auth->is_allowed_to(isys_auth::SUPERVISOR, 'NAGIOS_HOST_ESCALATIONS'));

        $p_tree->add_node(C__MODULE__NAGIOS . '7', $p_parent, 'Service escalations', isys_helper_link::create_url([
            C__GET__MODULE_ID                => $_GET[C__GET__MODULE_ID],
            C__GET__TREE_NODE                => C__MODULE__NAGIOS . '7',
            C__GET__NAGIOS_PAGE              => C__NAGIOS_PAGE__SERVICE_ESCALATIONS,
            C__GET__MAIN_MENU__NAVIGATION_ID => $_GET[C__GET__MAIN_MENU__NAVIGATION_ID]
        ]), '', $g_dirs['images'] . 'axialis/database/data.svg', 0, '', '', $auth->is_allowed_to(isys_auth::SUPERVISOR, 'NAGIOS_SERVICE_ESCALATIONS'));

        $p_tree->add_node(
            C__MODULE__NAGIOS . '9',
            $p_parent,
            $this->language->get('LC__OBJTYPE__NAGIOS_HOST_TPL'),
            "javascript:tree_obj_type_click('" . C__OBJTYPE__NAGIOS_HOST_TPL . "');",
            '',
            $g_dirs['images'] . 'axialis/database/database.svg',
            0,
            '',
            '',
            isys_auth_cmdb::instance()
                ->is_allowed_to(isys_auth::SUPERVISOR, 'OBJ_IN_TYPE/C__OBJTYPE__NAGIOS_HOST_TPL')
        );

        $p_tree->add_node(
            C__MODULE__NAGIOS . '10',
            $p_parent,
            $this->language->get('LC__OBJTYPE__NAGIOS_SERVICE_TPL'),
            "javascript:tree_obj_type_click('" . C__OBJTYPE__NAGIOS_SERVICE_TPL . "');",
            '',
            $g_dirs['images'] . 'axialis/database/database.svg',
            0,
            '',
            '',
            isys_auth_cmdb::instance()
                ->is_allowed_to(isys_auth::SUPERVISOR, 'OBJ_IN_TYPE/C__OBJTYPE__NAGIOS_SERVICE_TPL')
        );

        $p_tree->add_node(
            C__MODULE__NAGIOS . '11',
            $p_parent,
            $this->language->get('LC__OBJTYPE__NAGIOS_SERVICE'),
            "javascript:tree_obj_type_click('" . C__OBJTYPE__NAGIOS_SERVICE . "');",
            '',
            $g_dirs['images'] . 'axialis/database/database.svg',
            0,
            '',
            '',
            isys_auth_cmdb::instance()
                ->is_allowed_to(isys_auth::SUPERVISOR, 'OBJ_IN_TYPE/C__OBJTYPE__NAGIOS_SERVICE')
        );

        $p_tree->add_node(C__MODULE__NAGIOS . '12', $p_parent, 'Nagios Export', isys_helper_link::create_url([
            C__GET__MODULE_ID                => $_GET[C__GET__MODULE_ID],
            C__GET__TREE_NODE                => C__MODULE__NAGIOS . '12',
            C__GET__NAGIOS_PAGE              => C__NAGIOS_PAGE__EXPORT,
            C__GET__MAIN_MENU__NAVIGATION_ID => $_GET[C__GET__MAIN_MENU__NAVIGATION_ID]
        ]), '', $g_dirs['images'] . 'icons/nagios.png', 0, '', '', $auth->is_allowed_to(isys_auth::EXECUTE, 'NAGIOS_EXPORT'));
    }

    /**
     * Start module Nagios.
     *
     * @author Dennis Bluemer <dbluemer@synetics.de>
     * @return void
     */
    public function start()
    {
        if (isys_glob_get_param('ajax') && !isys_glob_get_param('call')) {
            $this->processAjaxRequest();
            die;
        }

        $l_gets = isys_module_request::get_instance()->get_gets();
        $l_posts = isys_module_request::get_instance()->get_posts();
        $auth = self::getAuth();

        // Build the menutree.
        if ($_GET[C__GET__MODULE_ID] != C__MODULE__SYSTEM) {
            $l_tree = isys_module_request::get_instance()
                ->get_menutree();

            $this->build_tree($l_tree, false);
            isys_application::instance()->container->get('template')
                ->assign('menu_tree', $l_tree->process($_GET[C__GET__TREE_NODE]));
        }

        switch ($l_posts[C__GET__NAVMODE]) {
            case C__NAVMODE__NEW:
            case C__NAVMODE__EDIT:
                isys_component_template_navbar::getInstance()
                    ->set_active(true, C__NAVBAR_BUTTON__SAVE)
                    ->set_active(true, C__NAVBAR_BUTTON__CANCEL);
                break;
        }

        switch ($l_gets[C__GET__NAGIOS_PAGE]) {
            case C__NAGIOS_PAGE__CONFIG:
                $auth->check(isys_auth::VIEW, 'NAGIOS_CONFIG');
                switch ($l_posts[C__GET__NAVMODE]) {
                    case C__NAVMODE__SAVE:
                        $this->saveConfig();
                        break;

                    default:
                        $this->processConfigPage();
                }
                break;

            case C__NAGIOS_PAGE__TIMEPERIODS:
                $auth->check(isys_auth::VIEW, 'NAGIOS_TIMEPERIODS');
                switch ($l_posts[C__GET__NAVMODE]) {
                    case C__NAVMODE__NEW:
                        $this->createTimeperiod();
                        break;

                    case C__NAVMODE__SAVE:
                        $this->saveTimeperiod();
                        break;

                    case C__NAVMODE__DELETE:
                        $this->deleteTimeperiod();
                        break;

                    default:
                        if (isset($l_gets[C__GET__NAGIOS_TPID])) {
                            $this->processTimeperiodsPage($l_gets[C__GET__NAGIOS_TPID]);
                        } else {
                            if (is_array($_POST["id"])) {
                                $this->processTimeperiodsPage($_POST["id"][0]);
                            } else {
                                $this->processTimeperiodsListing();
                            }
                        }
                }
                break;

            case C__NAGIOS_PAGE__COMMANDS:
                $auth->check(isys_auth::VIEW, 'NAGIOS_COMMANDS');
                switch ($l_posts[C__GET__NAVMODE]) {
                    case C__NAVMODE__NEW:
                        $this->createCommand();
                        break;

                    case C__NAVMODE__SAVE:
                        $this->saveCommand();
                        break;

                    case C__NAVMODE__DELETE:
                        $this->deleteCommand();
                        break;

                    default:
                        if (isset($l_gets[C__GET__NAGIOS_CID])) {
                            $this->processCommandsPage($l_gets[C__GET__NAGIOS_CID]);
                        } else {
                            $onclickJsText = "var checkedRows = $$('tr.tr-selected'); if(checkedRows.length === 0){return;}; 
                                document.isys_form.action = checkedRows[0].getAttribute('data-link');  
                                document.isys_form.sort.value=''; document.isys_form.navMode.value='" . C__NAVMODE__EDIT . "'; form_submit();";
                            isys_component_template_navbar::getInstance()
                                ->set_js_onclick($onclickJsText, C__NAVBAR_BUTTON__EDIT);
                            if (is_array($_POST["id"])) {
                                $this->processCommandsPage($_POST["id"][0]);
                            } else {
                                $this->processCommandsListing();
                            }
                        }
                }
                break;

            case C__NAGIOS_PAGE__HOST_ESCALATIONS:
                $auth->check(isys_auth::VIEW, 'NAGIOS_HOST_ESCALATIONS');
                switch ($l_posts[C__GET__NAVMODE]) {
                    case C__NAVMODE__NEW:
                        $this->createHostEscalation();
                        break;

                    case C__NAVMODE__SAVE:
                        $this->saveHostEscalation();
                        break;

                    case C__NAVMODE__DELETE:
                        $this->deleteHostEscalation();
                        break;

                    default:
                        if (isset($l_gets[C__GET__NAGIOS_EID])) {
                            $this->processHostEscalationsPage($l_gets[C__GET__NAGIOS_EID]);
                        } elseif (is_array($_POST["id"])) {
                            $this->processHostEscalationsPage($_POST["id"][0]);
                        } else {
                            $this->processHostEscalationsListing();
                        }
                }
                break;

            case C__NAGIOS_PAGE__SERVICE_ESCALATIONS:
                $auth->check(isys_auth::VIEW, 'NAGIOS_SERVICE_ESCALATIONS');
                switch ($l_posts[C__GET__NAVMODE]) {
                    case C__NAVMODE__NEW:
                        $this->createServiceEscalation();
                        break;

                    case C__NAVMODE__SAVE:
                        $this->saveServiceEscalation();
                        break;

                    case C__NAVMODE__DELETE:
                        $this->deleteServiceEscalation();
                        break;

                    default:
                        if (isset($l_gets[C__GET__NAGIOS_EID])) {
                            $this->processServiceEscalationsPage($l_gets[C__GET__NAGIOS_EID]);
                        } else {
                            if (is_array($_POST["id"])) {
                                $this->processServiceEscalationsPage($_POST["id"][0]);
                            } else {
                                $this->processServiceEscalationsListing();
                            }
                        }
                }
                break;

            case C__NAGIOS_PAGE__EXPORT:
                $auth->check(isys_auth::EXECUTE, 'NAGIOS_EXPORT');
                $this->processExportPage();
                break;

            default:
                // Do nothing to display the dashboard.
        }
    }

    /**
     * @throws  isys_exception_dao
     */
    public function saveHostEscalation()
    {
        $l_daoNagios = new isys_component_dao_nagios(isys_application::instance()->container->get('database'));
        $l_optArr = [];
        $l_id = null;

        try {
            if (!isset($_POST['id']) || empty($_POST['id'])) {
                $l_id = $l_daoNagios->createHostEscalation();
            } else {
                $l_id = $_POST['id'];
            }

            $l_daoNagios->saveHostEscalation($l_id);
            $l_contactID = $l_daoNagios->getHostEscalationContactID($l_id);
            $l_daoNagios->saveContacts($l_contactID, $_POST['C__MODULE__NAGIOS__CONTACTS__HIDDEN']);

            if (!$l_daoNagios->apply_update()) {
                throw new isys_exception_dao($l_daoNagios->get_database_component()
                    ->get_last_error_as_string());
            }
        } catch (isys_exception_dao $e) {
            $_POST['navMode'] = C__NAVMODE__EDIT;
        }

        $l_opt = $l_daoNagios->getHostEscalationOptions();

        $l_assOpt = explode(',', $_POST['C__MODULE__NAGIOS__ESCALATION_OPTIONS__selected_values']);

        foreach ($l_opt as $l_key => $l_val) {
            $l_optArr[] = [
                'id'  => $l_key,
                'val' => $l_val,
                'sel' => in_array($l_key, $l_assOpt),
            ];
        }

        $l_rules = [
            'C__MODULE__NAGIOS__CONTACTS'               => [
                'p_strSelectedID' => $_POST['C__MODULE__NAGIOS__CONTACTS__HIDDEN']
            ],
            'C__MODULE__NAGIOS__ESCALATION_OPTIONS'     => [
                'p_arData' => $l_optArr
            ],
            'C__MODULE__NAGIOS__ESCALATION_PERIOD'      => [
                'p_arData'        => $l_daoNagios->getTimeperiodsAssoc(),
                'p_strSelectedID' => $_POST['C__MODULE__NAGIOS__ESCALATION_PERIOD']
            ],
            'C__MODULE__NAGIOS__ESCALATION_PERIOD_PLUS' => [
                'p_strSelectedID' => $_POST['C__MODULE__NAGIOS__ESCALATION_PERIOD_PLUS']
            ],
            'C__MODULE__NAGIOS__TITLE'                  => [
                'p_strValue' => $_POST['C__MODULE__NAGIOS__TITLE']
            ],
            'C__MODULE__NAGIOS__FIRST_NOTIFICATION'     => [
                'p_strValue' => $_POST['C__MODULE__NAGIOS__FIRST_NOTIFICATION']
            ],
            'C__MODULE__NAGIOS__LAST_NOTIFICATION'      => [
                'p_strValue' => $_POST['C__MODULE__NAGIOS__LAST_NOTIFICATION']
            ],
            'C__MODULE__NAGIOS__NOTIFICATION_INTERVAL'  => [
                'p_strValue' => $_POST['C__MODULE__NAGIOS__NOTIFICATION_INTERVAL']
            ]
        ];

        isys_component_template_navbar::getInstance()
            ->set_active(true, C__NAVBAR_BUTTON__EDIT)
            ->set_active(true, C__NAVBAR_BUTTON__DELETE);

        isys_application::instance()->container->get('template')
            ->assign('eID', $l_id)
            ->smarty_tom_add_rules('tom.content.bottom.content', $l_rules)
            ->include_template('contentbottomcontent', self::getPath() . 'templates/escalation.tpl');
    }

    /**
     *
     * @throws  isys_exception_dao
     */
    public function saveServiceEscalation()
    {
        $l_daoNagios = new isys_component_dao_nagios(isys_application::instance()->container->get('database'));
        $l_optArr = [];
        $l_id = null;

        try {
            if (!isset($_POST["id"]) || empty($_POST["id"])) {
                $l_id = $l_daoNagios->createServiceEscalation();
            } else {
                $l_id = $_POST["id"];
            }

            $l_daoNagios->saveServiceEscalation($l_id);
            $l_contactID = $l_daoNagios->getServiceEscalationContactID($l_id);
            $l_daoNagios->saveContacts($l_contactID, $_POST['C__MODULE__NAGIOS__CONTACTS__HIDDEN']);

            if (!$l_daoNagios->apply_update()) {
                throw new isys_exception_dao($l_daoNagios->get_database_component()
                    ->get_last_error_as_string());
            }
        } catch (isys_exception_dao $e) {
            $_POST['navMode'] = C__NAVMODE__EDIT;
        }

        $l_opt = $l_daoNagios->getServiceEscalationOptions();

        $l_assOpt = explode(',', $_POST['C__MODULE__NAGIOS__ESCALATION_OPTIONS__selected_values']);

        foreach ($l_opt as $l_key => $l_val) {
            $l_optArr[] = [
                'id'  => $l_key,
                'val' => $l_val,
                'sel' => in_array($l_key, $l_assOpt),
            ];
        }

        $l_rules = [
            'C__MODULE__NAGIOS__CONTACTS'               => [
                'p_strSelectedID' => $_POST['C__MODULE__NAGIOS__CONTACTS__HIDDEN']
            ],
            'C__MODULE__NAGIOS__ESCALATION_OPTIONS'     => [
                'p_arData' => $l_optArr
            ],
            'C__MODULE__NAGIOS__ESCALATION_PERIOD'      => [
                'p_arData'        => $l_daoNagios->getTimeperiodsAssoc(),
                'p_strSelectedID' => $_POST['C__MODULE__NAGIOS__ESCALATION_PERIOD']
            ],
            'C__MODULE__NAGIOS__ESCALATION_PERIOD_PLUS' => [
                'p_strSelectedID' => $_POST['C__MODULE__NAGIOS__ESCALATION_PERIOD_PLUS']
            ],
            'C__MODULE__NAGIOS__TITLE'                  => [
                'p_strValue' => $_POST['C__MODULE__NAGIOS__TITLE']
            ],
            'C__MODULE__NAGIOS__FIRST_NOTIFICATION'     => [
                'p_strValue' => $_POST['C__MODULE__NAGIOS__FIRST_NOTIFICATION']
            ],
            'C__MODULE__NAGIOS__LAST_NOTIFICATION'      => [
                'p_strValue' => $_POST['C__MODULE__NAGIOS__LAST_NOTIFICATION']
            ],
            'C__MODULE__NAGIOS__NOTIFICATION_INTERVAL'  => [
                'p_strValue' => $_POST['C__MODULE__NAGIOS__NOTIFICATION_INTERVAL']
            ]
        ];

        isys_component_template_navbar::getInstance()
            ->set_active(true, C__NAVBAR_BUTTON__EDIT)
            ->set_active(true, C__NAVBAR_BUTTON__DELETE);

        isys_application::instance()->container->get('template')
            ->assign('eID', $l_id)
            ->smarty_tom_add_rules('tom.content.bottom.content', $l_rules)
            ->include_template('contentbottomcontent', self::getPath() . 'templates/escalation.tpl');
    }

    /**
     * Saves a command with its parameters provided by $_POST
     *
     * @return void
     *
     */
    public function saveCommand()
    {
        isys_component_template_navbar::getInstance()
            ->set_active(true, C__NAVBAR_BUTTON__EDIT)
            ->set_active(true, C__NAVBAR_BUTTON__DELETE);

        $l_rules = [];
        $l_id = null;

        $l_daoNagios = new isys_component_dao_nagios(isys_application::instance()->container->get('database'));

        try {
            if (!isset($_POST["id"]) || $_POST["id"] == "") {
                $l_id = $l_daoNagios->createCommand();
            } else {
                $l_id = $_POST["id"];
            }

            $l_daoNagios->validateCommandPost($_POST);
            $l_daoNagios->saveCommand($l_id);
        } catch (UnexpectedValueException $e) {
            switch ($e->getCode()) {
                case 0:
                    // @todo  Check if "p_strInfoIconError" can be removed.
                    $l_rules['C__MODULE__NAGIOS__COMMAND_NAME']['p_strInfoIconError'] = $e->getMessage();
                    $l_rules['C__MODULE__NAGIOS__COMMAND_NAME']['message'] = $e->getMessage();
                    break;

                case 1:
                    // @todo  Check if "p_strInfoIconError" can be removed.
                    $l_rules['C__MODULE__NAGIOS__COMMAND_LINE']['p_strInfoIconError'] = $e->getMessage();
                    $l_rules['C__MODULE__NAGIOS__COMMAND_LINE']['message'] = $e->getMessage();
                    break;

                case 2:
                    // @todo  Check if "p_strInfoIconError" can be removed.
                    $l_rules['C__MODULE__NAGIOS__COMMAND_NAME']['p_strInfoIconError'] = $e->getMessage();
                    $l_rules['C__MODULE__NAGIOS__COMMAND_LINE']['p_strInfoIconError'] = $e->getMessage();
                    $l_rules['C__MODULE__NAGIOS__COMMAND_NAME']['message'] = $e->getMessage();
                    $l_rules['C__MODULE__NAGIOS__COMMAND_LINE']['message'] = $e->getMessage();
                    break;
            }

            $_POST['navMode'] = C__NAVMODE__EDIT;
        }

        $l_rules['C__MODULE__NAGIOS__COMMAND_NAME']['p_strValue'] = $_POST['C__MODULE__NAGIOS__COMMAND_NAME'];
        $l_rules['C__MODULE__NAGIOS__COMMAND_LINE']['p_strValue'] = $_POST['C__MODULE__NAGIOS__COMMAND_LINE'];
        $l_rules['C__MODULE__NAGIOS__COMMAND_DESCRIPTION']['p_strValue'] = $_POST['C__MODULE__NAGIOS__COMMAND_DESCRIPTION'];

        isys_application::instance()->container->get('template')
            ->assign("cID", $l_id)
            ->smarty_tom_add_rules("tom.content.bottom.content", $l_rules)
            ->include_template('contentbottomcontent', self::getPath() . 'templates/commands.tpl');
    }

    /**
     * Gathers all Nagios-relevant information from i-doit and writes a config into the local filesystem. Objects, which are not
     * well-defined and all depending objects will be skipped during this process to avoid invalid confgurations.
     *
     * @param array $exportConfiguration
     * @param bool  $validateExport
     *
     * @return isys_nagios_export
     * @throws isys_exception_database
     * @throws isys_exception_filesystem
     * @throws isys_exception_general
     */
    public function exportNagiosConfig($exportConfiguration, $validateExport = false)
    {
        $exportContactsByRole = isys_component_dao_nagios::instance(isys_application::instance()->container->get('database'))
            ->getConfig('EXPORT_CONTACTS_BY_ROLE')
            ->get_row_value('value');

        $l_config = [
            'export_dir'              => $exportConfiguration['isys_monitoring_export_config__path'],
            'export_config'           => $exportConfiguration,
            'export_subdir'           => 'objects',
            'debug'                   => true,
            'validation'              => $validateExport,
            'export_contacts_by_role' => $exportContactsByRole
        ];

        return isys_nagios_export::instance()
            ->init_export($l_config)
            ->start_export();
    }

    /**
     * @throws isys_exception_general
     */
    private function processAjaxRequest()
    {
        isys_application::instance()->container->get('session')->write_close();
        $database = isys_application::instance()->container->get('database');

        switch (isys_glob_get_param("request")) {
            case 'nagios_host_state_table':
                if (!isys_glob_get_param(C__CMDB__GET__OBJECT)) {
                    die('No ID given');
                } else {
                    $l_id = isys_glob_get_param(C__CMDB__GET__OBJECT);
                }

                $l_daoCMDBNagios = isys_cmdb_dao_category_g_nagios::instance($database);
                $l_daoNagios = new isys_component_dao_nagios($database);

                $l_catData = $l_daoCMDBNagios->getCatDataById($l_id);

                if (!$l_daoNagios->is_ndo_instance_active($l_catData["isys_catg_monitoring_list__isys_monitoring_hosts__id"])) {
                    die;
                }

                $l_ndo = isys_monitoring_ndo::factory($l_catData["isys_catg_monitoring_list__isys_monitoring_hosts__id"]);

                try {
                    $l_daoNagios->set_ndo($l_ndo->get_db_connection(), $l_ndo->get_db_prefix());

                    $l_host = $l_daoNagios->getHostStatusTable($l_catData);
                } catch (Exception $e) {
                    die($e->getMessage());
                }

                echo $l_host;

                break;

            case 'nagios_host_state':
                // Enable cache lifetime of 2 minutes
                // isys_core::expire(120);

                if (!isys_glob_get_param(C__CMDB__GET__OBJECT)) {
                    die('No ID given');
                } else {
                    $l_id = isys_glob_get_param(C__CMDB__GET__OBJECT);
                }

                $l_daoCMDBNagios = isys_cmdb_dao_category_g_nagios::instance($database);
                $l_daoNagios = new isys_component_dao_nagios($database);

                $l_catData = $l_daoCMDBNagios->getCatDataById($l_id);

                if (!$l_daoNagios->is_ndo_instance_active($l_catData['isys_catg_monitoring_list__isys_monitoring_hosts__id'])) {
                    die;
                }

                try {
                    $l_ndo = isys_monitoring_ndo::factory($l_catData['isys_catg_monitoring_list__isys_monitoring_hosts__id']);

                    $l_daoNagios->set_ndo($l_ndo->get_db_connection(), $l_ndo->get_db_prefix());

                    $l_host = $l_daoNagios->getHostStatus($l_catData);
                } catch (Exception $e) {
                    $l_host = $e->getMessage();
                }

                echo $l_host;

                break;

            case 'nagios_service_state_table':
                if (!isys_glob_get_param('hostObjID') || !isys_glob_get_param('service_description')) {
                    die('Insufficient parameters given');
                } else {
                    $l_hostID = isys_glob_get_param('hostObjID');
                    $l_sDesc = isys_glob_get_param('service_description');
                }

                $l_daoCMDBNagios = isys_cmdb_dao_category_g_nagios::instance($database);
                $l_daoNagios = new isys_component_dao_nagios($database);

                $l_catData = $l_daoCMDBNagios->getCatDataById($l_hostID);

                if (!$l_daoNagios->is_ndo_instance_active($l_catData['isys_catg_monitoring_list__isys_monitoring_hosts__id'])) {
                    die;
                }

                $l_ndo = isys_monitoring_ndo::factory($l_catData['isys_catg_monitoring_list__isys_monitoring_hosts__id']);

                $l_daoNagios->set_ndo($l_ndo->get_db_connection(), $l_ndo->get_db_prefix());

                echo $l_daoNagios->getServiceStatusTable($l_catData, $l_sDesc);
                break;

            case 'nagios_service_state':
                if (!isys_glob_get_param('hostObjID') || !isys_glob_get_param('service_description')) {
                    die('Insufficient parameters given');
                } else {
                    $l_hostID = isys_glob_get_param('hostObjID');
                    $l_sDesc = isys_glob_get_param('service_description');
                }

                $l_daoCMDBNagios = isys_cmdb_dao_category_g_nagios::instance($database);
                $l_daoNagios = new isys_component_dao_nagios($database);

                $l_catData = $l_daoCMDBNagios->getCatDataById($l_hostID);

                if (!$l_daoNagios->is_ndo_instance_active($l_catData['isys_catg_monitoring_list__isys_monitoring_hosts__id'])) {
                    die;
                }

                $l_ndo = isys_monitoring_ndo::factory($l_catData['isys_catg_monitoring_list__isys_monitoring_hosts__id']);

                $l_daoNagios->set_ndo($l_ndo->get_db_connection(), $l_ndo->get_db_prefix());

                echo $l_daoNagios->getServiceStatus($l_catData, $l_sDesc);
                break;
            case 'export':
                if ($_GET['hid']) {
                    $l_row = isys_monitoring_dao_hosts::instance($database)
                        ->get_export_data($_GET['hid'])
                        ->get_row();

                    if (empty($l_row['isys_monitoring_export_config__path'])) {
                        $l_row['isys_monitoring_export_config__path'] = 'nagiosexport';
                    }

                    $l_export = $this->exportNagiosConfig($l_row, (bool)$_GET['validate']);

                    if (file_exists($l_export->get_log()->get_log_file())) {
                        $l_log_contents = file_get_contents($l_export->get_log()
                            ->get_log_file());
                    } else {
                        $l_log_contents = '';
                    }

                    isys_application::instance()->container->get('template')
                        ->assign('nagios_dir', $l_row['isys_monitoring_export_config__path'])
                        ->assign('exportdest', 'Nagios config written to ' . $l_row['isys_monitoring_export_config__path'])
                        ->assign('logfile', $l_log_contents)
                        ->display(self::getPath() . 'templates/export_done.tpl');
                } else {
                    echo 'Error exporting: No ID for Nagioshost given';
                }
                break;
        }
    }

    /**
     * Method for processing the configuration page.
     *
     * @throws Exception
     */
    private function processConfigPage()
    {
        $database = isys_application::instance()->container->get('database');

        $l_rules = [];
        $l_config_res = isys_component_dao_nagios::instance($database)
            ->getConfig();

        $l_edit_right = isys_auth_nagios::instance()
            ->is_allowed_to(isys_auth::EDIT, 'NAGIOS_CONFIG');

        if (count($l_config_res) > 0) {
            while ($l_row = $l_config_res->get_row()) {
                // Workaround for dialog field.
                if (in_array($l_row['key'], ['PERSON_NAME_OPTION', 'EXPORT_CONTACTS_BY_ROLE'], true)) {
                    $l_rules['C__MODULE__NAGIOS__' . $l_row['key']]['p_strSelectedID'] = $l_row['value'];
                }

                $l_rules['C__MODULE__NAGIOS__' . $l_row['key']]['p_strValue'] = $l_row['value'];
            }
        }

        // @see  NAGIOS-16  Necessary to be able to pass multiple values to the backend.
        $l_rules['C__MODULE__NAGIOS__EXPORT_CONTACTS_BY_ROLE[]'] = $l_rules['C__MODULE__NAGIOS__EXPORT_CONTACTS_BY_ROLE'];

        $l_rules['C__MODULE__NAGIOS__PERSON_NAME_OPTION']['p_arData'] = [
            C__NAGIOS__PERSON_OPTION__OBJECT_TITLE => $this->language->get('LC__UNIVERSAL__OBJECT_TITLE'),
            C__NAGIOS__PERSON_OPTION__USERNAME     => $this->language->get('LC__LOGIN__USERNAME')
        ];

        $l_resource_files = unserialize($l_rules['C__MODULE__NAGIOS__resource_file']['p_strValue']);
        $l_broker_module = unserialize($l_rules['C__MODULE__NAGIOS__broker_module']['p_strValue']);
        $l_cfg_file = unserialize($l_rules['C__MODULE__NAGIOS__cfg_file']['p_strValue']);
        $l_cfg_dir = unserialize($l_rules['C__MODULE__NAGIOS__cfg_dir']['p_strValue']);

        if ($l_resource_files === false) {
            $l_resource_files = [];
        }

        if ($l_broker_module === false) {
            $l_broker_module = [];
        }

        if ($l_cfg_file === false) {
            $l_cfg_file = [];
        }

        if ($l_cfg_dir === false) {
            $l_cfg_dir = [];
        }

        if ($_POST[C__GET__NAVMODE] == C__NAVMODE__EDIT) {
            isys_component_template_navbar::getInstance()
                ->set_active(true, C__NAVBAR_BUTTON__SAVE)
                ->set_active(true, C__NAVBAR_BUTTON__CANCEL);
        } else {
            isys_component_template_navbar::getInstance()
                ->set_active($l_edit_right, C__NAVBAR_BUTTON__EDIT);
        }

        isys_application::instance()->container->get('template')
            ->assign('resource_files', $l_resource_files)
            ->assign('broker_modules', $l_broker_module)
            ->assign('cfg_files', $l_cfg_file)
            ->assign('cfg_dirs', $l_cfg_dir)
            ->assign('object_dir', 'objects/')
            ->assign('content_title', 'Nagios Config')
            ->smarty_tom_add_rules('tom.content.bottom.content', $l_rules)
            ->include_template('contentbottomcontent', self::getPath() . 'templates/config.tpl');
    }

    /**
     * Method for saving the Nagios configuration.
     */
    private function saveConfig()
    {
        isys_component_dao_nagios::instance(isys_application::instance()->container->get('database'))
            ->saveConfig($_POST);

        $this->processConfigPage();
    }

    /**
     * @throws Exception
     */
    private function processExportPage()
    {
        $l_rules = [
            'C__EXPORT_WITH_VALIDATION'     => [
                'p_arData'   => get_smarty_arr_YES_NO(),
                'p_strClass' => 'input input-mini'
            ],
            'C__MODULE__NAGIOS__NAGIOSHOST' => [
                'p_arData'   => isys_component_dao_nagios::instance(isys_application::instance()->container->get('database'))->getNagiosHostsAssoc(),
                'p_strClass' => 'input input-small'
            ]
        ];

        isys_application::instance()->container->get('template')
            ->assign('ajax_url', '?' . C__GET__MODULE_ID . '=' . C__MODULE__NAGIOS . '&request=export&ajax=1')
            ->smarty_tom_add_rules("tom.content.bottom.content", $l_rules)
            ->include_template('contentbottomcontent', self::getPath() . 'templates/export.tpl');
    }

    /**
     * @throws Exception
     * @throws isys_exception_general
     */
    private function processTimeperiodsListing()
    {
        $l_edit_right = isys_auth_nagios::instance()->is_allowed_to(isys_auth::EDIT, 'NAGIOS_TIMEPERIODS');
        $l_delete_right = isys_auth_nagios::instance()->is_allowed_to(isys_auth::DELETE, 'NAGIOS_TIMEPERIODS');
        $template = isys_application::instance()->container->get('template');

        $l_daoNagios = new isys_component_dao_nagios(isys_application::instance()->container->get('database'));

        $l_list = new isys_component_list();

        $l_list_headers = [
            'id'   => 'ID',
            'name' => 'Name'
        ];

        $l_timeperiods_result = $l_daoNagios->getTimeperiods();
        $l_data_count = $l_timeperiods_result->num_rows();

        $l_list->set_data(null, $l_timeperiods_result);
        $l_list->config($l_list_headers, isys_glob_build_url(http_build_query($_GET) . "&" . C__GET__NAGIOS_TPID . "=[{id}]"), "[{id}]");

        if ($l_list->createTempTable()) {
            $template->assign("objectTableList", $l_list->getTempTableHtml());
        }

        isys_component_template_navbar::getInstance()
            ->set_active($l_edit_right, C__NAVBAR_BUTTON__NEW)
            ->set_active(($l_edit_right && $l_data_count > 0), C__NAVBAR_BUTTON__EDIT)
            ->set_active(($l_delete_right && $l_data_count > 0), C__NAVBAR_BUTTON__DELETE)
            ->set_visible(true, C__NAVBAR_BUTTON__NEW)
            ->set_visible(true, C__NAVBAR_BUTTON__EDIT)
            ->set_visible(true, C__NAVBAR_BUTTON__DELETE);

        $template
            ->assign("content_title", "Timeperiods")
            ->smarty_tom_add_rule("tom.content.bottom.buttons.*.p_bInvisible=1")
            ->include_template('contentbottomcontent', 'content/bottom/content/object_table_list.tpl');
    }

    /**
     * @param $p_id
     */
    private function processTimeperiodsPage($p_id)
    {
        $l_navbar = isys_component_template_navbar::getInstance();

        $l_edit_right = isys_auth_nagios::instance()->is_allowed_to(isys_auth::EDIT, 'NAGIOS_TIMEPERIODS');
        $l_delete_right = isys_auth_nagios::instance()->is_allowed_to(isys_auth::DELETE, 'NAGIOS_TIMEPERIODS');
        $l_exArr = [];

        if ($_POST[C__GET__NAVMODE] == C__NAVMODE__EDIT) {
            $l_navbar->set_selected(true, C__NAVBAR_BUTTON__EDIT)
                ->set_active(true, C__NAVBAR_BUTTON__SAVE)
                ->set_active(true, C__NAVBAR_BUTTON__CANCEL);
        } else {
            $l_navbar->set_active($l_edit_right, C__NAVBAR_BUTTON__EDIT)
                ->set_active($l_delete_right, C__NAVBAR_BUTTON__DELETE)
                ->set_visible(true, C__NAVBAR_BUTTON__EDIT)
                ->set_visible(true, C__NAVBAR_BUTTON__DELETE);
        }

        $l_daoNagios = new isys_component_dao_nagios(isys_application::instance()->container->get('database'));

        $l_tp = $l_daoNagios->getTimeperiod($p_id);

        $l_exclude = $l_daoNagios->getTimeperiodsAssoc();

        $l_assExclude = explode(",", $l_tp["exclude"]);
        foreach ($l_exclude as $key => $value) {
            $l_exArr[] = [
                'id'  => $key,
                'val' => $value,
                'sel' => in_array($key, $l_assExclude),
            ];
        }

        $l_rules = [
            'C__MODULE__NAGIOS__TIMEPERIOD_EXCLUDE'          => [
                'p_arData' => $l_exArr
            ],
            'C__MODULE__NAGIOS__DEFAULT_CHECK_PERIOD'        => [
                'p_arData'        => get_smarty_arr_YES_NO(),
                'p_strSelectedID' => $l_tp["def_check"]
            ],
            'C__MODULE__NAGIOS__DEFAULT_NOTIFICATION_PERIOD' => [
                'p_arData'        => [],
                'p_strSelectedID' => $l_tp["def_not"]
            ],
            'C__MODULE__NAGIOS__TIMEPERIOD_NAME'             => [
                'p_strValue' => $l_tp["name"]
            ],
            'C__MODULE__NAGIOS__TIMEPERIOD_ALIAS'            => [
                'p_strValue' => $l_tp["alias"]
            ],
            'C__MODULE__NAGIOS__TIMEPERIOD_DEFINITION'       => [
                'p_strValue' => $l_tp["definition"]
            ]
        ];

        $l_rules['C__MODULE__NAGIOS__DEFAULT_NOTIFICATION_PERIOD']['p_arData'] = $l_rules['C__MODULE__NAGIOS__DEFAULT_CHECK_PERIOD']['p_arData'];

        isys_application::instance()->container->get('template')
            ->assign("tpID", $p_id)
            ->smarty_tom_add_rules("tom.content.bottom.content", $l_rules)
            ->include_template('contentbottomcontent', self::getPath() . 'templates/timeperiods.tpl');
    }

    /**
     * @throws Exception
     */
    private function createTimeperiod()
    {
        isys_component_template_navbar::getInstance()
            ->set_selected(true, C__NAVBAR_BUTTON__NEW);

        $_POST[C__GET__NAVMODE] = C__NAVMODE__EDIT;
        $rulesContent = [
            'p_arData'        => get_smarty_arr_YES_NO(),
            'p_strSelectedID' => 0
        ];

        $rules = [
           'C__MODULE__NAGIOS__DEFAULT_NOTIFICATION_PERIOD' => $rulesContent,
           'C__MODULE__NAGIOS__DEFAULT_CHECK_PERIOD' => $rulesContent
        ];

        isys_application::instance()->container->get('template')
            ->smarty_tom_add_rules("tom.content.bottom.content", $rules)
            ->include_template('contentbottomcontent', self::getPath() . 'templates/timeperiods.tpl');
    }

    /**
     * @throws Exception
     */
    private function saveTimeperiod()
    {
        isys_component_template_navbar::getInstance()
            ->set_active(true, C__NAVBAR_BUTTON__EDIT)
            ->set_active(true, C__NAVBAR_BUTTON__DELETE);

        $l_exArr = $l_rules = [];
        $l_id = null;

        $l_daoNagios = new isys_component_dao_nagios(isys_application::instance()->container->get('database'));

        try {
            if (empty($_POST["id"])) {
                $l_id = $l_daoNagios->createTimeperiod();
            } else {
                $l_id = $_POST["id"];
            }

            $l_daoNagios->validateTimeperiodPost($_POST);
            $l_daoNagios->saveTimeperiod($l_id);
        } catch (UnexpectedValueException $e) {
            switch ($e->getCode()) {
                case 0:
                    // @todo  Check if "p_strInfoIconError" can be removed.
                    $l_rules["C__MODULE__NAGIOS__TIMEPERIOD_NAME"]["p_strInfoIconError"] = $e->getMessage();
                    $l_rules["C__MODULE__NAGIOS__TIMEPERIOD_NAME"]["message"] = $e->getMessage();
                    break;

                case 1:
                    // @todo  Check if "p_strInfoIconError" can be removed.
                    $l_rules["C__MODULE__NAGIOS__TIMEPERIOD_ALIAS"]["p_strInfoIconError"] = $e->getMessage();
                    $l_rules["C__MODULE__NAGIOS__TIMEPERIOD_ALIAS"]["message"] = $e->getMessage();
                    break;

                case 2:
                    // @todo  Check if "p_strInfoIconError" can be removed.
                    $l_rules["C__MODULE__NAGIOS__TIMEPERIOD_NAME"]["p_strInfoIconError"] = $e->getMessage();
                    $l_rules["C__MODULE__NAGIOS__TIMEPERIOD_ALIAS"]["p_strInfoIconError"] = $e->getMessage();
                    $l_rules["C__MODULE__NAGIOS__TIMEPERIOD_NAME"]["message"] = $e->getMessage();
                    $l_rules["C__MODULE__NAGIOS__TIMEPERIOD_ALIAS"]["message"] = $e->getMessage();
                    break;
            }

            $_POST["navMode"] = C__NAVMODE__EDIT;
        }

        $l_exclude = $l_daoNagios->getTimeperiodsAssoc();

        $l_assExclude = explode(",", $_POST["C__MODULE__NAGIOS__TIMEPERIOD_EXCLUDE__selected_values"]);
        foreach ($l_exclude as $l_key => $l_val) {
            $l_exArr[] = [
                'id'  => $l_key,
                'val' => $l_val,
                'sel' => in_array($l_key, $l_assExclude),
            ];
        }

        $l_rules["C__MODULE__NAGIOS__TIMEPERIOD_EXCLUDE"]["p_arData"] = $l_exArr;
        $l_rules["C__MODULE__NAGIOS__DEFAULT_CHECK_PERIOD"]["p_arData"] = get_smarty_arr_YES_NO();
        $l_rules["C__MODULE__NAGIOS__DEFAULT_CHECK_PERIOD"]["p_strSelectedID"] = $_POST["C__MODULE__NAGIOS__DEFAULT_CHECK_PERIOD"];
        $l_rules["C__MODULE__NAGIOS__DEFAULT_NOTIFICATION_PERIOD"]["p_arData"] = $l_rules["C__MODULE__NAGIOS__DEFAULT_CHECK_PERIOD"]["p_arData"];
        $l_rules["C__MODULE__NAGIOS__DEFAULT_NOTIFICATION_PERIOD"]["p_strSelectedID"] = $_POST["C__MODULE__NAGIOS__DEFAULT_NOTIFICATION_PERIOD"];
        $l_rules["C__MODULE__NAGIOS__TIMEPERIOD_NAME"]["p_strValue"] = $_POST["C__MODULE__NAGIOS__TIMEPERIOD_NAME"];
        $l_rules["C__MODULE__NAGIOS__TIMEPERIOD_ALIAS"]["p_strValue"] = $_POST["C__MODULE__NAGIOS__TIMEPERIOD_ALIAS"];
        $l_rules["C__MODULE__NAGIOS__TIMEPERIOD_DEFINITION"]["p_strValue"] = $_POST["C__MODULE__NAGIOS__TIMEPERIOD_DEFINITION"];

        isys_application::instance()->container->get('template')
            ->assign("tpID", $l_id)
            ->smarty_tom_add_rules("tom.content.bottom.content", $l_rules)
            ->include_template('contentbottomcontent', self::getPath() . 'templates/timeperiods.tpl');
    }

    /**
     * @throws isys_exception_general
     * @throws Exception
     */
    private function deleteCommand()
    {
        $l_daoNagios = new isys_component_dao_nagios(isys_application::instance()->container->get('database'));
        try {
            if (isset($_POST["id"])) {
                if (is_array($_POST["id"])) {
                    foreach ($_POST["id"] as $l_key => $l_value) {
                        $l_daoNagios->deleteCommand($l_value);
                    }
                } elseif ($_POST["id"] != "") {
                    $l_daoNagios->deleteCommand($_POST["id"]);
                }
            } else {
                throw new Exception("No id given");
            }
        } catch (Exception $e) {
            $l_tError = $e->getMessage();
        }
        $this->processCommandsListing();
    }

    /**
     * @throws isys_exception_general
     * @throws Exception
     */
    private function deleteTimeperiod()
    {
        $l_daoNagios = new isys_component_dao_nagios(isys_application::instance()->container->get('database'));
        try {
            if (isset($_POST["id"])) {
                if (is_array($_POST["id"])) {
                    foreach ($_POST["id"] as $l_key => $l_value) {
                        $l_daoNagios->deleteTimeperiod($l_value);
                    }
                } elseif ($_POST["id"] != "") {
                    $l_daoNagios->deleteTimeperiod($_POST["id"]);
                }
            } else {
                throw new Exception("No id given");
            }
        } catch (Exception $e) {
            $l_tError = $e->getMessage();
        }
        $this->processTimeperiodsListing();
    }

    /**
     * @throws Exception
     * @throws isys_exception_general
     */
    private function processHostEscalationsListing()
    {
        $l_navbar = isys_component_template_navbar::getInstance();

        $l_edit_right = isys_auth_nagios::instance()->is_allowed_to(isys_auth::EDIT, 'NAGIOS_HOST_ESCALATIONS');
        $l_delete_right = isys_auth_nagios::instance()->is_allowed_to(isys_auth::DELETE, 'NAGIOS_HOST_ESCALATIONS');

        $l_list = new isys_component_list();
        $l_list_headers = [
            "id"    => "ID",
            "title" => "Name"
        ];
        $l_list_data = [];

        $l_daoNagios = new isys_component_dao_nagios(isys_application::instance()->container->get('database'));
        $l_list_data = $l_daoNagios->getHostEscalations();

        $l_list->set_data($l_list_data);
        $l_list->config($l_list_headers, isys_glob_build_url(http_build_query($_GET) . "&" . C__GET__NAGIOS_EID . "=[{id}]"), "[{id}]");

        $l_navbar->set_active($l_edit_right, C__NAVBAR_BUTTON__NEW)
            ->set_active(($l_edit_right && count($l_list_data) > 0), C__NAVBAR_BUTTON__EDIT)
            ->set_active(($l_delete_right && count($l_list_data) > 0), C__NAVBAR_BUTTON__DELETE)
            ->set_visible(true, C__NAVBAR_BUTTON__NEW)
            ->set_visible(true, C__NAVBAR_BUTTON__EDIT)
            ->set_visible(true, C__NAVBAR_BUTTON__DELETE);

        if ($l_list->createTempTable()) {
            isys_application::instance()->container->get('template')
                ->assign("objectTableList", $l_list->getTempTableHtml());
        }

        isys_application::instance()->container->get('template')
            ->assign("content_title", "Host escalations")
            ->smarty_tom_add_rule("tom.content.bottom.buttons.*.p_bInvisible=1")
            ->include_template('contentbottomcontent', 'content/bottom/content/object_table_list.tpl');
    }

    /**
     *
     */
    private function createHostEscalation()
    {
        isys_component_template_navbar::getInstance()
            ->set_selected(true, C__NAVBAR_BUTTON__NEW);

        $l_daoNagios = new isys_component_dao_nagios(isys_application::instance()->container->get('database'));

        $_POST[C__GET__NAVMODE] = C__NAVMODE__EDIT;
        $l_optArr = [];

        $l_opt = $l_daoNagios->getHostEscalationOptions();

        foreach ($l_opt as $l_key => $l_val) {
            $l_optArr[] = [
                'id'  => $l_key,
                'val' => $l_val,
                'sel' => 0
            ];
        }

        $l_rules = [
            'C__MODULE__NAGIOS__ESCALATION_PERIOD'  => [
                'p_arData' => $l_daoNagios->getTimeperiodsAssoc()
            ],
            'C__MODULE__NAGIOS__ESCALATION_OPTIONS' => [
                'p_arData' => $l_optArr
            ]
        ];

        isys_application::instance()->container->get('template')
            ->smarty_tom_add_rules("tom.content.bottom.content", $l_rules)
            ->include_template('contentbottomcontent', self::getPath() . 'templates/escalation.tpl');
    }

    /**
     *
     */
    private function createServiceEscalation()
    {
        isys_component_template_navbar::getInstance()
            ->set_selected(true, C__NAVBAR_BUTTON__NEW);

        $l_daoNagios = new isys_component_dao_nagios(isys_application::instance()->container->get('database'));
        $l_optArr = [];

        $_POST[C__GET__NAVMODE] = C__NAVMODE__EDIT;

        // rules
        $l_opt = $l_daoNagios->getServiceEscalationOptions();

        foreach ($l_opt as $l_key => $l_val) {
            $l_optArr[] = [
                'id'  => $l_key,
                'val' => $l_val,
                'sel' => 0
            ];
        }

        $l_rules = [
            'C__MODULE__NAGIOS__ESCALATION_PERIOD'  => [
                'p_arData' => $l_daoNagios->getTimeperiodsAssoc()
            ],
            'C__MODULE__NAGIOS__ESCALATION_OPTIONS' => [
                'p_arData' => $l_optArr
            ]
        ];

        isys_application::instance()->container->get('template')
            ->smarty_tom_add_rules("tom.content.bottom.content", $l_rules)
            ->include_template('contentbottomcontent', self::getPath() . 'templates/escalation.tpl');
    }

    /**
     * Method for processing the host escalation-page.
     *
     * @param  integer $p_id
     */
    private function processHostEscalationsPage($p_id)
    {
        $database = isys_application::instance()->container->get('database');

        // Prepare the navigation-bar.
        $l_navbar = isys_component_template_navbar::getInstance();
        $l_optArr = [];

        $l_edit_right = isys_auth_nagios::instance()
            ->is_allowed_to(isys_auth::EDIT, 'NAGIOS_HOST_ESCALATIONS');
        $l_delete_right = isys_auth_nagios::instance()
            ->is_allowed_to(isys_auth::DELETE, 'NAGIOS_HOST_ESCALATIONS');

        if ($_POST["navMode"] == C__NAVMODE__EDIT) {
            $l_navbar->set_selected(true, C__NAVBAR_BUTTON__EDIT)
                ->set_active(true, C__NAVBAR_BUTTON__SAVE)
                ->set_active(true, C__NAVBAR_BUTTON__CANCEL);
        } else {
            $l_navbar->set_active($l_edit_right, C__NAVBAR_BUTTON__EDIT)
                ->set_active($l_delete_right, C__NAVBAR_BUTTON__DELETE);
        }

        $l_daoNagios = new isys_component_dao_nagios($database);

        $l_he = $l_daoNagios->getHostEscalation($p_id);

        $l_opt = $l_daoNagios->getHostEscalationOptions();

        $l_assOpt = explode(",", $l_he["isys_nagios_host_escalations__escalation_options"]);

        foreach ($l_opt as $l_key => $l_val) {
            $l_optArr[] = [
                'id'  => $l_key,
                'val' => $l_val,
                'sel' => in_array($l_key, $l_assOpt)
            ];
        }

        $l_contactID = $l_daoNagios->getHostEscalationContactID($p_id);

        $l_daoContact = new isys_contact_dao_reference($database);
        $l_daoContact->load($l_contactID);
        $l_data_items = $l_daoContact->get_data_item_array();

        $l_rules = [
            'C__MODULE__NAGIOS__ESCALATION_OPTIONS'     => [
                'p_arData' => $l_optArr
            ],
            'C__MODULE__NAGIOS__ESCALATION_PERIOD'      => [
                'p_arData'        => $l_daoNagios->getTimeperiodsAssoc(),
                'p_strSelectedID' => $l_he["isys_nagios_host_escalations__escalation_period"]
            ],
            'C__MODULE__NAGIOS__ESCALATION_PERIOD_PLUS' => [
                'p_strSelectedID' => $l_he["isys_nagios_host_escalations__escalation_period_plus"]
            ],
            'C__MODULE__NAGIOS__TITLE'                  => [
                'p_strValue' => $l_he["isys_nagios_host_escalations__title"]
            ],
            'C__MODULE__NAGIOS__FIRST_NOTIFICATION'     => [
                'p_strValue' => $l_he["isys_nagios_host_escalations__first_notification"]
            ],
            'C__MODULE__NAGIOS__LAST_NOTIFICATION'      => [
                'p_strValue' => $l_he["isys_nagios_host_escalations__last_notification"]
            ],
            'C__MODULE__NAGIOS__NOTIFICATION_INTERVAL'  => [
                'p_strValue' => $l_he["isys_nagios_host_escalations__notification_interval"]
            ]
        ];

        if (is_array($l_data_items) && count($l_data_items)) {
            $l_persons = array_keys($l_data_items);
            $l_rules['C__MODULE__NAGIOS__CONTACTS']['p_strSelectedID'] = implode(',', $l_persons);
        }

        isys_application::instance()->container->get('template')
            ->assign('eID', $p_id)
            ->smarty_tom_add_rules('tom.content.bottom.content', $l_rules)
            ->include_template('contentbottomcontent', self::getPath() . 'templates/escalation.tpl');
    }

    /**
     * Method for processing the service-escalations page.
     *
     * @param   integer $p_id
     *
     * @author  Leonard Fischer <lfischer@i-doit.org>
     */
    private function processServiceEscalationsPage($p_id)
    {
        $database = isys_application::instance()->container->get('database');
        $l_edit_right = isys_auth_nagios::instance()->is_allowed_to(isys_auth::EDIT, 'NAGIOS_SERVICE_ESCALATIONS');
        $l_delete_right = isys_auth_nagios::instance()->is_allowed_to(isys_auth::DELETE, 'NAGIOS_SERVICE_ESCALATIONS');

        // Navbar instance
        $l_navbar = isys_component_template_navbar::getInstance();
        $l_optArr = [];

        if ($_POST["navMode"] == C__NAVMODE__EDIT) {
            $l_navbar->set_selected(true, C__NAVBAR_BUTTON__EDIT)
                ->set_active(true, C__NAVBAR_BUTTON__SAVE)
                ->set_active(true, C__NAVBAR_BUTTON__CANCEL);
        } else {
            $l_navbar->set_active($l_edit_right, C__NAVBAR_BUTTON__EDIT)
                ->set_active($l_delete_right, C__NAVBAR_BUTTON__DELETE)
                ->set_visible(true, C__NAVBAR_BUTTON__EDIT)
                ->set_visible(true, C__NAVBAR_BUTTON__DELETE);
        }

        $l_daoNagios = new isys_component_dao_nagios($database);

        $l_he = $l_daoNagios->getServiceEscalation($p_id);

        $l_opt = $l_daoNagios->getServiceEscalationOptions();

        $l_assOpt = explode(",", $l_he["isys_nagios_service_escalations__escalation_options"]);

        foreach ($l_opt as $l_key => $l_val) {
            $l_optArr[] = [
                'id'  => $l_key,
                'val' => $l_val,
                'sel' => in_array($l_key, $l_assOpt),
            ];
        }

        $l_contactID = $l_daoNagios->getServiceEscalationContactID($p_id);

        $l_daoContact = new isys_contact_dao_reference($database);
        $l_daoContact->load($l_contactID);
        $l_data_items = $l_daoContact->get_data_item_array();

        $l_rules = [
            'C__MODULE__NAGIOS__ESCALATION_OPTIONS'     => [
                'p_arData' => $l_optArr
            ],
            'C__MODULE__NAGIOS__ESCALATION_PERIOD'      => [
                'p_arData'        => $l_daoNagios->getTimeperiodsAssoc(),
                'p_strSelectedID' => $l_he["isys_nagios_service_escalations__escalation_period"]
            ],
            'C__MODULE__NAGIOS__ESCALATION_PERIOD_PLUS' => [
                'p_strSelectedID' => $l_he["isys_nagios_service_escalations__escalation_period_plus"]
            ],
            'C__MODULE__NAGIOS__TITLE'                  => [
                'p_strValue' => $l_he["isys_nagios_service_escalations__title"]
            ],
            'C__MODULE__NAGIOS__FIRST_NOTIFICATION'     => [
                'p_strValue' => $l_he["isys_nagios_service_escalations__first_notification"]
            ],
            'C__MODULE__NAGIOS__LAST_NOTIFICATION'      => [
                'p_strValue' => $l_he["isys_nagios_service_escalations__last_notification"]
            ],
            'C__MODULE__NAGIOS__NOTIFICATION_INTERVAL'  => [
                'p_strValue' => $l_he["isys_nagios_service_escalations__notification_interval"]
            ]
        ];

        if (is_array($l_data_items)) {
            $l_persons = array_keys($l_data_items);

            if (count($l_persons) > 0) {
                $l_rules['C__MODULE__NAGIOS__CONTACTS']['p_strSelectedID'] = implode(',', $l_persons);
            }
        }

        isys_application::instance()->container->get('template')
            ->assign("eID", $p_id)
            ->smarty_tom_add_rules("tom.content.bottom.content", $l_rules)
            ->include_template('contentbottomcontent', self::getPath() . 'templates/escalation.tpl');
    }

    /**
     * Method for deleting host-escalations.
     *
     * @throws Exception
     */
    private function deleteHostEscalation()
    {
        $l_daoNagios = new isys_component_dao_nagios(isys_application::instance()->container->get('database'));

        try {
            if (isset($_POST['id'])) {
                if (is_array($_POST['id'])) {
                    foreach ($_POST['id'] as $l_key => $l_value) {
                        $l_daoNagios->deleteHostEscalation($l_value);
                    }
                } elseif ($_POST['id'] != '') {
                    $l_daoNagios->deleteHostEscalation($_POST['id']);
                }
            } else {
                throw new Exception('No id given');
            }
        } catch (Exception $e) {
            $l_tError = $e->getMessage();
        }

        $this->processHostEscalationsListing();
    }

    /**
     * Method for deleting service-escalations.
     *
     * @throws Exception
     */
    private function deleteServiceEscalation()
    {
        $l_daoNagios = new isys_component_dao_nagios(isys_application::instance()->container->get('database'));

        try {
            if (isset($_POST['id'])) {
                if (is_array($_POST['id'])) {
                    foreach ($_POST['id'] as $l_key => $l_value) {
                        $l_daoNagios->deleteServiceEscalation($l_value);
                    }
                } elseif ($_POST['id'] != '') {
                    $l_daoNagios->deleteServiceEscalation($_POST['id']);
                }
            } else {
                throw new Exception("No id given");
            }
        } catch (Exception $e) {
            $l_tError = $e->getMessage();
        }

        $this->processServiceEscalationsListing();
    }

    /**
     * @throws Exception
     * @throws isys_exception_general
     */
    private function processServiceEscalationsListing()
    {
        $l_edit_right = isys_auth_nagios::instance()->is_allowed_to(isys_auth::EDIT, 'NAGIOS_SERVICE_ESCALATIONS');
        $l_delete_right = isys_auth_nagios::instance()->is_allowed_to(isys_auth::DELETE, 'NAGIOS_SERVICE_ESCALATIONS');

        $l_list = new isys_component_list();
        $l_list_headers = [
            'id'    => 'ID',
            'title' => 'Name'
        ];

        $l_daoNagios = new isys_component_dao_nagios(isys_application::instance()->container->get('database'));
        $l_list_data = $l_daoNagios->getServiceEscalations();

        $l_list->set_data($l_list_data)
            ->config($l_list_headers, isys_glob_build_url(http_build_query($_GET) . "&" . C__GET__NAGIOS_EID . "=[{id}]"), "[{id}]");

        isys_component_template_navbar::getInstance()
            ->set_active($l_edit_right, C__NAVBAR_BUTTON__NEW)
            ->set_active(($l_edit_right && count($l_list_data) > 0), C__NAVBAR_BUTTON__EDIT)
            ->set_active(($l_delete_right && count($l_list_data) > 0), C__NAVBAR_BUTTON__DELETE)
            ->set_visible(true, C__NAVBAR_BUTTON__NEW)
            ->set_visible(true, C__NAVBAR_BUTTON__EDIT)
            ->set_visible(true, C__NAVBAR_BUTTON__DELETE);

        if ($l_list->createTempTable()) {
            isys_application::instance()->container->get('template')
                ->assign('objectTableList', $l_list->getTempTableHtml());
        }

        isys_application::instance()->container->get('template')
            ->assign('content_title', 'Service escalations')
            ->smarty_tom_add_rule('tom.content.bottom.buttons.*.p_bInvisible=1')
            ->include_template('contentbottomcontent', 'content/bottom/content/object_table_list.tpl');
    }

    /**
     * @throws Exception
     * @throws isys_exception_general
     */
    private function processCommandsListing()
    {
        $l_edit_right = isys_auth_nagios::instance()->is_allowed_to(isys_auth::EDIT, 'NAGIOS_COMMANDS');
        $l_delete_right = isys_auth_nagios::instance()->is_allowed_to(isys_auth::DELETE, 'NAGIOS_COMMANDS');

        $l_list = new isys_component_list();
        $l_list_headers = [
            'id'   => 'ID',
            'name' => 'Name'
        ];
        $l_daoNagios = new isys_component_dao_nagios(isys_application::instance()->container->get('database'));

        $l_commands_result = $l_daoNagios->getCommands();
        $l_commands_count = $l_commands_result->num_rows();
        $l_list->set_data(null, $l_commands_result);
        $l_list->config($l_list_headers, isys_glob_build_url(http_build_query($_GET) . "&" . C__GET__NAGIOS_CID . "=[{id}]"), "[{id}]");

        isys_component_template_navbar::getInstance()
            ->set_active($l_edit_right, C__NAVBAR_BUTTON__NEW)
            ->set_active(($l_edit_right && $l_commands_count > 0), C__NAVBAR_BUTTON__EDIT)
            ->set_active(($l_delete_right && $l_commands_count > 0), C__NAVBAR_BUTTON__DELETE)
            ->set_visible(true, C__NAVBAR_BUTTON__NEW)
            ->set_visible(true, C__NAVBAR_BUTTON__EDIT)
            ->set_visible(true, C__NAVBAR_BUTTON__DELETE);

        if ($l_list->createTempTable()) {
            isys_application::instance()->container->get('template')
                ->assign('objectTableList', $l_list->getTempTableHtml());
        }

        isys_application::instance()->container->get('template')
            ->assign('content_title', 'Commands')
            ->smarty_tom_add_rule('tom.content.bottom.buttons.*.p_bInvisible=1')
            ->include_template('contentbottomcontent', 'content/bottom/content/object_table_list.tpl');
    }

    /**
     * @param $p_id
     */
    private function processCommandsPage($p_id)
    {
        $l_navbar = isys_component_template_navbar::getInstance();

        $l_edit_right = isys_auth_nagios::instance()->is_allowed_to(isys_auth::EDIT, 'NAGIOS_COMMANDS');
        $l_delete_right = isys_auth_nagios::instance()->is_allowed_to(isys_auth::DELETE, 'NAGIOS_COMMANDS');

        if ($_POST[C__GET__NAVMODE] == C__NAVMODE__EDIT) {
            $l_navbar->set_selected(true, C__NAVBAR_BUTTON__EDIT)
                ->set_active(true, C__NAVBAR_BUTTON__SAVE)
                ->set_active(true, C__NAVBAR_BUTTON__CANCEL);
        } else {
            $l_navbar->set_active($l_edit_right, C__NAVBAR_BUTTON__EDIT)
                ->set_active($l_delete_right, C__NAVBAR_BUTTON__DELETE)
                ->set_visible(true, C__NAVBAR_BUTTON__EDIT)
                ->set_visible(true, C__NAVBAR_BUTTON__DELETE);
        }

        $command = (new isys_component_dao_nagios(isys_application::instance()->container->get('database')))->getCommand($p_id);

        $l_rules = [
            'C__MODULE__NAGIOS__COMMAND_NAME'        => [
                'p_strValue' => $command["name"]
            ],
            'C__MODULE__NAGIOS__COMMAND_LINE'        => [
                'p_strValue' => $command["line"]
            ],
            'C__MODULE__NAGIOS__COMMAND_DESCRIPTION' => [
                'p_strValue' => $command["description"]
            ]
        ];

        isys_application::instance()->container->get('template')
            ->assign("cID", $p_id)
            ->smarty_tom_add_rules("tom.content.bottom.content", $l_rules)
            ->include_template('contentbottomcontent', self::getPath() . 'templates/commands.tpl');
    }

    /**
     *
     */
    private function createCommand()
    {
        isys_component_template_navbar::getInstance()
            ->set_selected(true, C__NAVBAR_BUTTON__NEW);

        $_POST[C__GET__NAVMODE] = C__NAVMODE__EDIT;

        isys_application::instance()->container->get('template')
            ->include_template('contentbottomcontent', self::getPath() . 'templates/commands.tpl');
    }

    /**
     * Checks if a add-on is active.
     *
     * @return integer|bool
     */
    public static function isActive()
    {
        return isys_module_manager::instance()->is_active('nagios');
    }

    /**
     * Method that is called after clicking "activate" in admin center.
     * Override this methode for special handling.
     *
     * @param isys_component_database $tenantDatabase
     *
     * @return boolean
     */
    public static function activate($tenantDatabase)
    {
        $cmdbDao = isys_cmdb_dao::instance($tenantDatabase);
        $categoryCollection = [
            'cats' => [
                'C__CATS__PERSON_NAGIOS',
                'C__CATS__PERSON_GROUP_NAGIOS'
            ],
            'catg' => [
                'C__CATG__NAGIOS',
                'C__CATG__NAGIOS_APPLICATION_FOLDER',
                'C__CATG__NAGIOS_APPLICATION_REFS_NAGIOS_SERVICE',
                'C__CATG__NAGIOS_GROUP',
                'C__CATG__NAGIOS_HOST_FOLDER',
                'C__CATG__NAGIOS_HOST_TPL_ASSIGNED_OBJECTS',
                'C__CATG__NAGIOS_HOST_TPL_DEF',
                'C__CATG__NAGIOS_HOST_TPL_FOLDER',
                'C__CATG__NAGIOS_REFS_SERVICES',
                'C__CATG__NAGIOS_REFS_SERVICES_BACKWARDS',
                'C__CATG__NAGIOS_SERVICE_DEF',
                'C__CATG__NAGIOS_SERVICE_DEP',
                'C__CATG__NAGIOS_SERVICE_FOLDER',
                'C__CATG__NAGIOS_SERVICE_REFS_TPL_BACKWARDS',
                'C__CATG__NAGIOS_SERVICE_TPL_DEF',
                'C__CATG__NAGIOS_SERVICE_TPL_FOLDER'
            ]
        ];

        try {
            foreach ($categoryCollection as $categoryType => $categories) {
                $sql = 'UPDATE isysgui_' . $categoryType . ' 
                    SET isysgui_' . $categoryType . '__status = ' . $cmdbDao->convert_sql_int(C__RECORD_STATUS__NORMAL) . '
                    WHERE isysgui_' . $categoryType . '__id ' . $cmdbDao->prepare_in_condition($categories) . ';';

                $cmdbDao->update($sql) && $cmdbDao->apply_update();
            }

            return true;
        } catch (Exception $e) {
            return false;
        }
    }

    /**
     * Method that is called after clicking "deactivate" in admin center.
     * Override this methode for special handling.
     *
     * @param isys_component_database $tenantDatabase
     *
     * @return boolean
     * @author Leonard Fischer <lfischer@i-doit.com>
     */
    public static function deactivate($tenantDatabase)
    {
        $cmdbDao = isys_cmdb_dao::instance($tenantDatabase);
        $categoryCollection = [
            'cats' => [
                'C__CATS__PERSON_NAGIOS',
                'C__CATS__PERSON_GROUP_NAGIOS'
            ],
            'catg' => [
                'C__CATG__NAGIOS',
                'C__CATG__NAGIOS_APPLICATION_FOLDER',
                'C__CATG__NAGIOS_APPLICATION_REFS_NAGIOS_SERVICE',
                'C__CATG__NAGIOS_GROUP',
                'C__CATG__NAGIOS_HOST_FOLDER',
                'C__CATG__NAGIOS_HOST_TPL_ASSIGNED_OBJECTS',
                'C__CATG__NAGIOS_HOST_TPL_DEF',
                'C__CATG__NAGIOS_HOST_TPL_FOLDER',
                'C__CATG__NAGIOS_REFS_SERVICES',
                'C__CATG__NAGIOS_REFS_SERVICES_BACKWARDS',
                'C__CATG__NAGIOS_SERVICE_DEF',
                'C__CATG__NAGIOS_SERVICE_DEP',
                'C__CATG__NAGIOS_SERVICE_FOLDER',
                'C__CATG__NAGIOS_SERVICE_REFS_TPL_BACKWARDS',
                'C__CATG__NAGIOS_SERVICE_TPL_DEF',
                'C__CATG__NAGIOS_SERVICE_TPL_FOLDER'
            ]
        ];

        try {
            foreach ($categoryCollection as $categoryType => $categories) {
                $sql = 'UPDATE isysgui_' . $categoryType . ' 
                    SET isysgui_' . $categoryType . '__status = ' . $cmdbDao->convert_sql_int(C__RECORD_STATUS__DELETED) . '
                    WHERE isysgui_' . $categoryType . '__id ' . $cmdbDao->prepare_in_condition($categories) . ';';

                $cmdbDao->update($sql) && $cmdbDao->apply_update();
            }

            return true;
        } catch (Exception $e) {
            return false;
        }
    }

    /**
     * Checks if a add-on is installed.
     *
     * @return int|bool
     */
    public static function isInstalled()
    {
        return isys_module_manager::instance()->is_installed('nagios');
    }

    /**
     * Basic installation process for all mandators.
     *
     * @param  isys_component_database $tenantDatabase
     * @param  isys_component_database $systemDatabase
     * @param  integer                 $moduleId
     * @param  string                  $type
     * @param  integer                 $tenantId
     *
     * @return boolean
     * @throws isys_exception_dao
     * @throws isys_exception_database
     */
    public static function install($tenantDatabase, $systemDatabase, $moduleId, $type, $tenantId)
    {
        if ($type === 'install') {
            $dao = isys_cmdb_dao::instance($tenantDatabase);

            $adminPersonGroupId = $dao->retrieve('SELECT isys_obj__id FROM isys_obj WHERE isys_obj__const = "C__OBJ__PERSON_GROUP_ADMIN";')->get_row_value('isys_obj__id');
            $adminPersonGroup = $dao->convert_sql_id($adminPersonGroupId);
            $module = $dao->convert_sql_id($moduleId);
            $status = $dao->convert_sql_int(C__RECORD_STATUS__NORMAL);

            $sql = "INSERT INTO isys_auth (isys_auth__isys_obj__id, isys_auth__type, isys_auth__isys_module__id, isys_auth__path, isys_auth__status) VALUES 
                ({$adminPersonGroup}, 4095, {$module}, 'NAGIOS_COMMANDS', {$status}), 
                ({$adminPersonGroup}, 4095, {$module}, 'NAGIOS_CONFIG', {$status}),
                ({$adminPersonGroup}, 9, {$module}, 'NAGIOS_EXPORT', {$status}),
                ({$adminPersonGroup}, 4095, {$module}, 'NAGIOS_HOST_ESCALATIONS', {$status}),
                ({$adminPersonGroup}, 4095, {$module}, 'NAGIOS_NAGIOS_HOSTS', {$status}),
                ({$adminPersonGroup}, 4095, {$module}, 'NAGIOS_SERVICE_ESCALATIONS', {$status}),
                ({$adminPersonGroup}, 4095, {$module}, 'NAGIOS_TIMEPERIODS', {$status});";

            return $dao->update($sql) && $dao->apply_update();
        }

        return true;
    }

    /**
     * Uninstall add-on for all mandators.
     *
     * @param isys_component_database $tenantDatabase
     *
     * @return bool
     */
    public static function uninstall($tenantDatabase)
    {
        $cmdbDao = isys_cmdb_dao::instance($tenantDatabase);
        $categoryCollection = [
            'cats' => [
                'C__CATS__PERSON_NAGIOS',
                'C__CATS__PERSON_GROUP_NAGIOS'
            ],
            'catg' => [
                'C__CATG__NAGIOS',
                'C__CATG__NAGIOS_APPLICATION_FOLDER',
                'C__CATG__NAGIOS_APPLICATION_REFS_NAGIOS_SERVICE',
                'C__CATG__NAGIOS_GROUP',
                'C__CATG__NAGIOS_HOST_FOLDER',
                'C__CATG__NAGIOS_HOST_TPL_ASSIGNED_OBJECTS',
                'C__CATG__NAGIOS_HOST_TPL_DEF',
                'C__CATG__NAGIOS_HOST_TPL_FOLDER',
                'C__CATG__NAGIOS_REFS_SERVICES',
                'C__CATG__NAGIOS_REFS_SERVICES_BACKWARDS',
                'C__CATG__NAGIOS_SERVICE_DEF',
                'C__CATG__NAGIOS_SERVICE_DEP',
                'C__CATG__NAGIOS_SERVICE_FOLDER',
                'C__CATG__NAGIOS_SERVICE_REFS_TPL_BACKWARDS',
                'C__CATG__NAGIOS_SERVICE_TPL_DEF',
                'C__CATG__NAGIOS_SERVICE_TPL_FOLDER'
            ]
        ];

        try {
            foreach ($categoryCollection as $categoryType => $categories) {
                // Do NOT use "$cmdbDao->prepare_in_condition(...)" because it will try to process the constants.
                $sql = 'DELETE FROM isysgui_' . $categoryType . ' 
                    WHERE isysgui_' . $categoryType . '__const IN (' . implode(',', array_map([$cmdbDao, 'convert_sql_text'], $categories)) . ');';

                $cmdbDao->update($sql) && $cmdbDao->apply_update();
            }

            return true;
        } catch (Exception $e) {
            return false;
        }
    }

    /**
     * @deprecated
     */
    public static function get_auth()
    {
        return self::getAuth();
    }

    /**
     * @return isys_auth_nagios
     */
    public static function getAuth()
    {
        return isys_auth_nagios::instance();
    }
}
