<?php

namespace idoit\Module\Packager\Controller;

use Exception;
use idoit\Controller\Base;
use idoit\Controller\NavbarHandable;
use idoit\Module\Packager\Component\Packager;
use idoit\Module\Packager\Model\Addon as AddonModel;
use idoit\Module\Packager\Model\Repository as RepositoryModel;
use idoit\Module\Packager\View\Addon;
use idoit\Module\Packager\View\AddonList;
use idoit\Tree\Node;
use idoit\View\Renderable;
use isys_application as Application;
use isys_auth;
use isys_component_filemanager;
use isys_component_template_navbar as ComponentNavbar;
use isys_component_tree as TreeComponent;
use isys_controller as Controllable;
use isys_exception_database;
use isys_module as Module;
use isys_module_packager as ModulePackager;
use isys_register as Register;

/**
 * Class Main
 *
 * @package   idoit\Module\Packager\Controller
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class Main extends Base implements Controllable, NavbarHandable
{
    /**
     * @var Module
     */
    protected $module;

    /**
     * Default request handler, gets called in every request.
     *
     * @param Register    $request
     * @param Application $application
     *
     * @return void
     * @throws \isys_exception_auth
     */
    public function handle(Register $request, Application $application)
    {
        // Check if the user is allowed to see this add-on.
        ModulePackager::getAuth()->check(isys_auth::VIEW, 'packager');

        $onClick = "document.isys_form.sort.value=''; document.isys_form.navMode.value=this.getAttribute('data-navmode'); $('isys_form').submit();";

        // Reset form_submit url since form_submit is not capable of the new url-rewrite paths..
        ComponentNavbar::getInstance()
            ->set_js_onclick($onClick, C__NAVBAR_BUTTON__SAVE)
            ->set_js_onclick($onClick, C__NAVBAR_BUTTON__EDIT)
            ->set_js_onclick($onClick, C__NAVBAR_BUTTON__CANCEL)
            ->set_js_onclick($onClick, C__NAVBAR_BUTTON__DELETE);
    }

    /**
     * @param Application $application
     *
     * @return AddonModel
     * @throws Exception
     */
    public function dao(Application $application)
    {
        return AddonModel::instance($this->getDi()->get('database'));
    }

    /**
     * @param Register      $request
     * @param Application   $application
     * @param TreeComponent $tree
     *
     * @return Node|\isys_tree_node
     * @throws Exception
     */
    public function tree(Register $request, Application $application, TreeComponent $tree)
    {
        // Disable node sorting.
        $tree->set_tree_sort(false);

        // Prepare the language manager.
        $language = $this->getDi()->get('language');

        // Prepare the root node.
        $rootNode = new Node(
            $language->get('LC__MODULE__PACKAGER'),
            '',
            $this->module::getWwwPath() . 'assets/icons/package_green.png'
        );

        $addonListNode = new Node(
            $language->get('LC__MODULE__PACKAGER__ADD_ON_LIST'),
            $request->get('BASE') . 'packager',
            $this->module::getWwwPath() . 'assets/icons/application_view_list.png'
        );

        $packageListNode = new Node(
            $language->get('LC__MODULE__PACKAGER__ADD_ON_PACKAGES'),
            $request->get('BASE') . 'packager/repository',
            $this->module::getWwwPath() . 'assets/icons/application_cascade.png'
        );

        $addonResult = AddonModel::instance($this->getDi()->get('database'))->getAddonList();

        while ($row = $addonResult->get_row()) {
            $packageListNode->add(new Node(
                $row['title'],
                $request->get('BASE') . 'packager/repository-by-addon/' . $row['id'],
                $this->module::getWwwPath() . 'assets/icons/application.png'
            ));
        }

        return $rootNode
            ->add($addonListNode)
            ->add($packageListNode);
    }

    /**
     * @param Register    $request
     * @param Application $application
     *
     * @return Renderable
     * @throws isys_exception_database
     */
    public function onDefault(Register $request, Application $application)
    {
        if ($request->get('id') > 0) {
            ComponentNavbar::getInstance()->set_active(false, C__NAVBAR_BUTTON__SAVE);

            return $this->onEdit($request, $application);
        }

        return new AddonList($request);
    }

    /**
     * @param Register    $request
     * @param Application $application
     *
     * @return Renderable
     * @throws isys_exception_database
     */
    public function onDuplicate(Register $request, Application $application)
    {
        return $this->onDefault($request, $application);
    }

    /**
     * @param Register    $request
     * @param Application $application
     *
     * @return Renderable
     * @throws isys_exception_database
     */
    public function onCancel(Register $request, Application $application)
    {
        return $this->onDefault($request, $application);
    }

    /**
     * @param Register    $request
     * @param Application $application
     *
     * @return Renderable
     */
    public function onNew(Register $request, Application $application)
    {
        ComponentNavbar::getInstance()
            ->set_active(true, C__NAVBAR_BUTTON__SAVE)
            ->set_active(true, C__NAVBAR_BUTTON__CANCEL);

        return new Addon($request);
    }

    /**
     * @param Register    $request
     * @param Application $application
     *
     * @return Renderable
     * @throws \isys_exception_auth
     * @throws \isys_exception_dao
     */
    public function onSave(Register $request, Application $application)
    {
        // Check for edit right
        // AuthPackager::instance()->hooks(Auth::EDIT);

        $id = null;
        $post = $request->get('POST');

        $data = [
            'title'                     => $post['title'] ?: 'untitled',
            'description'               => $post['description'],
            'manufacturer'              => $post['manufacturer'],
            'website'                   => $post['website'],
            'requiresLicense'           => $post['requiresLicense'] ?: 0,
            'idoitRequirement'          => $post['idoitRequirement'],
            'packagerVersion'           => $post['createdOnPackagerVersion'],
            'assignedObjectTypeGroups'  => $post['objectTypeGroup_selection'],
            'assignedObjectTypes'       => $post['objectType_selection'],
            'assignedCustomCategories'  => $post['customCategory_selection'],
            'selectedDialogPlus'        => $post['dialog_plus_selection'],
            'selectedReports'           => $post['report_selection'],
            'selectedRelationTypes'     => $post['relation_type_selection'],
            'selectedCsvImportProfiles' => $post['csv_import_profile_selection'],
            'updated'                   => 'NOW()',
            'created'                   => 'NOW()',
        ];

        if (isset($post['identifier'])) {
            $data['identifier'] = AddonModel::sanitizeIdentifier($post['manufacturer'] . '_' . $post['identifier']);
        }

        if (isset($post['addonId']) && $post['addonId'] > 0) {
            $id = (int)$post['addonId'];
            unset($data['created']);
        }

        $id = $this->dao($application)->saveAddon($id, $data);

        // Return the view.
        return (new Addon($request))->setID($id);
    }

    /**
     * @param Register    $request
     * @param Application $application
     *
     * @return Renderable
     */
    public function onEdit(Register $request, Application $application)
    {
        if (isset($_POST['id']) && !$request->get('id')) {
            if (is_array($_POST['id'])) {
                $id = reset($_POST['id']);
            } else {
                $id = $_POST['id'];
            }

            // Always work with GET requests.
            \isys_core::send_header('location', $application->www_path . 'packager/main/' . (int)$id);
            die;
        }

        // Return the view.
        $view = new Addon($request);

        // Retrieve id from URL "packager/main/123" (default way).
        if ($request->get('id')) {
            $view->setID((int)$request->get('id'));
        }

        return $view;
    }

    /**
     * @param Register    $request
     * @param Application $application
     *
     * @return Renderable
     * @throws isys_exception_database
     */
    public function onPrint(Register $request, Application $application)
    {
        return $this->onDefault($request, $application);
    }

    /**
     * @param Register    $request
     * @param Application $application
     *
     * @return Renderable
     * @throws isys_exception_database
     */
    public function onPurge(Register $request, Application $application)
    {
        return $this->onDelete($request, $application);
    }

    /**
     * @param Register    $request
     * @param Application $application
     *
     * @return Renderable
     * @throws isys_exception_database
     */
    public function onArchive(Register $request, Application $application)
    {
        return $this->onDelete($request, $application);
    }

    /**
     * @param Register    $request
     * @param Application $application
     *
     * @return Renderable
     * @throws Exception
     * @throws isys_exception_database
     */
    public function onDelete(Register $request, Application $application)
    {
        $post = $request->get('POST');

        if (isset($post['id']) && \is_array($post['id'])) {
            $this->dao($application)->deleteAddon(array_map(function ($id) { return (int)$id; }, $post['id']));
        }

        return $this->onDefault($request, $application);
    }

    /**
     * @param Register    $request
     * @param Application $application
     *
     * @return Renderable
     * @throws isys_exception_database
     */
    public function onQuickPurge(Register $request, Application $application)
    {
        return $this->onDefault($request, $application);
    }

    /**
     * @param Register    $request
     * @param Application $application
     *
     * @return Renderable
     * @throws isys_exception_database
     */
    public function onRecycle(Register $request, Application $application)
    {
        return $this->onDefault($request, $application);
    }

    /**
     * @param Register    $request
     * @param Application $application
     *
     * @return Renderable
     * @throws isys_exception_database
     */
    public function onReset(Register $request, Application $application)
    {
        return $this->onDefault($request, $application);
    }

    /**
     * @param Register    $request
     * @param Application $application
     *
     * @return Renderable
     * @throws isys_exception_database
     */
    public function onUp(Register $request, Application $application)
    {
        return $this->onDefault($request, $application);
    }

    /**
     * Method for downloading a created add-on ZIP.
     *
     * @param Register $request
     *
     * @return void
     * @throws \idoit\Exception\JsonException
     * @throws isys_exception_database
     */
    public function download(Register $request)
    {
        $application = Application::instance();
        $repositoryId = (int)$request->get('id');

        // We do this to check if the repository file even exists.
        $repository = RepositoryModel::instance($this->getDi()->get('database'))
            ->getById($repositoryId)
            ->get_row();

        if (!is_array($repository)) {
            // If someone tries to download a "non-existing" repository, we'll forward them to the add-on index.
            \isys_core::send_header('location', $application->www_path . 'packager/');
            die;
        }

        (new isys_component_filemanager())->send((new Packager($repositoryId, $application))->getZipFilePath());

        die;
    }

    /**
     * Main constructor.
     *
     * @param Module $p_module
     */
    public function __construct(Module $p_module)
    {
        $this->module = $p_module;
    }
}
