<?php

namespace idoit\Module\Packager\View;

use Exception;
use idoit\Exception\JsonException;
use idoit\Model\Dao\Base as DaoBase;
use idoit\Module\Packager\Model\Repository as RepositoryModel;
use idoit\View\Base;
use idoit\View\Renderable;
use isys_application;
use isys_auth;
use isys_component_template as ComponentTemplate;
use isys_exception_database;
use isys_module as ModuleBase;
use isys_module_packager;

/**
 * Class Repository
 *
 * @package   idoit\Module\Packager\View
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class Repository extends Base
{
    /**
     * @param ModuleBase        $module
     * @param ComponentTemplate $template
     * @param DaoBase           $model
     *
     * @return $this|Renderable
     * @throws Exception
     * @throws JsonException
     * @throws isys_exception_database
     */
    public function process(ModuleBase $module, ComponentTemplate $template, DaoBase $model)
    {
        $language = isys_application::instance()->container->get('language');
        $database = isys_application::instance()->container->get('database');
        $locales = isys_application::instance()->container->get('locales');

        $editMode = true; // isys_glob_is_edit_mode();
        $isAllowedToEdit = true; // $module::getAuth()->is_allowed_to(Auth::EDIT, 'hooks');
        $referenceOptionData = [];

        // Check if the user is allowed to see this page.
        isys_module_packager::getAuth()
            ->check(isys_auth::EXECUTE, 'packager');

        $repositoryModel = RepositoryModel::instance($database);
        $repositoryData = $repositoryModel->getById($this->id)->get_row();

        $previousChangelogEntries = [];
        $changelogResult = $repositoryModel->getChangelogOfAddon((int)$repositoryData['addonId'], (int)$this->id);

        while ($changelogRow = $changelogResult->get_row()) {
            $previousChangelogEntries[] = $changelogRow['changelog'];
        }

        $rules = [
            'identifier'               => [
                'p_strValue'  => $repositoryData['identifier'],
                'p_nMaxLen'   => 32,
                'p_bDisabled' => !empty($repositoryData['identifier'])
            ],
            'title'                    => [
                'p_strValue' => $repositoryData['title']
            ],
            'manufacturer'             => [
                'p_strValue' => $repositoryData['manufacturer']
            ],
            'website'                  => [
                'p_strValue' => $repositoryData['website'],
                'p_strPlaceholder' => 'https:// ...'
            ],
            'description'              => [
                'p_strValue' => $repositoryData['description']
            ],
            'idoitRequirement'         => [
                'p_strValue'       => $repositoryData['idoitRequirement'] ?: '1.11',
                'p_strPlaceholder' => '1.11'
            ],
            'createdOnPackagerVersion' => [
                'p_strValue'  => trim(file_get_contents($module::getPath() . 'VERSION')),
                'p_bReadonly' => true
            ],
            'requiresLicense'          => [
                'p_arData'        => get_smarty_arr_YES_NO(),
                'p_strSelectedID' => $repositoryData['requiresLicense'],
                'p_bDbFieldNN'    => true
            ],
            'includeDefinition'        => [
                'p_arData'        => get_smarty_arr_YES_NO(),
                'p_strSelectedID' => $repositoryData['includeDefinition'],
                'p_bDbFieldNN'    => true
            ],
            'version'               => [
                'p_strValue' => $repositoryData['version'],
                'p_strClass' => 'input-mini'
            ],
            'releaseDate'              => [
                'p_strValue' => $locales->fmt_datetime($repositoryData['created']),
                'p_strClass' => 'input-small',
                'p_strStyle' => 'width:70%;',
                'p_bTime'    => true
            ]
        ];

        // Assign template variables.
        $template
            ->assign('baseUrl', isys_application::instance()->www_path)
            ->assign('packagerPath', $module::getPath())
            ->assign('packagerWwwPath', $module::getWwwPath())
            ->assign('repositoryId', $this->id)
            ->assign('changelog', implode(PHP_EOL . PHP_EOL, $previousChangelogEntries))
            ->smarty_tom_add_rules('tom.content.bottom.content', $rules);

        // Set paths to templates.
        $this->paths['contentbottomcontent'] = $module::getPath() . 'templates/repository.tpl';

        return $this;
    }
}
