/**
 * i-doit object radius editor javascript class.
 *
 * @author  Leonard Fischer <lfischer@i-doit.com>
 */
window.FileSelector = Class.create({
    options: {},
    
    initialize: function ($el, options) {
        this.processing = true;
        this.currentId = null;
        
        this.$element = $el;
        this.data = [];
        this.options = {
            addonId: null,
            editMode: false,
            $itemCounter: null
        };
        
        Object.extend(this.options, options || {});
        //return;
        this.prepareGui();
        this.prepareObserver();
        this.prepareFileUploader();
        this.retrieveData();
        
        // Simulate a "click" on the first radiobox to trigger some internal logic.
        this.$element.down('[name="file-selector-type"]').setValue(1).simulate('change');
    },
    
    prepareGui: function () {
        this.$newFileButton = new Element('button', {type:'button', className:'btn mb5'})
            .update(new Element('img', {src:window.dir_images + 'axialis/basic/symbol-add.svg', alt: ''}))
            .insert(new Element('span').update(idoit.Translate.get('LC__MODULE__PACKAGER__FILE__LABEL_NEW_FILE')));
        this.$fileList = new Element('ul', {className: 'file-list list-style-none m0 p0'});
        this.$fileDetailOverlay = new Element('div', {className:'overlay'});
        this.$fileDetailUpload = new Element('div');
        this.$fileDetailName = new Element('input', {type:'text', name:'file-selector-filename', className:'input input-size-small'});
        this.$fileDetailDescription = new Element('textarea', {name:'file-selector-description', className:'input input-size-medium noresize'});
        this.$fileDetailCustomPath = new Element('input', {name:'file-selector-custom-path', className:'input', placeholder:'src/'});
        this.$fileDetailSaveButton = new Element('button', {type:'button', className:'btn mt10 mr5'})
            .update(new Element('img', {src:window.dir_images + 'axialis/basic/symbol-ok.svg', alt:''}))
            .insert(new Element('span').update(idoit.Translate.get('LC_UNIVERSAL__ACCEPT')));
        this.$fileDetailCancelButton = new Element('button', {type:'button', className:'btn mt10 mr5'})
            .update(new Element('img', {src:window.dir_images + 'axialis/basic/symbol-cancel.svg', alt:''}))
            .insert(new Element('span').update(idoit.Translate.get('LC_UNIVERSAL__ABORT')));
        this.$fileDetailDeleteButton = new Element('button', {type:'button', className:'btn btn-red mt10 hide'})
            .update(new Element('img', {src:window.dir_images + 'axialis/basic/button-remove.svg', alt:''}))
            .insert(new Element('span').update(idoit.Translate.get('LC_UNIVERSAL__DELETE')));
    
        this.$element
            .update(this.$newFileButton)
            .insert(new Element('br', {className:'br'}))
            .insert(new Element('div', {className:'file-list-container'})
                .update(this.$fileList))
            .insert(new Element('div', {className:'file-details-container'})
                .update(this.$fileDetailOverlay)
                .insert(new Element('label', { className: 'mb5' })
                    .update(new Element('span', { className: 'display-block mb5' }).update(idoit.Translate.get('LC__MODULE__PACKAGER__FILE__LABEL_UPLOAD')))
                    .insert(this.$fileDetailUpload))
                .insert(new Element('label', { className: 'mb5' })
                    .update(new Element('span', { className: 'display-block mb5' }).update(idoit.Translate.get('LC__MODULE__PACKAGER__FILE__LABEL_FILENAME')))
                    .insert(this.$fileDetailName))
                .insert(new Element('label', { className: 'mb15' })
                    .update(new Element('span', { className: 'display-block mb5' }).update(idoit.Translate.get('LC__MODULE__PACKAGER__FILE__LABEL_DESCRIPTION')))
                    .insert(this.$fileDetailDescription))
                .insert(new Element('label', { className: 'mb5' })
                    .update(new Element('input', {type:'radio', name:'file-selector-type', value:'translation', className:'mr5'}))
                    .insert(new Element('span').update(idoit.Translate.get('LC__MODULE__PACKAGER__FILE__TYPE_TRANSLATION'))))
                .insert(new Element('label', { className: 'mb5' })
                    .update(new Element('input', {type:'radio', name:'file-selector-type', value:'importCsv', className:'mr5'}))
                    .insert(new Element('span').update(idoit.Translate.get('LC__MODULE__PACKAGER__FILE__TYPE_IMPORT_CSV'))))
                .insert(new Element('label', {style: 'height:50px !important;'})
                    .update(new Element('input', {type:'radio', name:'file-selector-type', value:'custom', className:'mr5'}))
                    .insert(new Element('span').update(idoit.Translate.get('LC__MODULE__PACKAGER__FILE__TYPE_CUSTOM')))
                    .insert(new Element('br'))
                    .insert(new Element('div', {className:'ml20 input-group input-size-small'})
                        .update(new Element('div', {className:'input-group-addon input-group-addon-unstyled'})
                            .update(new Element('code').update('./')))
                        .insert(this.$fileDetailCustomPath)
                        .insert(new Element('div', {className:'input-group-addon input-group-addon-unstyled'})
                            .update(new Element('code').update('filename.ext')))))
                .insert(this.$fileDetailSaveButton)
                .insert(this.$fileDetailCancelButton)
                .insert(this.$fileDetailDeleteButton));
    },
    
    prepareObserver: function () {
        var self = this;
    
        this.$newFileButton.on('click', function () {
            self.processFileDetails();
        });
        
        this.$element.on('click', '.file-list button', function (ev) {
            var $li = ev.findElement('li');
            
            // Don't do anything while processing.
            if (self.processing) {
                idoit.Notify.info(idoit.Translate.get('LC__UNIVERSAL__LOADING'));
                
                return;
            }
            
            self.processFileDetails($li.readAttribute('data-id'));
        });
    
        this.$element.on('change', '[name="file-selector-type"]', function (ev) {
            self.$fileDetailCustomPath.setValue('');
            self.$fileDetailCustomPath.disable();
            
            if (ev.findElement('input').getValue() === 'custom') {
                self.$fileDetailCustomPath.enable();
            }
        });
    
        this.$fileDetailName.on('change', this.updateFilename.bind(this));
        
        // Check if the given path is "valid".
        this.$fileDetailCustomPath.on('change', function () {
            var sourceValue, targetValue;
        
            sourceValue = self.$fileDetailCustomPath.getValue();
            
            // Clean up the string a little.
            targetValue = self.cleanFilepath(sourceValue);
            
            if (sourceValue !== targetValue) {
                idoit.Notify.info('The path was cleaned from <code>' + sourceValue + '</code> to <code>' + targetValue + '</code>.', {life: 10});
                
                self.$fileDetailCustomPath.setValue(targetValue);
            }
        });
        
        this.$fileDetailSaveButton.on('click', function () {
            self.saveFile();
        });
    
        this.$fileDetailCancelButton.on('click', function () {
            self.reset();
        });
    
        this.$fileDetailDeleteButton.on('click', function () {
            self.deleteFile();
        })
    },
    
    reset: function () {
        // Reset the details fields.
        this.$fileDetailName.setValue('');
        this.$fileDetailDescription.setValue('');
        this.$fileDetailCustomPath.setValue('');
        this.$fileDetailCustomPath.disable();
        this.$element.select('[name="file-selector-type"]').invoke('setValue', 0);
        this.$element.down('[name="file-selector-type"]').setValue(1);
        this.$element.select('.file-list li').invoke('removeClassName', 'selected');
        this.$fileDetailUpload.down('.qq-upload-button').childNodes[0].nodeValue = idoit.Translate.get('LC__MODULE__PACKAGER__FILE__LABEL_UPLOAD_BUTTON');
        this.$fileDetailDeleteButton.addClassName('hide');
        this.uploader.clearStoredFiles();
        
        // Display the overlay.
        this.$fileDetailOverlay.removeClassName('hide');
    
        this.processing = false;
        this.currentId = null;
    },
    
    updateFilename: function () {
        this.$fileDetailCustomPath.next('div').down('code').update(this.$fileDetailName.getValue());
    },
    
    cleanFilepath: function (path) {
        // Clean up the string a little.
        path = path.replace(/[^a-z0-9_\-\/\.]/g, '');
    
        if (path.startsWith('/')) {
            path = '.' + path;
        }
    
        // Replace strings like './' and '../'.
        path = path.replace(/\.\.\//g, '');
        path = path.replace(/\.\//g, '');
        
        return path;
    },
    
    prepareFileUploader: function () {
        var self = this;
        
        this.uploader = new qq.FileUploader({
            element: this.$fileDetailUpload,
            action: window.www_dir + 'packager/ajax/uploadFile',
            multiple: false,
            autoUpload: false,
            params: {},
            onSubmit: function (id, fileName) {
                self.$fileDetailName.setValue(fileName);
                self.updateFilename();
                
                // Since we only want to replace the "text" itself, we have to use this:
                self.$fileDetailUpload.down('.qq-upload-button').childNodes[0].nodeValue = idoit.Translate.get('LC__MODULE__PACKAGER__FILE__LABEL_UPLOAD_CHANGE').replace('%s', fileName);
                
                return true;
            },
            onComplete: function (id, fileName, result) {
                // Set the VIEW and HIDDEN field accordingly
                self.uploader.clearStoredFiles();
                
                // Since we only want to replace the "text" itself, we have to use this:
                self.$fileDetailUpload.down('.qq-upload-button').childNodes[0].nodeValue = idoit.Translate.get('LC__MODULE__PACKAGER__FILE__LABEL_UPLOAD_BUTTON');
                
                self.retrieveData();
            },
            onError: function (id, fileName, xhr) {
                // Since we only want to replace the "text" itself, we have to use this:
                self.$fileDetailUpload.down('.qq-upload-button').childNodes[0].nodeValue = idoit.Translate.get('LC__MODULE__PACKAGER__FILE__LABEL_UPLOAD_BUTTON');
                
                // Error handling.
                idoit.Notify.error(xhr.responseText, {sticky: true});
            },
            dragText: idoit.Translate.get('LC__MODULE__PACKAGER__FILE__LABEL_DRAG_TEXT'),
            uploadButtonText: idoit.Translate.get('LC__MODULE__PACKAGER__FILE__LABEL_UPLOAD_BUTTON'),
            failUploadText: 'upload error',
            multipleFileDropNotAllowedMessage: idoit.Translate.get('LC_FILEBROWSER__SINGLE_FILE_UPLOAD')
        })
    },
    
    retrieveData: function () {
        var self = this;
        
        new Ajax.Request(window.www_dir + 'packager/ajax/loadFiles', {
            parameters: {
                addonId: self.options.addonId
            },
            onComplete: function (xhr) {
                try {
                    if (!is_json_response(xhr)) {
                        throw 'Could not retrieve files.';
                    }
                    
                    var json = xhr.responseJSON;
                    
                    if (!json.success) {
                        throw 'The file retrieval did not finish successfully.';
                    }
                    
                    self.data = json.data;
                    
                    self.reset();
                    self.process();
                } catch (e) {
                    idoit.Notify.error(e, {sticky: true});
                }
            }
        });
    },
    
    process: function () {
        var i;
        
        this.$fileList.update();
        
        for (i in this.data) {
            if (!this.data.hasOwnProperty(i)) {
                continue;
            }
            
            this.renderListItem(this.data[i]);
        }
    
        if (Object.isElement(this.options.$itemCounter)) {
            this.options.$itemCounter.update(this.data.length + ' ' + idoit.Translate.get(this.data.length === 1 ? 'LC__MODULE__PACKAGER__FILE' : 'LC__MODULE__PACKAGER__FILES'))
        }
        
        this.processing = false;
    },
    
    processFileDetails: function (fileId) {
        var data;
    
        this.reset();
        
        if (fileId !== undefined) {
            this.currentId = fileId;
    
            this.$element.down('.file-list li[data-id="' + this.currentId + '"]').addClassName('selected');
            data = this.data.filter(function(item) { return item.id == fileId; });
            
            if (data.length) {
                data = data[0];
                
                this.$fileDetailName.setValue(data.filename);
                this.$fileDetailDescription.setValue(data.description);
                this.$fileDetailDeleteButton.removeClassName('hide');
                
                this.$element.down('[name="file-selector-type"][value="' + data.type + '"]').setValue(1);
                
                if (data.type === 'custom') {
                    this.$fileDetailCustomPath.enable();
                    this.$fileDetailCustomPath.setValue(this.cleanFilepath(data.path));
                }
            }
        }
    
        this.$fileDetailOverlay.addClassName('hide');
    },
    
    renderListItem: function (item) {
        var $actionButton = null,
            typeElement = (item.type === 'custom' ? 'code' : 'span'),
            typeSpacing = (item.type === 'custom' ? '' : 'mr5');
        
        if (this.options.editMode) {
            $actionButton = new Element('button', {type: 'button', className: 'btn btn-small btn-secondary'})
                .update(new Element('img', {src: window.dir_images + 'axialis/basic/tool-pencil.svg'}));
        }
        
        this.$fileList
            .insert(new Element('li', {'data-id': item.id})
                .update(new Element(typeElement, {className:typeSpacing + ' text-neutral-400'}).update((item.type === 'custom' ? item.path : item.translatedType)))
                .insert(new Element('code', {className:'text-black text-bold'}).update(item.filename))
                .insert($actionButton));
    },
    
    saveFile: function () {
        var self = this,
            fileTitle = this.$fileDetailName.getValue(),
            fileDescription = this.$fileDetailDescription.getValue(),
            fileType = this.$element.down('[name="file-selector-type"]:checked').getValue();
        
        if (this.currentId === null && (fileTitle.blank() || this.uploader._storedFileIds.length === 0)) {
            idoit.Notify.error(idoit.Translate.get('LC__MODULE__PACKAGER__FILE__PROVIDE_FILE_AND_TITLE'));
            
            return;
        }
        
        this.processing = true;
        
        if (fileType === 'custom') {
            fileTitle = this.$fileDetailCustomPath.getValue() + '/' + fileTitle;
        }
        
        new Ajax.Request(window.www_dir + 'packager/ajax/saveFile', {
            parameters: {
                id: this.currentId,
                addonId: this.options.addonId,
                title: fileTitle,
                description: fileDescription,
                type: fileType
            },
            onComplete: function (xhr) {
                try {
                    if (!is_json_response(xhr)) {
                        throw 'Could not save file.';
                    }
                
                    var json = xhr.responseJSON;
                
                    if (!json.success) {
                        throw 'The file could not be saved successfully: ' + json.message;
                    }
    
                    // The "data" field will contain the newly created (or updated) file ID.
                    self.uploader.setParams({
                        fileId: json.data
                    });
                    
                    // When all is good we upload the file and attach it to the DB entry.
                    if (self.uploader._storedFileIds.length) {
                        self.uploader.uploadStoredFiles();
                    } else {
                        self.retrieveData();
                    }
                } catch (e) {
                    idoit.Notify.error(e, {sticky: true});
                }
            }
        });
    },
    
    deleteFile: function () {
        var self = this;
        
        if (!this.currentId || !confirm(idoit.Translate.get('LC__MODULE__PACKAGER__FILE__CONFIRM_DELETION'))) {
            return;
        }
    
        new Ajax.Request(window.www_dir + 'packager/ajax/deleteFile', {
            parameters: {
                id: this.currentId
            },
            onComplete: function (xhr) {
                try {
                    if (!is_json_response(xhr)) {
                        throw 'Could not remove file.';
                    }
                
                    var json = xhr.responseJSON;
                
                    if (!json.success) {
                        throw 'The file could not be removed successfully: ' + json.message;
                    }
                
                    self.retrieveData();
                } catch (e) {
                    idoit.Notify.error(e, {sticky: true});
                }
            }
        });
    }
});
