<?php

namespace idoit\Module\Packager\Controller;

use DateTime;
use Exception;
use idoit\Controller\Base;
use idoit\Module\Packager\Component\Packager;
use idoit\Module\Packager\Component\Repository;
use idoit\Module\Packager\Model\CsvImportProfile;
use idoit\Module\Packager\Model\CustomCategory;
use idoit\Module\Packager\Model\Dao;
use idoit\Module\Packager\Model\DialogPlus;
use idoit\Module\Packager\Model\File;
use idoit\Module\Packager\Model\ObjectType;
use idoit\Module\Packager\Model\ObjectTypeGroup;
use idoit\Module\Packager\Model\RelationType;
use idoit\Module\Packager\Model\Reports;
use idoit\Tree\Node;
use isys_application as Application;
use isys_auth;
use isys_component_dao_result;
use isys_component_tree as TreeComponent;
use isys_controller as Controllable;
use isys_format_json as JSON;
use isys_library_fileupload;
use isys_module as Module;
use isys_module_packager as ModulePackager;
use isys_register as Register;

/**
 * Class Ajax
 *
 * @package   idoit\Module\Packager\Controller
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class Ajax extends Base implements Controllable
{
    /**
     * @var Module
     */
    protected $module;

    /**
     * @var array
     */
    protected $response;

    /**
     * Pre method gets called by the framework.
     */
    public function pre()
    {
        header('Content-Type: application/json');

        $this->response = [
            'success' => true,
            'data'    => null,
            'message' => null
        ];
    }

    /**
     * Default request handler, gets called in every /packager/ajax request.
     *
     * @param Register    $request
     * @param Application $application
     *
     * @return null
     * @throws \isys_exception_auth
     */
    public function handle(Register $request, Application $application)
    {
        // Check if the user is allowed to see this add-on.
        ModulePackager::getAuth()->check(isys_auth::VIEW, 'packager');
    }

    /**
     * @param Register $request
     */
    public function loadData(Register $request)
    {
        try {
            $data = [];
            $language = $this->getDi()->get('language');
            $database = $this->getDi()->get('database');

            $callback = $request->get('POST')->get('callback');

            switch ($callback) {
                case 'loadRelationTypes':
                    $result = RelationType::instance($database)->retrieveForGui();
                    break;

                case 'loadObjectTypeGroups':
                    $result = ObjectTypeGroup::instance($database)->retrieveForGui();
                    break;

                case 'loadObjectTypes':
                    $result = ObjectType::instance($database)->retrieveForGui();
                    break;

                case 'loadCustomCategories':
                    $result = CustomCategory::instance($database)->retrieveForGui();
                    break;

                case 'loadReports':
                    $result = Reports::instance($database)->retrieveForGui();
                    break;

                case 'loadDialogPlus':
                    $result = DialogPlus::instance($database)->retrieveForGui();
                    break;

                case 'loadCsvImportProfiles':
                    $result = CsvImportProfile::instance($database)->retrieveForGui();
                    break;
            }

            if ($result instanceof isys_component_dao_result) {
                while ($row = $result->get_row()) {
                    $row['title'] = $language->get($row['title']);

                    $data[] = $row;
                }
            } elseif (is_array($result)) {
                foreach ($result as $item) {
                    $item['title'] = $language->get($item['title']);

                    $data[] = $item;
                }
            }

            usort($data, function ($a, $b) {
                return strnatcasecmp($a['title'], $b['title']);
            });

            $this->response['data'] = $data;
        } catch (Exception $e) {
            $this->response['success'] = false;
            $this->response['data'] = null;
            $this->response['message'] = $e->getMessage();
        }
    }

    /**
     * @param Register $request
     */
    public function loadFiles(Register $request)
    {
        try {
            $data = [];
            $database = $this->getDi()->get('database');

            $addonId = (int)$request->get('POST')->get('addonId');

            $result = File::instance($database)->getFileList($addonId);

            while ($row = $result->get_row()) {
                $row['path'] = \dirname($row['title']) . '/';
                $row['filename'] = basename($row['title']);
                $row['translatedType'] = File::getTranslatedType($row['type']);

                $data[] = $row;
            }

            usort($data, function ($a, $b) {
                return strnatcasecmp($a['type'] . '...' . $a['title'], $b['type'] . '...' . $b['title']);
            });

            $this->response['data'] = $data;
        } catch (Exception $e) {
            $this->response['success'] = false;
            $this->response['data'] = null;
            $this->response['message'] = $e->getMessage();
        }
    }

    /**
     * @param Register $request
     */
    public function saveFile(Register $request)
    {
        try {
            $data = [];
            $database = $this->getDi()->get('database');

            $id = null;
            $postData = (array)$request->get('POST');

            if (isset($postData['id']) && is_numeric($postData['id'])) {
                $id = (int)$postData['id'];
            }

            $data = [
                'addonId'     => $postData['addonId'],
                'title'       => $postData['title'],
                'created'     => 'NOW()',
                'updated'     => 'NOW()',
                'description' => $postData['description'],
            ];

            if (isset($postData['type'])) {
                // This will throw an exception, if a wrong type has been provided.
                File::getTranslatedType($postData['type']);

                $data['type'] = $postData['type'];

                // If we have a specific "type" we'll set the title accordingly.
                if (isset($postData['title'])) {
                    $data['title'] = File::getPathByType($postData['type']) . $postData['title'];
                }
            }

            if ($id !== null) {
                unset($data['created']);
            }

            $this->response['data'] = File::instance($database)->saveFile($id, $data);
        } catch (Exception $e) {
            $this->response['success'] = false;
            $this->response['data'] = null;
            $this->response['message'] = $e->getMessage();
        }
    }

    /**
     * @param Register $request
     */
    public function deleteFile(Register $request)
    {
        try {
            $database = $this->getDi()->get('database');

            $postData = (array)$request->get('POST');

            if (!isset($postData['id']) || !is_numeric($postData['id'])) {
                throw new Exception('You need to provide at least one file ID.');
            }

            $this->response['data'] = File::instance($database)->deleteFile((int)$postData['id']);
        } catch (Exception $e) {
            $this->response['success'] = false;
            $this->response['data'] = null;
            $this->response['message'] = $e->getMessage();
        }
    }

    /**
     * @param Register $request
     */
    public function uploadFile(Register $request)
    {
        try {
            $data = [];
            $fileId = (int)$request->get('GET')->get('fileId');
            $database = $this->getDi()->get('database');

            $uploader = new isys_library_fileupload;

            $prefix = 'packager-upload-';
            $filepath = rtrim(isys_glob_get_temp_dir(), '/') . '/';
            $filename = $prefix . $uploader->getName();

            $result = $uploader
                ->set_prefix($prefix)
                ->handleUpload($filepath, true);

            if (isset($result['error'])) {
                throw new Exception($result['error']);
            }

            if (!file_exists($filepath . $filename)) {
                throw new Exception('The file "' . $filename . '" could not be uploaded properly.');
            }

            $this->response['data'] = File::instance($database)->saveFile($fileId, [
                'content' => file_get_contents($filepath . $filename)
            ]);

            // After saving the uploaded file to the DB we remove the file.
            @unlink($filepath . $filename);
        } catch (Exception $e) {
            $this->response['success'] = false;
            $this->response['data'] = null;
            $this->response['message'] = $e->getMessage();
        }
    }

    /**
     * @param Register $request
     */
    public function createPackage(Register $request)
    {
        try {
            /*
             * Workflow for creating new packages:
             *
             * 1 - Create a repository entry from current "live data".
             * 2 - Create a package, based on that repository entry
             *
             * This way we can later re-built older add-on versions from the repository.
             */
            $application = Application::instance();
            $releaseDate = new DateTime($request->get('POST')->get('releaseDate'));

            $version = $request->get('POST')->get('version');
            $changelogEntry = trim($request->get('POST')->get('changelog'));

            if (empty($changelogEntry)) {
                $changelogEntry = '-';
            }

            $changelog = $version . PHP_EOL .
                '--------------------------------------------------------------------------------' . PHP_EOL .
                $changelogEntry;

            $repositoryEntryId = (new Repository($application))
                ->createEntryByAddon(
                    (int)$request->get('POST')->get('addonId'),
                    $version,
                    $changelog,
                    $releaseDate
                );

            $this->response['data'] = (new Packager(
                $repositoryEntryId,
                $application
            ))->createPackage();
        } catch (Exception $e) {
            $this->response['success'] = false;
            $this->response['data'] = null;
            $this->response['message'] = $e->getMessage();
        }
    }

    /**
     * @param Register $request
     */
    public function recreatePackage(Register $request)
    {
        try {
            $packager = new Packager(
                (int)$request->get('POST')->get('repositoryId'),
                Application::instance()
            );

            // If the ZIP file does not exist, we'll create it.
            if (!file_exists($packager->getZipFilePath())) {
                $packager->createPackage();
            }

            $this->response['data'] = true;
        } catch (Exception $e) {
            $this->response['success'] = false;
            $this->response['data'] = null;
            $this->response['message'] = $e->getMessage();
        }
    }

    /**
     * @param Application $application
     *
     * @return Dao
     * @throws Exception
     */
    public function dao(Application $application)
    {
        return Dao::instance($this->getDi()->get('database'));
    }

    /**
     * @param Register      $request
     * @param Application   $application
     * @param TreeComponent $tree
     *
     * @return Node|\isys_tree_node
     * @throws Exception
     */
    public function tree(Register $request, Application $application, TreeComponent $tree)
    {
        return null;
    }

    /**
     * @param Module $p_module
     */
    public function __construct(Module $p_module)
    {
        $this->module = $p_module;
    }

    /**
     * Post method gets called by the framework.
     */
    public function post()
    {
        echo JSON::encode($this->response);
        die;
    }
}
