<?php

namespace idoit\Module\Packager\Component;

use DateTime;
use Exception;
use idoit\Exception\JsonException;
use idoit\Module\Packager\Model\Addon as AddonModel;
use idoit\Module\Packager\Model\CsvImportProfile;
use idoit\Module\Packager\Model\CustomCategory;
use idoit\Module\Packager\Model\DialogPlus;
use idoit\Module\Packager\Model\ObjectType;
use idoit\Module\Packager\Model\ObjectTypeGroup;
use idoit\Module\Packager\Model\RelationType;
use idoit\Module\Packager\Model\Reports;
use idoit\Module\Packager\Model\Repository as RepositoryModel;
use idoit\Module\Packager\Model\FileRepository as FileRepositoryModel;
use isys_application;
use isys_component_database;
use isys_exception_dao;
use isys_exception_database;
use isys_format_json;

/**
 * Class Repository
 *
 * @package   idoit\Module\Packager\Component
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class Repository
{
    /**
     * @var isys_component_database
     */
    private $database;

    /**
     * @var AddonModel
     */
    private $addonModel;

    /**
     * @var RepositoryModel
     */
    private $repositoryModel;

    /**
     * @var FileRepositoryModel
     */
    private $fileRepositoryModel;

    /**
     * Repository constructor.
     *
     * @param isys_application $application
     *
     * @throws Exception
     */
    public function __construct(isys_application $application)
    {
        $this->database = $application->container->get('database');

        $this->addonModel = AddonModel::instance($this->database);
        $this->repositoryModel = RepositoryModel::instance($this->database);
        $this->fileRepositoryModel = FileRepositoryModel::instance($this->database);
    }

    /**
     * @param int      $addonId
     * @param string   $version
     * @param string   $changelog
     * @param DateTime $releaseDate
     *
     * @return int
     * @throws isys_exception_dao
     * @throws isys_exception_database
     * @throws JsonException
     */
    public function createEntryByAddon(int $addonId, string $version, string $changelog, DateTime $releaseDate): int
    {
        // Collect and process meta data (the two database tables have a very similiar structure).
        $addonMetaData = $this->addonModel->getById($addonId)->get_row();
        $customCategories = [];

        $addonMetaData['addonId'] = $addonId;
        $addonMetaData['created'] = $releaseDate->format('Y-m-d H:i:s');
        $addonMetaData['version'] = $version;
        $addonMetaData['changelog'] = $changelog;

        // Collect object type group data.
        if (isys_format_json::is_json_array($addonMetaData['assignedObjectTypeGroups'])) {
            $addonMetaData['assignedObjectTypeGroups'] = $this->collectObjectTypeGroupData(isys_format_json::decode($addonMetaData['assignedObjectTypeGroups']));
        } else {
            unset($addonMetaData['assignedObjectTypeGroups']);
        }

        // Collect custom category data.
        if (isys_format_json::is_json_array($addonMetaData['assignedCustomCategories'])) {
            $customCategories = isys_format_json::decode($addonMetaData['assignedCustomCategories']);
        } else {
            unset($addonMetaData['assignedCustomCategories']);
        }

        // Collect object type data.
        if (isys_format_json::is_json_array($addonMetaData['assignedObjectTypes'])) {
            $addonMetaData['assignedObjectTypes'] = $this->collectObjectTypeData(isys_format_json::decode($addonMetaData['assignedObjectTypes']));

            // Also collect assigned custom categories seperately, so they get collected correctly in the next step.
            foreach ($addonMetaData['assignedObjectTypes'] as $objectType) {
                $customCategories = array_merge($customCategories, $objectType['customCategories']);
            }
        } else {
            unset($addonMetaData['assignedObjectTypes']);
        }

        // Collect custom category data.
        if (!empty($customCategories) && \is_array($customCategories)) {
            $addonMetaData['assignedCustomCategories'] = $this->collectCustomCategoryData(array_unique($customCategories));
        } else {
            unset($addonMetaData['assignedCustomCategories']);
        }

        // Collect report data.
        if (isys_format_json::is_json_array($addonMetaData['selectedReports'])) {
            $addonMetaData['selectedReports'] = $this->collectReportData(isys_format_json::decode($addonMetaData['selectedReports']));
        } else {
            unset($addonMetaData['selectedReports']);
        }

        // Collect dialog-plus data.
        if (isys_format_json::is_json_array($addonMetaData['selectedDialogPlus'])) {
            $addonMetaData['selectedDialogPlus'] = $this->collectDialogPlusData(isys_format_json::decode($addonMetaData['selectedDialogPlus']));
        } else {
            unset($addonMetaData['selectedDialogPlus']);
        }

        // Collect relation type data.
        if (isys_format_json::is_json_array($addonMetaData['selectedRelationTypes'])) {
            $addonMetaData['selectedRelationTypes'] = $this->collectRelationType(isys_format_json::decode($addonMetaData['selectedRelationTypes']));
        } else {
            unset($addonMetaData['selectedRelationTypes']);
        }

        // Collect csv import profile data.
        if (isys_format_json::is_json_array($addonMetaData['selectedCsvImportProfiles'])) {
            $addonMetaData['selectedCsvImportProfiles'] = $this->collectCsvImportProfiles(isys_format_json::decode($addonMetaData['selectedCsvImportProfiles']));
        } else {
            unset($addonMetaData['selectedCsvImportProfiles']);
        }

        unset($addonMetaData['id'], $addonMetaData['identifier'], $addonMetaData['updated'], $addonMetaData['status']);

        // Save the meta data
        $repositoryId = $this->repositoryModel->saveRepository($addonMetaData);

        // Save files to "file repository"
        $this->fileRepositoryModel->transferFilesFromAddonToRepository($addonId, $repositoryId);

        return $repositoryId;
    }

    /**
     * @param array $objectTypeGroupConstants
     *
     * @return array
     * @throws \isys_exception_database
     */
    private function collectObjectTypeGroupData(array $objectTypeGroupConstants): array
    {
        if (empty($objectTypeGroupConstants)) {
            return [];
        }

        $return = [];
        $result = ObjectTypeGroup::instance($this->database)
            ->retrieveByConstants($objectTypeGroupConstants);

        while ($row = $result->get_row()) {
            $return[] = $row;
        }

        return $return;
    }

    /**
     * @param array $objectTypeConstants
     *
     * @return array
     * @throws \isys_exception_database
     */
    private function collectObjectTypeData(array $objectTypeConstants): array
    {
        if (empty($objectTypeConstants)) {
            return [];
        }

        $return = [];
        $objectTypeModel = ObjectType::instance($this->database);
        $result = $objectTypeModel->retrieveByConstants($objectTypeConstants);

        while ($row = $result->get_row()) {
            $row['globalCategories'] = [];
            $row['customCategories'] = [];
            $row['overviewAssignments'] = ['global' => [], 'custom' => []];

            // Collect global categories, assigned to the current object type.
            $globalCategoryResult = $objectTypeModel->retrieveGlobalCategoriesByObjectType((int)$row['id']);

            while ($globalCategoryRow = $globalCategoryResult->get_row()) {
                $row['globalCategories'][] = $globalCategoryRow['constant'];
            }

            // Collect custom categories, assigned to the current object type.
            $customCategoryResult = $objectTypeModel->retrieveCustomCategoriesByObjectType((int)$row['id']);

            while ($customCategoryRow = $customCategoryResult->get_row()) {
                $row['customCategories'][] = $customCategoryRow['constant'];
            }

            // @see  PACKAGER-10  Collect global category overview page assignments.
            $globalCategoryAssignmentResult = $objectTypeModel->retrieveGlobalOverviewPageAssignments((int)$row['id']);

            while ($globalCategoryAssignmentRow = $globalCategoryAssignmentResult->get_row()) {
                $row['overviewAssignments']['global'][] = $globalCategoryAssignmentRow['constant'];
            }

            // @see  PACKAGER-10  Collect custom category overview page assignments.
            $customCategoryAssignmentResult = $objectTypeModel->retrieveCustomOverviewPageAssignments((int)$row['id']);

            while ($customCategoryAssignmentRow = $customCategoryAssignmentResult->get_row()) {
                $row['overviewAssignments']['custom'][] = $customCategoryAssignmentRow['constant'];
            }

            // We do not need the ID any longer.
            unset($row['id']);

            $return[] = $row;
        }

        return $return;
    }

    /**
     * @param array $customCategoryConstants
     *
     * @return array
     * @throws \isys_exception_database
     */
    private function collectCustomCategoryData(array $customCategoryConstants): array
    {
        if (empty($customCategoryConstants)) {
            return [];
        }

        $return = [];
        $customCategoryModel = CustomCategory::instance($this->database);
        $result = $customCategoryModel->retrieveByConstants($customCategoryConstants);

        while ($row = $result->get_row()) {
            // We also wan't to include all object types the custom category is assigned to.
            $assignedObjectTypes = [];
            $objectTypeResult = $customCategoryModel->retrieveAssignedObjectTypes($row['constant']);

            while ($objectTypeRow = $objectTypeResult->get_row()) {
                $assignedObjectTypes[] = $objectTypeRow['constant'];
            }

            $row['objectTypeAssignment'] = $assignedObjectTypes;

            // @see PACKAGER-18  It is necessary to rewrite the "identifier" when displaying reports, since we can not work with IDs.
            // @todo PACKAGER-40 Update this code to use 'Unserialize::toArray()' after i-doit 25 release.
            $configuration = unserialize($row['configuration'], ['allowed_classes' => false]);

            // If this happenes, the configuration was not okay.
            if (!\is_array($configuration)) {
                continue;
            }

            foreach ($configuration as &$fieldConfiguration) {
                if ($fieldConfiguration['type'] === 'f_popup' && $fieldConfiguration['popup'] === 'report_browser' && is_numeric($fieldConfiguration['identifier'])) {
                    // We convert the ID to the corresponding constant.
                    $query = 'SELECT isys_report__const AS constant
                    FROM isys_report
                    WHERE isys_report__id = ' . $customCategoryModel->convert_sql_id($fieldConfiguration['identifier']) . '
                    LIMIT 1;';

                    $fieldConfiguration['identifier'] = $customCategoryModel->retrieve($query)->get_row_value('constant');
                }
            }

            $row['configuration'] = serialize($configuration);

            $return[] = $row;
        }

        return $return;
    }

    /**
     * @param array $reportConstants
     *
     * @return array
     * @throws isys_exception_database
     */
    private function collectReportData(array $reportConstants): array
    {
        if (empty($reportConstants)) {
            return [];
        }

        $return = [];
        $dao = Reports::instance($this->database);
        $result = $dao->retrieveByConstants($reportConstants);

        while ($row = $result->get_row()) {
            // @see PACKAGER-35 Swap object type IDs with constants
            if (isys_format_json::is_json_array($row['querybuilderData'])) {
                $querybuilderData = isys_format_json::decode($row['querybuilderData']);

                if (isset($querybuilderData['conditions']) && is_array($querybuilderData['conditions'])) {
                    foreach ($querybuilderData['conditions'] as &$conditions) {
                        if (!is_array($conditions)) {
                            continue;
                        }

                        foreach ($conditions as &$condition) {
                            if ($condition['property'] === 'C__CATG__GLOBAL-type' && is_numeric($condition['value'])) {
                                $objectTypeId = $dao->convert_sql_id($condition['value']);

                                $condition['value'] = $dao
                                    ->retrieve("SELECT isys_obj_type__const AS const FROM isys_obj_type WHERE isys_obj_type__id = {$objectTypeId} LIMIT 1;")
                                    ->get_row_value('const');
                            }
                        }
                    }

                    // Fixes a potential frontend bug, where the JSON contains an object instead of an array.
                    $querybuilderData['conditions'] = array_values($querybuilderData['conditions']);

                    // Only use the new querybuilder data, if it was changed.
                    $row['querybuilderData'] = isys_format_json::encode($querybuilderData);
                }
            }

            $return[] = $row;
        }

        return $return;
    }

    /**
     * @param array $dialogPlusTables
     *
     * @return array
     * @throws isys_exception_database
     */
    private function collectDialogPlusData(array $dialogPlusTables): array
    {
        if (empty($dialogPlusTables)) {
            return [];
        }

        $return = [];

        foreach ($dialogPlusTables as $table) {
            $dialogPlusValuesResult = DialogPlus::instance($this->database)->retrieveByTableName($table);

            while ($dialogPlusValuesRow = $dialogPlusValuesResult->get_row()) {
                $dialogPlusValuesRow['sort'] = (int)$dialogPlusValuesRow['sort'];

                $return[$table][] = $dialogPlusValuesRow;
            }
        }

        return $return;
    }

    /**
     * @param array $relationTypeConstants
     *
     * @return array
     * @throws isys_exception_database
     */
    private function collectRelationType(array $relationTypeConstants): array
    {
        if (empty($relationTypeConstants)) {
            return [];
        }

        $return = [];
        $result = RelationType::instance($this->database)->retrieveByConstants($relationTypeConstants);

        while ($row = $result->get_row()) {
            $return[] = $row;
        }

        return $return;
    }

    /**
     * @param array $csvImportProfileIds
     *
     * @return array
     * @throws isys_exception_database
     */
    private function collectCsvImportProfiles(array $csvImportProfileIds): array
    {
        if (empty($csvImportProfileIds)) {
            return [];
        }

        $return = [];
        $result = CsvImportProfile::instance($this->database)->retrieveByIds($csvImportProfileIds);

        while ($row = $result->get_row()) {
            $return[] = $row;
        }

        return $return;
    }
}
