<?php

namespace idoit\Module\Packager\Model;

use idoit\Model\Dao\Base;
use isys_component_dao_result;
use isys_format_json;

/**
 * Class Repository
 *
 * @package   idoit\Module\Packager\Model
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class Repository extends Base
{
    const FIELDS = [
        'isys_packager_repository__id'                  => 'id',
        'isys_packager_repository__isys_packager__id'   => 'addonId',
        'isys_packager_repository__title'               => 'title',
        'isys_packager_repository__version'             => 'version',
        'isys_packager_repository__manufacturer'        => 'manufacturer',
        'isys_packager_repository__website'             => 'website',
        'isys_packager_repository__requires_license'    => 'requiresLicense',
        'isys_packager_repository__idoit_requirement'   => 'idoitRequirement',
        'isys_packager_repository__include_definition'  => 'includeDefinition',
        'isys_packager_repository__created'             => 'created',
        'isys_packager_repository__description'         => 'description',
        'isys_packager_repository__changelog'           => 'changelog',
        'isys_packager_repository__status'              => 'status',
        'isys_packager_repository__object_type_groups'  => 'assignedObjectTypeGroups',
        'isys_packager_repository__object_types'        => 'assignedObjectTypes',
        'isys_packager_repository__custom_categories'   => 'assignedCustomCategories',
        'isys_packager_repository__reports'             => 'selectedReports',
        'isys_packager_repository__dialog_plus'         => 'selectedDialogPlus',
        'isys_packager_repository__relation_types'      => 'selectedRelationTypes',
        'isys_packager_repository__csv_import_profiles' => 'selectedCsvImportProfiles',
    ];

    /**
     * @param int $id
     *
     * @return isys_component_dao_result
     * @throws \isys_exception_database
     */
    public function getById(int $id): isys_component_dao_result
    {
        $selects = $this->selectImplode(self::FIELDS);

        $sql = "SELECT {$selects}, isys_packager__identifier AS identifier
            FROM isys_packager_repository
            INNER JOIN isys_packager ON isys_packager__id = isys_packager_repository__isys_packager__id
            WHERE isys_packager_repository__id = {$id};";

        return $this->retrieve($sql);
    }

    /**
     * @param int $addonId
     * @param int $untilRevisionId
     *
     * @return isys_component_dao_result
     * @throws \isys_exception_database
     */
    public function getChangelogOfAddon(int $addonId, int $untilRevisionId = null): isys_component_dao_result
    {
        $revisionCondition = '';

        if ($untilRevisionId !== null) {
            $revisionCondition = 'AND isys_packager_repository__id <= ' . $this->convert_sql_id($untilRevisionId);
        }

        $sql = "SELECT isys_packager_repository__changelog AS changelog
            FROM isys_packager_repository
            WHERE isys_packager_repository__isys_packager__id = {$addonId}
            {$revisionCondition}
            ORDER BY isys_packager_repository__id DESC;";

        return $this->retrieve($sql);
    }

    /**
     * @param int $addonId
     *
     * @return isys_component_dao_result
     * @throws \isys_exception_database
     */
    public function getLatestVersionOfAddon(int $addonId): isys_component_dao_result
    {
        $sql = "SELECT isys_packager_repository__version AS version
            FROM isys_packager_repository
            WHERE isys_packager_repository__isys_packager__id = {$addonId}
            ORDER BY isys_packager_repository__id DESC
            LIMIT 1;";

        return $this->retrieve($sql);
    }

    /**
     * @return isys_component_dao_result
     * @throws \isys_exception_database
     */
    public function getRepositoryList(int $id): isys_component_dao_result
    {
        $selects = $this->selectImplode(self::FIELDS);

        if ($id > 0) {
            $condition = "WHERE isys_packager_repository__isys_packager__id = {$id}";
        }

        $sql = "SELECT {$selects}, isys_packager__identifier AS identifier
            FROM isys_packager_repository 
            INNER JOIN isys_packager ON isys_packager__id = isys_packager_repository__isys_packager__id
            {$condition}
            ORDER BY isys_packager_repository__id DESC;";

        return $this->retrieve($sql);
    }

    /**
     * @param array    $data
     *
     * @return int
     * @throws \isys_exception_dao
     */
    public function saveRepository(array $data): int
    {
        $fieldValues = [];
        $fieldKeys = array_flip(self::FIELDS);

        foreach ($data as $key => $value) {
            if (!isset($fieldKeys[$key])) {
                continue;
            }

            switch ($key) {
                default:
                    $value = $this->convert_sql_text($value);
                    break;

                case 'addonId':
                    $value = $this->convert_sql_id($value);
                    break;

                case 'created':
                    $value = $this->convert_sql_datetime($value);
                    break;

                case 'status':
                    $value = $this->convert_sql_int($value);
                    break;

                case 'assignedObjectTypeGroups':
                case 'assignedObjectTypes':
                case 'assignedCustomCategories':
                case 'selectedReports':
                case 'selectedDialogPlus':
                case 'selectedRelationTypes':
                case 'selectedCsvImportProfiles':
                    $value = $this->convert_sql_text(isys_format_json::encode($value));
                    break;
            }

            $fieldValues[] = $fieldKeys[$key] . ' = ' . $value;
        }

        $values = implode(', ', $fieldValues);

        // Create context.
        $this->update("INSERT INTO isys_packager_repository SET {$values};");

        // Get the last created ID.
        return $this->get_last_insert_id();
    }

    /**
     * @param int|array $ids
     */
    public function deleteRepository($ids)
    {
        $ids = (array)$ids;

        foreach ($ids as $id) {
            if ($id > 0) {
                $sql = "DELETE FROM isys_packager_repository WHERE isys_packager_repository__id = {$id};";
            }
        }
    }

    /**
     * Sanitize the given version: digits and (max 2) periods only, not longer that 16 characters.
     *
     * @param string $version
     *
     * @return string
     */
    public static function sanitizeVersion(string $version): string
    {
        // If no identifier was provided, we use a hash.
        if (empty($version)) {
            $version = '0.1';
        }

        // Clean the identifier from unwanted input.
        $version = (string) preg_replace(['~[^0-9\.]~', '~^[\.]*~', '~[\.]*$~'], ['', '', ''], $version);

        // Only allow two periods: "x.y.z".
        $version = implode('.', \array_slice(explode('.', $version), 0, 3));

        // Only allow the identifier to be 32 characters long.
        return substr($version, 0, 32);
    }
}
