<?php

namespace idoit\Module\Packager\Component;

use DateTime;
use Exception;
use idoit\Exception\JsonException;
use idoit\Module\Packager\Component\Creator\CustomFile;
use idoit\Module\Packager\Component\Creator\DataFile;
use idoit\Module\Packager\Component\Creator\Directory;
use idoit\Module\Packager\Component\Creator\PackageJsonFile;
use idoit\Module\Packager\Component\Creator\PhpFile;
use idoit\Module\Packager\Component\Creator\StaticFile;
use idoit\Module\Packager\Component\Creator\UpdateXmlFile;
use idoit\Module\Packager\Configuration\Addon as AddonConfiguration;
use idoit\Module\Packager\Model\FileRepository as FileRepositoryModel;
use idoit\Module\Packager\Model\Repository as RepositoryModel;
use isys_application;
use isys_component_database;
use isys_exception_database;
use isys_format_json;
use isys_module_packager;
use ZipArchive;

/**
 * Class Packager
 *
 * @package   idoit\Module\Packager\Component
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class Packager
{
    /**
     * Will contain the path to the add-on build path.
     *
     * @var string
     */
    private string $buildPath;

    /** @var isys_component_database */
    private isys_component_database $database;

    /**
     * Will contain the relative path to the add-on directory.
     *
     * @var string
     */
    private $relativeAddonPath;

    /** @var array */
    private $addonMetaData;

    /** @var AddonConfiguration */
    private $addonConfiguration;

    /** @var int */
    private int $repositoryId;

    /**
     * Packager constructor.
     *
     * @param int              $repositoryEntryId
     * @param isys_application $application
     *
     * @throws isys_exception_database
     * @throws JsonException
     * @throws Exception
     */
    public function __construct(int $repositoryEntryId, isys_application $application)
    {
        $this->repositoryId = $repositoryEntryId;
        $this->database = $application->container->get('database');

        $repositoryModel = RepositoryModel::instance($this->database);

        $addonMetaData = $repositoryModel
            ->getById($this->repositoryId)
            ->get_row();

        $changelog = [];
        $changelogResult = $repositoryModel->getChangelogOfAddon((int)$addonMetaData['addonId'], $this->repositoryId);

        while ($row = $changelogResult->get_row()) {
            $changelog[] = $row['changelog'];
        }

        $this->addonConfiguration = new AddonConfiguration(
            $addonMetaData['identifier'],
            $addonMetaData['title'],
            $addonMetaData['description'],
            $addonMetaData['version'],
            (string)$addonMetaData['manufacturer'],
            (string)$addonMetaData['website'],
            (bool)$addonMetaData['requiresLicense'],
            $addonMetaData['idoitRequirement'] ?: $application->info['version'],
            new DateTime($addonMetaData['created']),
            new DateTime($addonMetaData['updated']),
            new DateTime(),
            implode(PHP_EOL . PHP_EOL, $changelog),
            (array)isys_format_json::decode($addonMetaData['assignedObjectTypeGroups']),
            (bool) $addonMetaData['uninstallObjectTypeGroups'],
            (array)isys_format_json::decode($addonMetaData['assignedObjectTypes']),
            (bool) $addonMetaData['uninstallObjectTypes'],
            (array)isys_format_json::decode($addonMetaData['assignedCustomCategories']),
            (bool) $addonMetaData['uninstallCustomCategories'],
            (array)isys_format_json::decode($addonMetaData['selectedReports']),
            (string)$addonMetaData['selectedDialogPlus'],
            (array)isys_format_json::decode($addonMetaData['selectedRelationTypes']),
            (array)isys_format_json::decode($addonMetaData['selectedCsvImportProfiles'])
        );

        $this->buildPath = isys_module_packager::getPath() . "build/{$this->addonConfiguration->identifier}/v{$this->addonConfiguration->version}/";
        $this->relativeAddonPath = "src/classes/modules/{$this->addonConfiguration->identifier}/";
    }

    /**
     * This method will actually create the add-on package.
     *
     * @return array
     * @throws isys_exception_database
     */
    public function createPackage(): array
    {
        $startTime = microtime(true);

        // First we clear any existing package inside the build directory.
        isys_glob_delete_recursive($this->buildPath, $deleted, $undeleted);

        // Now we re-build the structure and files.
        $this->createDirectories();
        $this->createCustomFiles();
        $this->createStaticFiles();
        $this->createDataFiles();
        $this->createPhpFiles();

        // Create the update XML file with all necessary queries.
        $this->createUpdateXmlFile();

        // We call this as last step, since it will scan all add-on files.
        $this->createPackageJsonFile();

        // Final step: compress the contents of the build into a ZIP.
        return [
            'zipFilename'  => $this->createZip(),
            'duration'     => microtime(true) - $startTime,
            'repositoryId' => $this->repositoryId
        ];
    }

    /**
     * Creates the directories before creating any files.
     */
    private function createDirectories()
    {
        $directories = [
            '',
            'data',
            'install',
            'src/Processor',
        ];

        (new Directory($this->buildPath . $this->relativeAddonPath))->createDirectories($directories);
    }

    /**
     * Creates custom directories and files, by user input.
     * We call this method quite early so that the user can not overwrite add-on specific files.
     */
    private function createCustomFiles()
    {
        // We do not work with the add-on configuration since we don't want to load all files into it.
        (new CustomFile($this->buildPath . $this->relativeAddonPath, new FileRepositoryModel($this->database)))->createDirectoriesAndFiles($this->repositoryId);
    }

    /**
     * Creates the package json file.
     */
    private function createPackageJsonFile()
    {
        (new PackageJsonFile($this->buildPath, $this->buildPath . $this->relativeAddonPath))->createPackageJsonFile($this->addonConfiguration);
    }

    private function createDataFiles()
    {
        $objectTypeAssignments = [];

        $uninstallConfiguration = [
            'object-type-groups' => $this->addonConfiguration->uninstallObjectTypeGroups,
            'object-types' => $this->addonConfiguration->uninstallObjectTypes,
            'custom-categories' => $this->addonConfiguration->uninstallCustomCategories,
        ];

        foreach ($this->addonConfiguration->assignedObjectTypes as $objectType) {
            $objectTypeAssignments[] = [
                'constant' => $objectType['constant'],
                'global' => $objectType['globalCategories'],
                'specific' => $objectType['specificCategory'],
                'custom' => $objectType['customCategories']
            ];
        }

        (new DataFile($this->buildPath . $this->relativeAddonPath))->createDataFiles(
            $this->addonConfiguration->dialogPlusData,
            $this->addonConfiguration->assignedObjectTypeGroups,
            $objectTypeAssignments,
            $this->addonConfiguration->assignedCustomCategories,
            $this->addonConfiguration->selectedReports,
            $uninstallConfiguration
        );
    }

    /**
     * Creates the module class and "init.php" file.
     */
    private function createPhpFiles()
    {
        (new PhpFile($this->buildPath . $this->relativeAddonPath, isys_module_packager::getPath() . 'templates/code/'))->createPhpFiles($this->addonConfiguration);
    }

    /**
     * Creates the static "VERSION" and "CHANGELOG" files.
     */
    private function createStaticFiles()
    {
        (new StaticFile($this->buildPath . $this->relativeAddonPath))->createStaticFiles($this->addonConfiguration->changelog, $this->addonConfiguration->version);
    }

    /**
     * Creates the "update_data.xml" that includes all necessary queries to create
     * object type groups, object types, custom categories and their assignments.
     *
     * @throws isys_exception_database
     */
    private function createUpdateXmlFile()
    {
        (new UpdateXmlFile($this->buildPath . $this->relativeAddonPath))->createUpdateXmlFile($this->addonConfiguration, $this->database);
    }

    /**
     * @return string
     */
    public function getZipFilePath()
    {
        return \dirname($this->buildPath) . '/' . 'idoit-' . $this->addonConfiguration->identifier . '-' . $this->addonConfiguration->version . '.zip';
    }

    /**
     * Method for creating the ZIP, containing all files inside the build directory. Will return the ZIPs filename.
     *
     * @return string
     */
    private function createZip(): string
    {
        // Now we ZIP the directory.
        $zipArchive = new ZipArchive();

        $zipFilePath = $this->getZipFilePath();

        if ($zipArchive->open($zipFilePath, ZipArchive::CREATE) === true) {
            $addonPathLength = mb_strlen($this->buildPath);
            $fileIterator = new \RecursiveIteratorIterator(new \RecursiveDirectoryIterator($this->buildPath));

            foreach ($fileIterator as $file) {
                /** @var \SplFileInfo $file */
                if (strncmp($file->getBasename(), '.', 1) === 0) {
                    // Skip "." and "..".
                    continue;
                }

                $zipArchive->addFile($file->getPathname(), str_replace('\\', '/', substr($file->getPathname(), $addonPathLength)));
            }

            $zipArchive->close();
        }

        return basename($zipFilePath);
    }
}
