<?php

namespace idoit\Module\Packager\Model;

use Exception;
use idoit\Model\Dao\Base;
use isys_component_dao_result;
use isys_exception_database;
use RuntimeException;

/**
 * Class Reports
 *
 * @package   idoit\Module\Packager\Model
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class Reports extends Base
{
    /**
     * @return isys_component_dao_result
     * @throws isys_exception_database
     */
    public function retrieveForGui(): isys_component_dao_result
    {
        $select = $this->selectImplode([
            'isys_report__id'    => 'id',
            'isys_report__title' => 'title',
            'isys_report__const' => 'constant',
        ]);

        return $this->retrieve("SELECT {$select} FROM isys_report;");
    }

    /**
     * @param array $constants
     *
     * @return isys_component_dao_result
     * @throws isys_exception_database
     */
    public function retrieveByConstants(array $constants): isys_component_dao_result
    {
        if (empty($constants)) {
            throw new RuntimeException('At least one report constant has to be passed!');
        }

        try {
            $this->ensureCategoryConstants($constants);
        } catch (Exception $e) {
            // Do nothing.
        }

        $select = $this->selectImplode([
            'isys_report__title'                         => 'title',
            'isys_report__description'                   => 'description',
            'isys_report__query'                         => 'query',
            'isys_report__query_row'                     => 'queryRow',
            'isys_report__datetime'                      => 'datetime',
            'isys_report__type'                          => 'type',
            'isys_report__user_specific'                 => 'userSpecific',
            'isys_report__querybuilder_data'             => 'querybuilderData',
            'isys_report__empty_values'                  => 'emptyValues',
            'isys_report__display_relations'             => 'displayRelations',
            'isys_report__category_report'               => 'categoryReport',
            'isys_report__const'                         => 'constant',
            'isys_report__compressed_multivalue_results' => 'compressedMultivalueResults',
            'isys_report__show_html'                     => 'showHtml',
            'isys_report_category__id'                   => 'categoryId',
            'isys_report_category__title'                => 'categoryTitle',
            'isys_report_category__const'                => 'categoryConstant',
        ]);

        $constantList = implode(', ', array_map(function ($constant) { return $this->convert_sql_text($constant); }, $constants));

        $sql = "SELECT {$select}
            FROM isys_report
            LEFT JOIN isys_report_category ON isys_report_category__id = isys_report__isys_report_category__id
            WHERE isys_report__const IN ({$constantList});";

        return $this->retrieve($sql);
    }

    /**
     * This method will go sure to create constants for used report categories.
     *
     * @param array $constants
     *
     * @throws \isys_exception_dao
     * @throws isys_exception_database
     */
    private function ensureCategoryConstants(array $constants)
    {
        foreach ($constants as $constant) {
            $constant = $this->convert_sql_text($constant);

            $sql = "SELECT isys_report_category__id AS id, isys_report_category__title AS title
                FROM isys_report
                LEFT JOIN isys_report_category ON isys_report_category__id = isys_report__isys_report_category__id
                WHERE isys_report__const IN ({$constant})
                AND isys_report_category__id != 1
                AND (isys_report_category__const IS NULL OR isys_report_category__const = '');";

            $row = $this->retrieve($sql)->get_row();

            if (!empty($row) && $row['id'] > 0 && trim($row['title']) !== '') {
                // We found a report category without constant - create it!
                $counter = 1;
                $newConstant = 'C__REPORT_CATEGORY__' . strtoupper(Addon::sanitizeIdentifier($row['title']));

                if ($this->categoryExists($newConstant)) {
                    while ($this->categoryExists($newConstant . $counter)) {
                        $counter++;
                    }

                    $newConstant = $newConstant . $counter;
                }

                $categoryId = $this->convert_sql_id($row['id']);
                $newConstant = $this->convert_sql_text($newConstant);

                $sql = "UPDATE isys_report_category
                    SET isys_report_category__const = {$newConstant}
                    WHERE isys_report_category__id = {$categoryId}
                    LIMIT 1;";

                $this->update($sql) && $this->apply_update();
            }
        }
    }

    /**
     * Method for checking if a category exists.
     *
     * @param string $constant
     *
     * @return bool
     */
    private function categoryExists(string $constant): bool
    {
        try {
            $constant = $this->convert_sql_text($constant);

            $sql = "SELECT isys_report_category__id
                FROM isys_report_category
                WHERE isys_report_category__const = {$constant} 
                LIMIT 1;";

            return count($this->retrieve($sql)) > 0;
        } catch (Exception $e) {
            return false;
        }
    }
}
