<?php

namespace idoit\Module\Packager\View;

use Exception;
use idoit\Exception\JsonException;
use idoit\Model\Dao\Base as DaoBase;
use idoit\Module\Packager\Model\Addon as AddonModel;
use idoit\Module\Packager\Model\Repository;
use idoit\View\Base;
use idoit\View\Renderable;
use isys_application;
use isys_auth;
use isys_component_template as ComponentTemplate;
use isys_component_template_navbar as ComponentNavbar;
use isys_exception_database;
use isys_format_json;
use isys_module as ModuleBase;
use isys_module_packager;

/**
 * Class Addon
 *
 * @package   idoit\Module\Packager\View
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class Addon extends Base
{
    /**
     * @param ModuleBase        $module
     * @param ComponentTemplate $template
     * @param DaoBase           $model
     *
     * @return $this|Renderable
     * @throws Exception
     * @throws JsonException
     * @throws isys_exception_database
     */
    public function process(ModuleBase $module, ComponentTemplate $template, DaoBase $model)
    {
        $language = isys_application::instance()->container->get('language');
        $database = isys_application::instance()->container->get('database');
        $locales = isys_application::instance()->container->get('locales');

        $editMode = true; // isys_glob_is_edit_mode();
        $referenceOptionData = [];

        // Check if the user is allowed to see this page.
        isys_module_packager::getAuth()->check(isys_auth::EXECUTE, 'packager');

        ComponentNavbar::getInstance()
            ->set_active(!$editMode, C__NAVBAR_BUTTON__EDIT)
            ->set_visible(!$editMode, C__NAVBAR_BUTTON__EDIT)
            ->set_active($editMode, C__NAVBAR_BUTTON__SAVE)
            ->set_visible($editMode, C__NAVBAR_BUTTON__SAVE);

        $lastVersion = null;
        $previousChangelogData = [];

        if ($this->id) {
            $addonData = AddonModel::instance($database)
                ->getById((int)$this->id)
                ->get_row();

            if (isys_format_json::is_json_array($addonData['assignedObjectTypeGroups'])) {
                $addonData['assignedObjectTypeGroups'] = isys_format_json::decode($addonData['assignedObjectTypeGroups']);
            }

            if (isys_format_json::is_json_array($addonData['assignedObjectTypes'])) {
                $addonData['assignedObjectTypes'] = isys_format_json::decode($addonData['assignedObjectTypes']);
            }

            if (isys_format_json::is_json_array($addonData['assignedCustomCategories'])) {
                $addonData['assignedCustomCategories'] = isys_format_json::decode($addonData['assignedCustomCategories']);
            }

            if (isys_format_json::is_json_array($addonData['selectedReports'])) {
                $addonData['selectedReports'] = isys_format_json::decode($addonData['selectedReports']);
            }

            if (isys_format_json::is_json_array($addonData['selectedDialogPlus'])) {
                $addonData['selectedDialogPlus'] = isys_format_json::decode($addonData['selectedDialogPlus']);
            }

            if (isys_format_json::is_json_array($addonData['selectedRelationTypes'])) {
                $addonData['selectedRelationTypes'] = isys_format_json::decode($addonData['selectedRelationTypes']);
            }

            if (isys_format_json::is_json_array($addonData['selectedCsvImportProfiles'])) {
                $addonData['selectedCsvImportProfiles'] = isys_format_json::decode($addonData['selectedCsvImportProfiles']);
            }

            $newVersion = '0.1';
            $lastVersion = Repository::instance($database)->getLatestVersionOfAddon((int)$this->id)->get_row_value('version');

            if ($lastVersion) {
                // If we have a "last version" we use that as the new version, but increase the version number by one step.
                $versionParts = explode('.', $lastVersion);
                $versionParts[\count($versionParts) - 1] ++;
                $newVersion = implode('.', $versionParts);
            }

            $changelogResult = Repository::instance($database)->getChangelogOfAddon((int)$this->id);

            while ($changelogRow = $changelogResult->get_row()) {
                $previousChangelogData[] = $changelogRow['changelog'];
            }
        } else {
            // Defaults
            $addonData = [
                'title'                     => '',
                'identifier'                => '',
                'requiresLicense'           => 0,
                'idoitRequirement'          => '1.11', // trim(isys_application::instance()->info['version']),
                'createdOnPackagerVersion'  => trim(file_get_contents($module::getPath() . 'VERSION')),
                'assignedObjectTypeGroups'  => [],
                'assignedObjectTypes'       => [],
                'assignedCustomCategories'  => [],
                'selectedReports'           => [],
                'selectedDialogPlus'        => [],
                'selectedRelationTypes'     => [],
                'selectedCsvImportProfiles' => [],
            ];

            $newVersion = '0.1';
        }

        $rules = [
            'identifier'               => [
                'p_strValue'  => $addonData['identifier'],
                'p_nMaxLen'   => 32,
                'p_bDisabled' => !empty($addonData['identifier'])
            ],
            'title'                    => [
                'p_strValue' => $addonData['title']
            ],
            'manufacturer'             => [
                'p_strValue' => $addonData['manufacturer']
            ],
            'website'                  => [
                'p_strValue' => $addonData['website'],
                'p_strPlaceholder' => 'https:// ...'
            ],
            'description'              => [
                'p_strValue' => $addonData['description']
            ],
            'idoitRequirement'         => [
                'p_strValue'       => $addonData['idoitRequirement'] ?: '1.11',
                'p_strPlaceholder' => '1.11'
            ],
            'createdOnPackagerVersion' => [
                'p_strValue'  => trim(file_get_contents($module::getPath() . 'VERSION')),
                'p_bReadonly' => true
            ],
            'requiresLicense'          => [
                'p_arData'        => get_smarty_arr_YES_NO(),
                'p_strSelectedID' => $addonData['requiresLicense'],
                'p_bDbFieldNN'    => true
            ],
            'includeDefinition'        => [
                'p_arData'        => get_smarty_arr_YES_NO(),
                'p_strSelectedID' => $addonData['includeDefinition'],
                'p_bDbFieldNN'    => true
            ],
            'objectTypeGroup_uninstall' => [
                'p_arData'        => get_smarty_arr_YES_NO(),
                'p_strSelectedID' => $addonData['uninstallObjectTypeGroups'] ?? 0,
                'p_bDbFieldNN'    => true
            ],
            'objectType_uninstall' => [
                'p_arData'        => get_smarty_arr_YES_NO(),
                'p_strSelectedID' => $addonData['uninstallObjectTypes'] ?? 0,
                'p_bDbFieldNN'    => true
            ],
            'report_uninstall'     => [
                'p_arData'        => ['1' => $language->get('LC__UNIVERSAL__YES')],
                'p_strSelectedID' => 1,
                'p_bDisabled'     => true
            ],
            'customCategory_uninstall' => [
                'p_arData'        => get_smarty_arr_YES_NO(),
                'p_strSelectedID' => $addonData['uninstallCustomCategories'] ?? 1,
                'p_bDbFieldNN'    => true
            ],
            'dialog_plus_uninstall' => [
                'p_arData'        => ['0' => $language->get('LC__UNIVERSAL__NO')],
                'p_strSelectedID' => 0,
                'p_bDisabled'     => true,
            ],
            'relation_type_uninstall' => [
                'p_arData'        => ['0' => $language->get('LC__UNIVERSAL__NO')],
                'p_strSelectedID' => 0,
                'p_bDisabled'     => true,
            ],
            'csv_import_profile_uninstall' => [
                'p_arData'        => ['0' => $language->get('LC__UNIVERSAL__NO')],
                'p_strSelectedID' => 0,
                'p_bDisabled'     => true,
            ],
            'file_uninstall' => [
                'p_arData'        => ['1' => $language->get('LC__UNIVERSAL__YES')],
                'p_strSelectedID' => 1,
                'p_bDisabled'     => true,
            ],
            'newVersion'               => [
                'p_strValue' => $newVersion,
                'p_strClass' => 'input-mini'
            ],
            'releaseDate'              => [
                'p_strValue' => date('Y-m-d H:i:s'),
                'p_strClass' => 'input-small',
                'p_strStyle' => 'width:70%;',
                'p_bTime'    => true
            ]
        ];

        // Assign template variables.
        $template
            ->activate_editmode()
            ->assign('encType', 'multipart/form-data')
            ->assign('baseUrl', isys_application::instance()->www_path)
            ->assign('addonId', $this->id)
            ->assign('isEditMode', $editMode)
            ->assign('identifier', $addonData['identifier'])
            ->assign('packagerPath', $module::getPath())
            ->assign('packagerWwwPath', $module::getWwwPath())
            ->assign('objectTypeGroupSelection', $addonData['assignedObjectTypeGroups'] ?: [])
            ->assign('objectTypeSelection', $addonData['assignedObjectTypes'] ?: [])
            ->assign('customCategorySelection', $addonData['assignedCustomCategories'] ?: [])
            ->assign('reportSelection', $addonData['selectedReports'] ?: [])
            ->assign('dialogPlusSelection', $addonData['selectedDialogPlus'] ?: [])
            ->assign('relationTypeSelection', $addonData['selectedRelationTypes'] ?: [])
            ->assign('csvImportProfileSelection', $addonData['selectedCsvImportProfiles'] ?: [])
            ->assign('lastVersion', $lastVersion)
            ->assign('previousChangelogData', implode(PHP_EOL . PHP_EOL, $previousChangelogData))
            ->smarty_tom_add_rules('tom.content.bottom.content', $rules)
            ->smarty_tom_add_rule('tom.content.bottom.buttons.*.p_bInvisible=1')
            ->smarty_tom_add_rule('tom.content.navbar.cRecStatus.p_bInvisible=1');

        // Set paths to templates.
        $this->paths['contentbottomcontent'] = $module::getPath() . 'templates/edit.tpl';

        return $this;
    }
}
