<?php

namespace idoit\Module\UpperCaseIdentifier\Processor;

use idoit\Component\Autoloader;
use idoit\Module\UpperCaseIdentifier\Processor;
use isys_cmdb_dao;
use isys_custom_fields_dao;
use isys_report;

/**
 * Class Uninstall
 *
 * @package   idoit\Module\UpperCaseIdentifier\Processor
 * @copyright <manufacturer>
 * @license   <website>
 */
class Uninstall extends Processor
{
    /**
     * @return void
     * @throws \idoit\Exception\JsonException
     * @throws \isys_exception_dao
     * @throws \Exception
     */
    public function process()
    {
        // @see PACKAGER-39 Read configuration
        $uninstallConfiguration = $this->getDataFileContent('uninstall-configuration.json');

        $objectTypeGroups = $this->getDataFileContent('object-type-groups.json');
        $objectTypes = $this->getDataFileContent('object-type-assignments.json');
        $customCategories = $this->getDataFileContent('custom-categories.json');
        $reports = $this->getDataFileContent('reports.json');

        if (isset($uninstallConfiguration['object-type-groups']) && $uninstallConfiguration['object-type-groups'] && \count($objectTypeGroups)) {
            $this->removeObjectTypeGroups($objectTypeGroups);
        }

        if (isset($uninstallConfiguration['object-types']) && $uninstallConfiguration['object-types'] && \count($objectTypes)) {
            $this->removeObjectTypes($objectTypes);
        }

        if (isset($uninstallConfiguration['custom-categories']) && $uninstallConfiguration['custom-categories'] && \count($customCategories)) {
            $this->removeCustomCategories($customCategories);
        }

        if (\count($reports)) {
            $this->removeReports($reports);
        }
    }

    /**
     * @param array $objectTypeGroups
     *
     * @return void
     * @throws \isys_exception_dao
     */
    private function removeObjectTypeGroups(array $objectTypeGroups): void
    {
        $dao = isys_cmdb_dao::instance($this->database);

        // First we filter all "non-self-defined" object type groups.
        $objectTypeGroups = array_filter($objectTypeGroups, fn ($objectTypeGroup) => strpos($objectTypeGroup['constant'], 'C__OBJTYPE_GROUP__SD_') === 0);
        // Only fetch the constants.
        $objectTypeGroups = array_map(fn ($objectTypeGroup) => $objectTypeGroup['constant'], $objectTypeGroups);
        // Unify and remove 'empty' values.
        $objectTypeGroups = array_filter(array_unique($objectTypeGroups));

        if (\count($objectTypeGroups) === 0) {
            return;
        }

        // Finally prepare the 'IN (...)' structure.
        $objectTypeGroupConstants = implode(',', array_map(fn ($objectTypeGroup) => $dao->convert_sql_text($objectTypeGroup), $objectTypeGroups));

        $dao->update("DELETE FROM isys_obj_type_group WHERE isys_obj_type_group__const IN ({$objectTypeGroupConstants});");
        $dao->apply_update();
    }

    /**
     * @param array $objectTypes
     *
     * @return void
     * @throws \isys_exception_dao
     * @throws \isys_exception_database
     */
    private function removeObjectTypes(array $objectTypes): void
    {
        $dao = isys_cmdb_dao::instance($this->database);

        $replacementObjectTypeQuery = "SELECT isys_obj_type__id AS id
            FROM isys_obj_type
            WHERE isys_obj_type__const IN ('C__OBJTYPE__MIGRATION_OBJECT', 'C__OBJTYPE__REPLICATION')
            LIMIT 1;";
        $replacementObjectType = (int)$dao->retrieve($replacementObjectTypeQuery)->get_row_value('id');

        if ($replacementObjectType <= 0) {
            return;
        }

        $objectTypeConstants = array_map(fn($objectType) => $objectType['constant'], $objectTypes);
        // Unify and remove 'empty' values.
        $objectTypeConstants = array_filter(array_unique($objectTypeConstants));

        if (\count($objectTypeConstants) === 0) {
            return;
        }

        $objectTypeConstants = implode(',', array_map(fn ($objectTypeGroup) => $dao->convert_sql_text($objectTypeGroup), $objectTypeConstants));

        // Set the "replacement" object type for all objects, that will be affected by the removal.
        $dao->update("UPDATE isys_obj
            SET isys_obj__isys_obj_type__id = {$replacementObjectType}
            WHERE isys_obj__isys_obj_type__id IN (SELECT isys_obj_type__id FROM isys_obj_type WHERE isys_obj_type__const IN ({$objectTypeConstants}));");

        $dao->update("DELETE FROM isys_obj_type WHERE isys_obj_type__const IN ({$objectTypeConstants});");
        $dao->apply_update();
    }

    /**
     * @param array $customCategories
     *
     * @return void
     * @throws \Exception
     */
    private function removeCustomCategories(array $customCategories): void
    {
        // @see PACKAGER-41 First go sure the autoloader exists.
        if (! class_exists(Autoloader::class)) {
            require BASE_DIR . 'src/autoload.inc.php';
        }

        // @see PACKAGER-41 Then we go sure to initialize the 'custom_fields' add-on.
        if (! class_exists(isys_custom_fields_dao::class)) {
            require BASE_DIR . 'src/classes/modules/custom_fields/init.php';
        }

        $customCategoryDao = isys_custom_fields_dao::instance($this->database);

        foreach ($customCategories as $customCategory) {
            if (\defined($customCategory['constant'])) {
                $customCategoryId = \constant($customCategory['constant']);
            } else {
                $sql = 'SELECT isysgui_catg_custom__id AS id
                    FROM isysgui_catg_custom
                    WHERE isysgui_catg_custom__const = ' . $customCategoryDao->convert_sql_text($customCategory['constant']) . '
                    LIMIT 1;';

                $customCategoryId = (int)$customCategoryDao->retrieve($sql)->get_row_value('id');
            }

            if ($customCategoryId) {
                $customCategoryDao->delete($customCategoryId);
            }
        }
    }

    /**
     * @param array $reports
     *
     * @return void
     * @throws \Exception
     */
    private function removeReports(array $reports): void
    {
        // @see PACKAGER-48 PACKAGER-54 Register report related autoloader for add-ons that bring reports.
        if (class_exists(\isys_module_report::class)) {
            \idoit\Psr4AutoloaderClass::factory()->addNamespace('idoit\Module\Report', \isys_module_report::getPath() . 'src/');
            \idoit\Component\Autoloader::appendClassmap(include(\isys_module_report::getPath() . 'classmap.php'));
        } else {
            return;
        }

        try {
            $reportDao = new isys_report([]);
            $cmdbDao = new isys_cmdb_dao($this->database);

            foreach ($reports as $report) {
                if (\defined($report['constant'])) {
                    $reportId = \constant($report['constant']);
                } else {
                    $sql = 'SELECT isys_report__id AS id
                    FROM isys_report
                    WHERE isys_report__const = ' . $cmdbDao->convert_sql_text($report['constant']) . '
                    LIMIT 1;';

                    $reportId = (int)$cmdbDao->retrieve($sql)->get_row_value('id');
                }

                $reportDao->setId($reportId)->delete();
            }
        } catch (\Throwable $e) {
        }
    }
}
