<?php

namespace idoit\Module\RelocateCi\Relocator;

use idoit\Component\Logger;
use idoit\Module\RelocateCi\Model\Ci;
use isys_cmdb_dao_category_g_location;
use isys_cmdb_dao_category_g_logical_unit;
use isys_component_dao_logbook;
use isys_component_database;
use isys_component_template_language_manager;
use isys_exception_database;
use isys_relocate_ci_dao;
use isys_tenantsettings;

abstract class AbstractRelocation
{
    public const PHYSICALLY = 'physically';
    public const LOGICALLY = 'logically';

    /**
     * @var Ci
     */
    protected $ciModel;

    /**
     * @var isys_component_dao_logbook
     */
    private $logbookDao;

    /**
     * @var isys_component_template_language_manager
     */
    protected $language;

    /**
     * @var Logger
     */
    protected $log;

    /**
     * @var isys_cmdb_dao_category_g_logical_unit
     */
    protected $logicalLocationDao;

    /**
     * @var isys_cmdb_dao_category_g_location
     */
    protected $physicalLocationDao;

    /**
     * @var isys_relocate_ci_dao
     */
    protected $relocateDao;

    /**
     * @var array
     */
    private $objectPathCache = [];

    /**
     * @var isys_component_database
     */
    protected $database;

    /**
     * @var array
     */
    private $childrenCache = [];

    /**
     * @param isys_component_database                  $database
     * @param isys_component_template_language_manager $language
     * @param Logger                                   $log
     */
    public function __construct(isys_component_database $database, isys_component_template_language_manager $language, Logger $log)
    {
        $this->database = $database;
        $this->language = $language;
        $this->log = $log;

        $this->ciModel = Ci::instance($this->database);
        $this->logbookDao  = isys_component_dao_logbook::instance($this->database);
        $this->logicalLocationDao = isys_cmdb_dao_category_g_logical_unit::instance($this->database);
        $this->physicalLocationDao = isys_cmdb_dao_category_g_location::instance($this->database);
        $this->relocateDao = isys_relocate_ci_dao::instance($this->database);
    }

    /**
     * @param int    $objectId
     * @param string $type
     * @param array  $changes
     * @param string $message
     * @param bool   $overwriteEventName
     *
     * @throws \isys_exception_dao
     * @throws \isys_exception_database
     */
    protected function writeLogbook(int $objectId, string $type, array $changes, string $message, bool $overwriteEventName = false): void
    {
        $objectData = $this->ciModel->getObjectData($objectId)->get_row();

        $categoryName = 'LC__CMDB__CATG__LOCATION';
        $translation = 'LC__MODULE__RELOCATE_CI__SIMPLE_LOGBOOK__PHYSICAL_LOCATION_CHANGED_IN_ADDON';

        if ($type === self::LOGICALLY) {
            $categoryName = 'LC__CMDB__CATG__LOGICAL_UNIT';
            $translation = 'LC__MODULE__RELOCATE_CI__SIMPLE_LOGBOOK__LOGICAL_LOCATION_CHANGED_IN_ADDON';
        }

        if ($objectData['typeConst'] === 'C__OBJTYPE__PERSON') {
            $translation = 'LC__MODULE__RELOCATE_CI__SIMPLE_LOGBOOK__PERSON_LOCATION_CHANGED';
        } elseif ($objectData['typeConst'] === 'C__OBJTYPE__WORKSTATION') {
            $translation = 'LC__MODULE__RELOCATE_CI__SIMPLE_LOGBOOK__WORKSTATION_LOCATION_CHANGED';
        }

        if ($overwriteEventName) {
            $translation = $message;
        }

        $currentChange = array_values($changes)[0];

        $this->logbookDao->set_entry(
            $this->language->get($translation, [$currentChange['from'] ?? '', $currentChange['to'] ?? '']),
            $this->logbookDao->get_last_query(),
            null,
            C__LOGBOOK__ALERT_LEVEL__0,
            $objectData['id'],
            $objectData['title'],
            $objectData['typeTitle'],
            $categoryName,
            C__LOGBOOK_SOURCE__RELOCATE_CI,
            serialize($changes),
            $message
        );
    }

    /**
     * @param string $rootObject
     * @param string $destinationObject
     * @param int    $objectId
     * @param string $type
     *
     * @throws \isys_exception_dao
     * @throws \isys_exception_database
     */
    public function writeRecursiveLogbook(string $rootObject, string $destinationObject, int $objectId, string $type): void
    {
        if (isset($this->childrenCache[$objectId])) {
            return;
        }

        $this->childrenCache[$objectId] = true;

        $this->log->debug($this->language->get('LC__MODULE__RELOCATE_CI__LOG__WRITING_RECURSIVE_LOG', $objectId));

        $this->writeLogbook(
            $objectId,
            $type,
            [],
            $this->language->get('LC__MODULE__RELOCATE_CI__SIMPLE_LOGBOOK__PARENT_CI_LOCATION_CHANGED', $rootObject, $destinationObject),
            true
        );

        $children = $this->ciModel->getLocatedObjects($objectId);

        foreach ($children as $child) {
            if (isset($this->childrenCache[$child['id']])) {
                continue;
            }

            $this->childrenCache[$child['id']] = true;

            // Iterate further.
            $this->writeRecursiveLogbook($rootObject, $destinationObject, $child['id'], $type);
        }
    }

    /**
     * @param int $objectId
     *
     * @return array
     * @throws \isys_exception_database
     */
    private function retrieveLocationPath(int $objectId): array
    {
        $return = [];

        $parentObjectId = $this->physicalLocationDao->get_parent_id_by_object($objectId);

        if ($parentObjectId > 0) {
            $l_locationpath = array_reverse($this->physicalLocationDao->get_location_path($objectId));

            // Parse location tree.
            foreach ($l_locationpath as $l_object_id) {
                if ($l_object_id > C__MPTT__ROOT_NODE && $l_object_id != $objectId) {
                    $return[] = $this->physicalLocationDao->get_obj_name_by_id_as_string($l_object_id);
                }
            }

            $return[] = $this->physicalLocationDao->get_obj_name_by_id_as_string($objectId);
        } else {
            $parentObjectId = $this->logicalLocationDao->get_parent_by_id($objectId);

            if ($parentObjectId > 0) {
                $return = $this->retrieveLocationPath($parentObjectId);

                $return[] = $this->physicalLocationDao->get_obj_name_by_id_as_string($objectId);
            }
        }

        return $return;
    }

    /**
     * @param int|null $objectId
     *
     * @return string|null
     * @throws \isys_exception_database
     */
    protected function formatPath(?int $objectId): ?string
    {
        if (!$objectId) {
            return null;
        }

        if (isset($this->objectPathCache[$objectId])) {
            return $this->objectPathCache[$objectId];
        }

        return $this->objectPathCache[$objectId] = implode(isys_tenantsettings::get('gui.separator.location', ' > '), $this->retrieveLocationPath($objectId));
    }

    /**
     * Method for relocating logical children to a new physical location parent.
     *
     * @param int   $currentObject
     * @param int   $newPhysicalParent
     * @param array $detachedChildren
     *
     * @return void
     * @throws \isys_exception_dao
     * @throws isys_exception_database
     * @see RELOCATE-35
     */
    protected function physicallyRelocateLogicalChildren(int $currentObject, int $newPhysicalParent, array $detachedChildren = []): void
    {
        static $physicalRelocator = null;

        if ($physicalRelocator === null) {
            $physicalRelocator = new PhysicalRelocator($this->database, $this->language, $this->log);
        }

        // So first of all: find the (logically assigned) children.
        $children = $this->ciModel->getLocatedObjects($currentObject);

        foreach ($children as $child) {
            if ($child['located'] !== 'logical' || in_array($child['id'], $detachedChildren)) {
                // We only need to relocate logical children.
                continue;
            }

            // Relocate the child to the new parent.
            $physicalRelocator->relocate($child['id'], $newPhysicalParent);

            // After we relocated the child, we continue with grand-children.
            $this->physicallyRelocateLogicalChildren($child['id'], $newPhysicalParent, $detachedChildren);
        }
    }
}
