<?php

namespace idoit\Module\RelocateCi\Relocator;

class PhysicalRelocator extends AbstractRelocation
{
    /**
     * @param int   $relocationObjectId
     * @param int   $destinationObjectId
     * @param array $detachedChildren
     *
     * @return void
     * @throws \isys_exception_dao
     * @throws \isys_exception_database
     */
    public function relocate(int $relocationObjectId, int $destinationObjectId, array $detachedChildren = []): void
    {
        $locationParentId = null;
        $locationData = $this->physicalLocationDao
            ->get_data(null, $relocationObjectId)
            ->get_row();

        if (is_array($locationData) && $locationData['isys_catg_location_list__id'] > 0) {
            $locationParentId = (int)$locationData['isys_catg_location_list__parentid'];

            $this->updateLocation($locationData, $relocationObjectId, $destinationObjectId);
        } else {
            $this->createLocation($relocationObjectId, $destinationObjectId);
        }

        // Skip unnecessary logs.
        if ($locationParentId === $destinationObjectId) {
            return;
        }

        $changes = [
            'isys_cmdb_dao_category_g_location::parent' => [
                'from' => $this->formatPath($locationParentId),
                'to'   => $this->formatPath($destinationObjectId)
            ]
        ];

        $message = $this->language->get('LC__MODULE__RELOCATE_CI__LOGBOOK__PHYSICAL_LOCATION_CHANGED', [
            $this->ciModel->objectTitle($relocationObjectId),
            $this->ciModel->objectTitle($destinationObjectId)
        ]);

        // After relocating the object, insert a logbook entry.
        $this->writeLogbook($relocationObjectId, self::PHYSICALLY, $changes, $message);

        // And also a log entry.
        $this->log->info($message);

        // @see RELOCATE-35 Relocate any logically assigned children physically to the new location.
        // @see RELOCATE-41 Prevent detached children of being moved.
        $this->physicallyRelocateLogicalChildren($relocationObjectId, $destinationObjectId, $detachedChildren);
    }

    /**
     * @param int $objectId
     *
     * @throws \isys_exception_dao
     * @throws \isys_exception_database
     */
    public function resetLocation(int $objectId): void
    {
        $locationData = $this->physicalLocationDao
            ->get_data(null, $objectId)
            ->get_row();

        if (!is_array($locationData) || !$locationData['isys_catg_location_list__id']) {
            return;
        }

        $locationParentId = (int)$locationData['isys_catg_location_list__parentid'];

        $this->updateLocation($locationData, $objectId, null);

        $changes = [
            'isys_cmdb_dao_category_g_location::parent' => [
                'from' => $this->formatPath($locationParentId),
                'to'   => null
            ]
        ];

        $message = $this->language->get('LC__MODULE__RELOCATE_CI__LOGBOOK__PHYSICAL_LOCATION_CHANGED', [
            $this->ciModel->objectTitle($objectId),
            null
        ]);

        // After relocating the object, insert a logbook entry.
        $this->writeLogbook($objectId, self::PHYSICALLY, $changes, $message);

        // And also a log entry.
        $this->log->info($message);
    }

    /**
     * @param int $relocationObjectId
     * @param int $destinationObjectId
     *
     * @throws \Exception
     * @throws \isys_exception_dao
     */
    private function createLocation(int $relocationObjectId, int $destinationObjectId): void
    {
        $this->physicalLocationDao->insert_node($relocationObjectId, $destinationObjectId);

        $this->physicalLocationDao->create_category(
            null,
            $relocationObjectId,
            $destinationObjectId,
            null,
            null,
            null,
            null,
            C__RECORD_STATUS__NORMAL
        );
    }

    /**
     * @param array $dataSet
     * @param int   $relocationObjectId
     * @param int   $destinationObjectId
     *
     * @throws \Exception
     * @throws \isys_exception_dao
     */
    private function updateLocation(array $dataSet, int $relocationObjectId, int $destinationObjectId): void
    {
        // First we move the node in the LFT/RGT tree.
        $this->physicalLocationDao->move_node($relocationObjectId, $destinationObjectId);

        // Then we update the category data.
        $this->physicalLocationDao->save_category(
            $dataSet['isys_catg_location_list__id'],
            $destinationObjectId,
            null,
            null,
            null,
            $dataSet['isys_catg_location_list__description'],
            $dataSet['isys_catg_location_list__status']
        );
    }
}
