<?php

namespace idoit\Module\RelocateCi\Relocator;

use isys_exception_database;
use isys_exception_general;
use isys_tenantsettings;

class LogicalRelocator extends AbstractRelocation
{
    /**
     * @param int   $relocationObjectId
     * @param int   $destinationObjectId
     * @param array $detachedChildren
     *
     * @return void
     * @throws \isys_exception_dao
     * @throws isys_exception_database
     * @throws isys_exception_general
     */
    public function relocate(int $relocationObjectId, int $destinationObjectId, array $detachedChildren = []): void
    {
        $relocationObject = $this->ciModel
            ->getObjectData($relocationObjectId)
            ->get_row();
        $logicalLocationData = $this->logicalLocationDao
            ->get_data(null, $relocationObjectId)
            ->get_row();

        if (!is_array($logicalLocationData) || !$logicalLocationData['isys_catg_logical_unit_list__id']) {
            $logicalLocationData = $this->logicalLocationDao
                ->get_data($this->logicalLocationDao->create_connector('isys_catg_logical_unit_list', $relocationObjectId))
                ->get_row();
        }

        $logicalParentObjectId = $logicalLocationData['isys_catg_logical_unit_list__isys_obj__id__parent'];

        $this->logicalLocationDao->save(
            $logicalLocationData['isys_catg_logical_unit_list__id'],
            $destinationObjectId,
            $logicalLocationData['isys_catg_logical_unit_list__status'],
            $relocationObjectId,
            $logicalLocationData['isys_catg_logical_unit_list__isys_catg_relation_list__id'],
            $logicalLocationData['isys_catg_logical_unit_list__description']
        );

        // Skip unnecessary logs.
        if ($logicalParentObjectId !== $destinationObjectId) {
            $changes = [
                'isys_cmdb_dao_category_g_logical_unit::parent' => [
                    'from' => $logicalParentObjectId > 0
                        ? implode(isys_tenantsettings::get('gui.separator.location', ' > '), $this->format_location_path($logicalParentObjectId))
                        : null,
                    'to'   => implode(isys_tenantsettings::get('gui.separator.location', ' > '), $this->format_location_path($destinationObjectId))
                ]
            ];

            $message = $this->language->get('LC__MODULE__RELOCATE_CI__LOGBOOK__LOGICAL_LOCATION_CHANGED', [
                $this->ciModel->objectTitle($relocationObjectId),
                $this->ciModel->objectTitle($destinationObjectId)
            ]);

            // After relocating the object, insert a logbook entry.
            $this->writeLogbook($relocationObjectId, self::LOGICALLY, $changes, $message);

            // And also a log entry.
            $this->log->info($message);
        }

        $nextPhysical = $this->findNextPhysicalLocation($destinationObjectId);

        if ($nextPhysical !== null) {
            if (isys_tenantsettings::get('relocate-ci.update-physical-location-after-logical-relocation', 0)) {
                $this->log->debug($this->language->get('LC__MODULE__RELOCATE_CI__LOG__FIND_NEW_PHYSICAL_LOCATION', $this->ciModel->objectTitle($relocationObjectId)));

                // In this case we do not need to call 'physicallyRelocateLogicalChildren' because it already happens internally.
                // Previously the children where located twice - first 'correct' and then wrong (due to RELOCATE-37).
                (new PhysicalRelocator($this->database, $this->language, $this->log))->relocate($relocationObjectId, $nextPhysical);
            } else {
                // @see RELOCATE-35 We assign the new "next best" physical parent to the objects underneath our current relocation object.
                // @see RELOCATE-37 Previously the "$nextPhysical" var was set to "$relocationObjectId", this led to the issue.
                // @see RELOCATE-41 Prevent detached children of being moved.
                $this->physicallyRelocateLogicalChildren($relocationObjectId, $nextPhysical, $detachedChildren);
            }
        }
    }

    /**
     * @param int $objectId
     *
     * @throws \isys_exception_dao
     * @throws isys_exception_database
     * @throws isys_exception_general
     */
    public function resetLocation(int $objectId): void
    {
        $logicalLocationData = $this->logicalLocationDao->get_data(null, $objectId)
            ->get_row();

        if (!is_array($logicalLocationData) || !$logicalLocationData['isys_catg_logical_unit_list__id']) {
            return;
        }

        $logicalParentObjectId = $logicalLocationData['isys_catg_logical_unit_list__isys_obj__id__parent'];

        if (!$logicalParentObjectId) {
            return;
        }

        $this->logicalLocationDao->save(
            $logicalLocationData['isys_catg_logical_unit_list__id'],
            null,
            $logicalLocationData['isys_catg_logical_unit_list__status'],
            $objectId,
            $logicalLocationData['isys_catg_logical_unit_list__isys_catg_relation_list__id'],
            $logicalLocationData['isys_catg_logical_unit_list__description']
        );

        $changes = [
            'isys_cmdb_dao_category_g_logical_unit::parent' => [
                'from' => implode(isys_tenantsettings::get('gui.separator.location', ' > '), $this->format_location_path($logicalParentObjectId)),
                'to'   => null
            ]
        ];

        $message = $this->language->get('LC__MODULE__RELOCATE_CI__LOGBOOK__LOGICAL_LOCATION_CHANGED', [
            $this->ciModel->objectTitle($objectId),
            null
        ]);

        // After relocating the object, insert a logbook entry.
        $this->writeLogbook($objectId, self::LOGICALLY, $changes, $message);

        // And also a log entry.
        $this->log->info($message);
    }

    /**
     * @param int $objectId
     *
     * @return int|null
     * @throws isys_exception_database
     */
    public function findNextPhysicalLocation(int $objectId): ?int
    {
        // The first check is really simple - just look, if a physical location exists for the given object.
        $sql = 'SELECT isys_catg_location_list__parentid
            FROM isys_catg_location_list
            WHERE isys_catg_location_list__isys_obj__id = ' . $this->relocateDao->convert_sql_id($objectId) . '
            AND isys_catg_location_list__parentid > 0;';

        $result = $this->relocateDao->retrieve($sql);

        if (count($result)) {
            return (int)$result->get_row_value('isys_catg_location_list__parentid');
        }

        // If there is no physical parent we'll look for a logical location.
        $sql = 'SELECT isys_catg_logical_unit_list__isys_obj__id__parent
            FROM isys_catg_logical_unit_list
            WHERE isys_catg_logical_unit_list__isys_obj__id = ' . $this->relocateDao->convert_sql_id($objectId) . '
            AND isys_catg_logical_unit_list__isys_obj__id__parent > 0;';

        $result = $this->relocateDao->retrieve($sql);

        if (count($result)) {
            // And if we found a logical location, we'll iterate further until we got something.
            return $this->findNextPhysicalLocation($result->get_row_value('isys_catg_logical_unit_list__isys_obj__id__parent'));
        }

        return null;
    }

    /**
     * This method will format the object location path, even if the given objects location is logical.
     *
     * @param int $objectId
     *
     * @return array
     * @throws isys_exception_database
     * @throws isys_exception_general
     */
    private function format_location_path(int $objectId): array
    {
        $return = [];

        $parentObjectId = $this->physicalLocationDao->get_parent_id_by_object($objectId);

        if ($parentObjectId > 0) {
            $l_locationpath = array_reverse($this->physicalLocationDao->get_location_path($objectId));

            // Parse location tree.
            foreach ($l_locationpath as $l_object_id) {
                if ($l_object_id > C__MPTT__ROOT_NODE && $l_object_id != $objectId) {
                    $return[] = $this->physicalLocationDao->get_obj_name_by_id_as_string($l_object_id);
                }
            }

            $return[] = $this->physicalLocationDao->get_obj_name_by_id_as_string($objectId);
        } else {
            $parentObjectId = $this->logicalLocationDao->get_parent_by_id($objectId);

            if ($parentObjectId > 0) {
                $return = $this->format_location_path($parentObjectId);

                $return[] = $this->physicalLocationDao->get_obj_name_by_id_as_string($objectId);
            }
        }

        return $return;
    }
}
