window.RelocateCiTree = Class.create(window.BaseTree, {
    selectedNodes: [],

    /**
     * Constructor method.
     *
     * @param   $super
     * @param   $container
     * @param   options
     * @returns {Window.LocationTree}
     */
    initialize: function ($super, $container, options) {
        this.options = {
            onSelect:           Prototype.emptyFunction,
            onDrop:             Prototype.emptyFunction,
            multiselect:        true,
            isSource:           true
        };

        // Empty the cache for each instantiation.
        this.selectedNodes = [];

        options = Object.extend(this.options, options || {});

        $super($container, options);

        if (this.options.isSource) {
            $('body')
                .insert(new Element('div', {
                    id:        'drag-ghost',
                    className: 'bg-white border border-neutral-400 shadow p5'
                }));
        }

        return this;
    },

    /**
     * Method for adding all necessary observers.
     */
    addObserver: function ($super) {
        $super();

        this.$container.on('change', 'input', function (ev) {
            const $input = ev.findElement('input');
            const nodeId = $input.up('li').readAttribute('data-id');

            if ($input.checked) {
                this.selectNode(nodeId);
            } else {
                this.unselectNode(nodeId);
            }

            this.process();
        }.bind(this));
    },

    getSelectedNodes: function () {
        const returnNodes = [];

        for (let i in this.selectedNodes) {
            if (!this.selectedNodes.hasOwnProperty(i)) {
                continue;
            }

            if (this.cache.hasOwnProperty(this.selectedNodes[i])) {
                returnNodes.push(this.cache[this.selectedNodes[i]]);
            }
        }

        return returnNodes;
    },

    /**
     * Method adding a node as "selected".
     *
     * @param   nodeId
     * @returns {Window.LocationTree}
     */
    selectNode: function (nodeId) {
        if (this.options.multiselect) {
            this.selectedNodes.push('n' + nodeId);
        } else {
            this.selectedNodes = ['n' + nodeId];
        }

        this.selectedNodes = this.selectedNodes.uniq();

        this.options.onSelect();

        return this;
    },

    /**
     * Method removing a node as "selected".
     *
     * @param   nodeId
     * @returns {Window.LocationTree}
     */
    unselectNode: function (nodeId) {
        this.selectedNodes = this.selectedNodes.filter((id) => 'n' + nodeId !== id);

        this.options.onSelect();

        return this;
    },

    process: function ($super) {
        $super();

        this.processInputState();
        this.prepareDraggables();
    },

    displayChildrenNodes: function ($super, nodeId) {
        $super(nodeId);

        this.processInputState();
        this.prepareDraggables();
    },

    processInputState: function () {
        if (!this.options.multiselect) {
            return;
        }

        // After the tree was processed, we need to disable all children of selected nodes.
        for (let i in this.selectedNodes) {
            if (!this.selectedNodes.hasOwnProperty(i)) {
                continue;
            }

            this.$container
                .select('li[data-id="' + this.selectedNodes[i].substring(1) + '"] li input')
                .invoke('setValue', 1)
                .invoke('disable');
        }
    },

    /**
     *
     * @param   nodeId
     * @returns {boolean}
     */
    isSelected: function (nodeId) {
        return this.selectedNodes.indexOf('n' + nodeId) !== -1;
    },

    prepareDraggables: function () {
        const $ghost = $('drag-ghost');
        const $nodes = this.$container.select('li[data-id]');

        for (let i in $nodes) {
            if (!$nodes.hasOwnProperty(i)) {
                continue;
            }

            if (this.options.isSource) {
                // Add draggables.
                new Draggable($nodes[i].down('label'), {
                    ghosting: true,
                    revert:   true,
                    zindex:   1200,
                    onStart:  function (drag) {
                        drag.element.addClassName('hide');

                        const $dragList = new Element('ul', { className: 'list-style-none m0 p0' });
                        const selectedNodes = this.getSelectedNodes();

                        $ghost
                            .update($dragList)
                            .appear({duration: 0.2});

                        for (let i in selectedNodes) {
                            if (!selectedNodes.hasOwnProperty(i)) {
                                continue;
                            }

                            $dragList.insert(new Element('li')
                                .update(new Element('img', { src: selectedNodes[i].nodeTypeIcon, className: 'mr5' }))
                                .insert(new Element('span').update(selectedNodes[i].nodeTitle)))
                        }
                    }.bind(this),
                    onDrag:   function (drag) {
                        $ghost
                            .setStyle({
                                left: (12 + window.mouseX) + 'px',
                                top:  window.mouseY + 'px'
                            });
                    },
                    onEnd:    function (drag) {
                        $ghost.fade({duration: 0.2});

                        // This is necessary for the tree, because it sets all elements to their inital height.
                        drag.element.removeClassName('hide')
                            .setStyle({
                                height: 'auto',
                                zIndex: 'auto'
                            });
                    }
                });
            } else {
                // Add droppables.
                Droppables.add($nodes[i].down('label'), {
                    hoverclass: 'drop',
                    onDrop:     function ($source, $destination) {
                        // Instead of duplicating a lot of code... We simply use the existing logic.
                        this.options.onDrop(
                            $source.up('li').readAttribute('data-id'),
                            $destination.up('li').readAttribute('data-id')
                        );
                    }.bind(this)
                });
            }
        }
    },

    /**
     * Method for rendering a node.
     *
     * @param   data
     * @returns {*}
     */
    renderNode: function (data) {
        const isRoot = (!data.nodeId || data.nodeId === 1);
        const open = this.isOpenNode(data.nodeId);
        const selected = this.isSelected(data.nodeId);
        const inputType = this.options.multiselect ? 'checkbox' : 'radio';
        const inputName = this.options.multiselect ? 'location-tree-selection[]' : 'location-tree-selection';

        const $selection = new Element('input', { type: inputType, value: data.nodeId, name: inputName })
            .setValue(selected ? 1 : null);

        const $objectLink = new Element('a', { href: window.www_dir + '?objID=' + data.nodeId, target: '_blank', className: 'ml-auto'})
            .update(new Element('img', { src: window.dir_images + 'axialis/basic/link.svg' }))

        return new Element('li', { 'data-id': data.nodeId })
            .update(new Element('img', { src: this.getToggleImage(open), className: 'child-toggle ' + (data.hasChildren && !isRoot ? '' : 'hide') }))
            .insert(new Element('label', { className: 'tree-inner' + (isRoot ? '' : ' mouse-pointer') + (selected ? ' text-bold' : '') })
                .update(isRoot ? '' : $selection)
                .insert(new Element('img', { src: data.nodeTypeIcon, className: 'mr5' + (isRoot ? '' : ' ml5'), title: data.nodeTypeTitle }))
                .insert(new Element('span').update(data.nodeTitle))
                .insert(isRoot ? '' : $objectLink))
            .insert(new Element('ul', { className: 'css-tree ' + (open ? '' : 'hide') }));
    },

    /**
     * Method for loading children nodes via ajax.
     *
     * @param nodeId
     * @param callback
     */
    loadChildrenNodes: function (nodeId, callback) {
        new Ajax.Request(window.www_dir + 'cmdb/browse-location/' + nodeId, {
            parameters: {
                mode:           'combined',
                onlyContainer:  0,
                considerRights: 0,
                filter:         (this.options.isSource ? '' : 'relocate_ci.target')
            },
            onComplete: function (xhr) {
                if (!is_json_response(xhr, true)) {
                    return;
                }

                const json = xhr.responseJSON;

                // Filter and sort alphabetically.
                const children = json.data.children
                    .sort((a, b) => a.nodeTitle.toLowerCase().localeCompare(b.nodeTitle.toLowerCase()))

                this.cache['n' + nodeId] = json.data;
                this.cache['n' + nodeId].children = children;
                this.cache['n' + nodeId].hasChildren = children.length > 0;

                for (let i in children) {
                    if (!children.hasOwnProperty(i)) {
                        continue;
                    }

                    this.cache['n' + children[i].nodeId] = children[i];
                }

                if (Object.isFunction(callback)) {
                    callback(nodeId, this.cache['n' + nodeId]);
                }

                this.prepareDraggables();
            }.bind(this)
        });
    },

    /**
     * Reset current setup and reload the tree.
     * Keep opened nodes.
     */
    reset: function () {
        this.cache = [];
        this.selectedNodes = [];

        this.process();
    }
});
