<?php

use idoit\Component\Logger;
use idoit\Module\RelocateCi\Model\Ci;
use idoit\Module\RelocateCi\Relocator\LogicalRelocator;
use idoit\Module\RelocateCi\Relocator\PhysicalRelocator;
use Monolog\Handler\TestHandler;

/**
 * i-doit
 * AJAX controller
 *
 * @package     modules
 * @subpackage  relocate_ci
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class isys_ajax_handler_relocate_ci extends isys_ajax_handler
{
    /**
     * @var isys_component_template_language_manager
     */
    private $language;

    /**
     * @var array
     */
    private $response = [];

    /**
     * This method defines, if the hypergate needs to be included for this request.
     *
     * @static
     * @return  boolean
     */
    public static function needs_hypergate()
    {
        return true;
    }

    /**
     * Init method, which gets called from the framework.
     *
     * @throws Exception
     */
    public function init()
    {
        // We set the header information because we don't accept anything than JSON.
        header('Content-Type: application/json');

        $this->language = isys_application::instance()->container->get('language');

        $this->response = [
            'success' => true,
            'message' => null,
            'data'    => null
        ];

        try {
            isys_auth_relocate_ci::instance()
                ->check(isys_auth::EXECUTE, 'relocate_gui');

            switch ($_GET['func']) {
                case 'relocate':
                    $this->relocateObject(
                        isys_format_json::decode($_POST['relocationObjects']),
                        (int)$_POST['destinationObjectId'],
                        isys_format_json::decode($_POST['children'])
                    );
                    break;

                case 'load-tree-level':
                    // The structure needs to look like this, because the response is directly fed to the "eTree" JS class.
                    $this->response = $this->load_tree_level($_GET['tree_name']);
                    break;
            }
        } catch (isys_exception $e) {
            $this->response = [
                'success' => false,
                'message' => $e->getMessage(),
                'data'    => null
            ];
        }

        echo isys_format_json::encode($this->response);

        $this->_die();
    }

    /**
     * This is the advanced "relocate logic" for workstations.
     *
     * @param array $relocationObjectIds
     * @param int   $destinationObjectId
     * @param array $children
     *
     * @throws Exception
     * @throws isys_exception_dao
     * @throws isys_exception_database
     * @throws isys_exception_general
     */
    protected function relocateObject(array $relocationObjectIds, int $destinationObjectId, array $children = []): void
    {
        $database = isys_application::instance()->container->get('database');
        $language = isys_application::instance()->container->get('language');
        $childrenRelocation = [];

        // Prepare a log instance with the 'TestHandler' to work with the results later.
        $logHandler = new TestHandler();
        $log = new Logger('relocate-ci', [$logHandler]);

        $physicalRelocator = new PhysicalRelocator($database, $language, $log);
        $logicalRelocator = new LogicalRelocator($database, $language, $log);

        $ciModel = Ci::instance($database);

        /*
         * We do two things here:
         * 1. We relocate children to their new "fallback parent"
         * 2. We relocate the 'relocation objects' themselves according to their new physical / logical location.
         * 3. Optionally we'll updated all child locations with logbook entries because the parents have been moved.
         */
        foreach ($children as $child) {
            if ($ciModel->usableForPhysicalRelocation($child['newParent'])) {
                $physicalRelocator->relocate($child['id'], $child['newParent']);

                if (isys_tenantsettings::get('relocate-ci.reset-logical-location-after-physical-relocation', 0)) {
                    $logicalRelocator->resetLocation($child['id']);
                }
            } else {
                $logicalRelocator->relocate($child['id'], $child['newParent']);
            }
        }

        foreach ($relocationObjectIds as $relocationObjectId) {
            if ($ciModel->usableForPhysicalRelocation($destinationObjectId)) {
                // @see RELOCATE-41 Prevent detached children of being moved.
                $physicalRelocator->relocate(
                    $relocationObjectId,
                    $destinationObjectId,
                    array_map(fn($child) => $child['id'], $children)
                );

                $childrenRelocation[] = [
                    'id' => $relocationObjectId,
                    'type' => $physicalRelocator::PHYSICALLY
                ];

                if (isys_tenantsettings::get('relocate-ci.reset-logical-location-after-physical-relocation', 0)) {
                    $logicalRelocator->resetLocation($relocationObjectId);
                }
            } else {
                // @see RELOCATE-41 Prevent detached children of being moved.
                $logicalRelocator->relocate(
                    $relocationObjectId,
                    $destinationObjectId,
                    array_map(fn($child) => $child['id'], $children)
                );

                $childrenRelocation[] = [
                    'id' => $relocationObjectId,
                    'type' => $logicalRelocator::LOGICALLY
                ];
            }
        }

        if (isys_tenantsettings::get('relocate-ci.write-logbook-changes-to-children', 0)) {
            $log->debug($language->get('LC__MODULE__RELOCATE_CI__LOG__ATTEMPTING_RECURSIVE_LOGS', count($childrenRelocation)));

            $destinationObject = $ciModel->objectTitle($destinationObjectId);

            foreach ($childrenRelocation as $parentData) {
                $children = $ciModel->getLocatedObjects($parentData['id']);
                $objectName = $ciModel->objectTitle($parentData['id']);

                foreach ($children as $child) {
                    $physicalRelocator->writeRecursiveLogbook($objectName, $destinationObject, $child['id'], $parentData['type']);
                }
            }
        }

        $result = $this->language->get('LC__MODULE__RELOCATE_CI__SUCCESSFULLY_RELOCATED_OBJECTS');

        if (!count($children) && count($relocationObjectIds) === 1) {
            $result = $this->language->get(
                'LC__MODULE__RELOCATE_CI__SUCCESSFULLY_RELOCATED_OBJECT',
                $ciModel->objectTitle($relocationObjectIds[0])
            );
        }

        $this->response['data'] = [
            'message' => $result,
            'log'     => array_map(function ($log) {
                return [
                    'message' => $log['message'],
                    'color' => Logger::getLevelColors($log['level']),
                    'icon' => Logger::getLevelIcons($log['level']),
                ];
            }, $logHandler->getRecords())
        ];
    }

    /**
     * Wrapper method for the "tree_level" handler - this method will wrap the response in a label and include a radio box.
     *
     * @param string $treeName
     *
     * @return array
     */
    protected function load_tree_level(string $treeName): array
    {
        $tree = (new isys_ajax_handler_tree_level($_GET, $_POST))->init(isys_module_request::get_instance());
        $type = $treeName === 'source' ? 'checkbox' : 'radio';

        if (is_array($tree) && count($tree)) {
            foreach ($tree as &$node) {
                $node['text'] = '<label>' .
                    '<input type="' . $type . '" name="' . $treeName . '" value="' . $node['id'] . '" class="mr5" />' .
                    '<span>' . $node['text'] . '</span>' .
                    '</label>';
            }
        }

        return $tree;
    }
}
