<?php

namespace idoit\Module\RelocateCi\Controller;

use Exception;
use idoit\Component\Logger;
use idoit\Module\RelocateCi\Model\Tree;
use idoit\Module\RelocateCi\Relocator\LogicalRelocator;
use isys_application as App;
use isys_auth;
use isys_cmdb_dao_category_g_location;
use isys_cmdb_dao_category_g_logical_unit;
use isys_component_database;
use isys_component_template_language_manager;
use isys_format_json as JSON;
use isys_module_cmdb;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Throwable;

/**
 *
 */
class RelocateCheckController
{
    private isys_component_template_language_manager $language;
    private isys_component_database $database;

    public function checkValidity(Request $request): Response
    {
        try {
            $this->database = App::instance()->container->get('database');
            $this->language = App::instance()->container->get('language');

            $sources = $this->getSources($request->request->get('sources'));
            $target = $this->getTarget($request->request->get('targets'));
            $sourceData = [];

            $this->checkRelations($sources, $target);

            $logicalRelocator = new LogicalRelocator($this->database, $this->language, Logger::factory('Dummy', BASE_DIR . 'temp/'));

            foreach ($sources as $sourceObjectId) {
                $sourceData[] = [
                    'id'     => $sourceObjectId,
                    'parent' => $logicalRelocator->findNextPhysicalLocation($sourceObjectId) ?? C__OBJ__ROOT_LOCATION
                ];
            }

            return new JsonResponse([
                'success' => true,
                'data'    => [
                    'sourceData'          => $sourceData,
                    'destinationObjectId' => $target
                ],
                'message' => ''
            ]);
        } catch (Throwable $e) {
            return new JsonResponse([
                'success' => false,
                'data'    => null,
                'message' => $e->getMessage()
            ]);
        }
    }

    /**
     * @param string $sources
     *
     * @return array
     * @throws \idoit\Exception\JsonException
     * @throws \isys_exception_auth
     */
    private function getSources(string $sources): array
    {
        if (!JSON::is_json_array($sources)) {
            throw new Exception($this->language->get('LC__MODULE__RELOCATE_CI__INVALID_DATA_GIVEN'));
        }

        $sources = array_map(fn ($item) => (int)$item, JSON::decode($sources));

        if (!count($sources)) {
            throw new Exception($this->language->get('LC__MODULE__RELOCATE_CI__YOU_NEED_TO_SELECT_AT_LEAST_TWO_OBJECTS'));
        }

        $auth = isys_module_cmdb::getAuth();

        foreach ($sources as $sourceObjectId) {
            $auth->check_rights_obj_and_category(isys_auth::EDIT, $sourceObjectId, 'C__CATG__LOCATION');
            $auth->check_rights_obj_and_category(isys_auth::EDIT, $sourceObjectId, 'C__CATG__LOGICAL_UNIT');
        }

        return $sources;
    }

    /**
     * @param string $targets
     *
     * @return int
     * @throws \idoit\Exception\JsonException
     */
    private function getTarget(string $targets): int
    {
        if (!JSON::is_json_array($targets)) {
            throw new Exception($this->language->get('LC__MODULE__RELOCATE_CI__INVALID_DATA_GIVEN'));
        }

        $targets = array_map(fn ($item) => (int)$item, JSON::decode($targets));

        if (!count($targets) || count($targets) > 1) {
            throw new Exception($this->language->get('LC__MODULE__RELOCATE_CI__YOU_NEED_TO_SELECT_AT_LEAST_TWO_OBJECTS'));
        }

        return current($targets);
    }

    /**
     * @param array $sources
     * @param int   $target
     *
     * @return void
     * @throws \isys_exception_database
     */
    private function checkRelations(array $sources, int $target): void
    {
        // Check if we try to relocate an object to itself.
        if (in_array($target, $sources, true)) {
            throw new Exception($this->language->get('LC__MODULE__RELOCATE_CI__YOU_CANNOT_MOVE_AN_OBJECT_TO_ITSELF'));
        }

        $locationDao = isys_cmdb_dao_category_g_location::instance($this->database);
        $loigalUnitDao = isys_cmdb_dao_category_g_logical_unit::instance($this->database);
        $treeModel = Tree::instance($this->database);

        $targetPhysicalPath = array_map(fn ($item) => $item['id'], $treeModel->getPhysicalPath($target));
        $targetLogicalPath = array_map(fn ($item) => $item['id'], $treeModel->getLogicalPath($target));

        foreach ($sources as $sourceObjectId) {
            $physicalParent = $locationDao->get_parent_id_by_object($sourceObjectId);
            $logicalParent = $loigalUnitDao->get_parent_by_id($sourceObjectId);

            // Check if we try to relocate and object to its current position.
            if ($target == $physicalParent || $target == $logicalParent) {
                throw new Exception($this->language->get('LC__MODULE__RELOCATE_CI__YOU_ARE_TRYING_TO_MOVE_AN_OBJECT_TO_ITS_CURRENT_LOCATION'));
            }

            if (in_array($sourceObjectId, $targetPhysicalPath, true) || in_array($sourceObjectId, $targetLogicalPath, true)) {
                throw new Exception($this->language->get('LC__MODULE__RELOCATE_CI__YOU_CANNOT_MOVE_AN_OBJECT_UNDERNEATH_ITSELF'));
            }
        }
    }
}
