<?php

/**
 * i-doit
 *
 * Auth: Class for CMDB module authorization rules.
 *
 * @package     i-doit
 * @subpackage  auth
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @version     1.1.0
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       i-doit 1.1.2
 */
class isys_auth_swapci extends isys_auth implements isys_auth_interface
{
    /**
     * Container for singleton instance
     *
     * @var isys_auth_swapci
     */
    private static $m_instance;

    /**
     * Method for returning the available auth-methods. This will be used for the GUI.
     *
     * @return  array
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function get_auth_methods()
    {
        return [
            'swap_gui'              => [
                'title'  => 'LC__AUTH_GUI__SWAP_GUI',
                'type'   => 'boolean',
                'rights' => [isys_auth::VIEW, isys_auth::EXECUTE, isys_auth::SUPERVISOR]
            ],
            'swap_gui_by_workplace' => [
                'title'  => 'LC__AUTH_GUI__SWAP_GUI_BY_WORKPLACE',
                'type'   => 'boolean',
                'rights' => [isys_auth::VIEW, isys_auth::EXECUTE, isys_auth::SUPERVISOR]
            ],
            'swap_config'           => [
                'title'  => 'LC__AUTH_GUI__SWAP_CONFIG',
                'type'   => 'boolean',
                'rights' => [isys_auth::VIEW, isys_auth::EDIT, isys_auth::SUPERVISOR]
            ],
            'swap_in_list'          => [
                'title'  => 'LC__AUTH_GUI__SWAP_IN_LIST',
                'type'   => 'boolean',
                'rights' => [isys_auth::VIEW]
            ],
            'swap_in_obj'           => [
                'title'  => 'LC__AUTH_GUI__SWAP_IN_OBJ',
                'type'   => 'boolean',
                'rights' => [isys_auth::VIEW]
            ]
        ];
    }

    /**
     * Method for checking, if the user is allowed to open and execute the SwapCI-GUI.
     *
     * @param   integer $p_right
     *
     * @return  boolean
     * @throws  isys_exception_auth
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function swap_gui($p_right)
    {
        if (!$this->is_auth_active()) {
            return true;
        }

        $l_right_name = isys_auth::get_right_name($p_right);

        return $this->generic_right(
            $p_right,
            'swap_gui',
            self::EMPTY_ID_PARAM,
            new isys_exception_auth(_L('LC__AUTH__SWAPCI_EXCEPTION__MISSING_RIGHT_FOR_SWAP_GUI', [$l_right_name]))
        );
    }

    /**
     * Method for checking, if the user is allowed to open and execute the SwapCI-GUI for workplaces.
     *
     * @param   integer $p_right
     *
     * @return  boolean
     * @throws  isys_exception_auth
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function swap_gui_by_workplace($p_right)
    {
        if (!$this->is_auth_active()) {
            return true;
        }

        $l_right_name = isys_auth::get_right_name($p_right);

        return $this->generic_right(
            $p_right,
            'swap_gui_by_workplace',
            self::EMPTY_ID_PARAM,
            new isys_exception_auth(_L('LC__AUTH__SWAPCI_EXCEPTION__MISSING_RIGHT_FOR_SWAP_GUI_BY_WORKPLACES', [$l_right_name]))
        );
    }

    /**
     * Method for checking, if the user is allowed to open and execute the SwapCI-configuration.
     *
     * @param   integer $p_right
     *
     * @return  boolean
     * @throws  isys_exception_auth
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function swap_config($p_right)
    {
        if (!$this->is_auth_active()) {
            return true;
        }

        $l_right_name = isys_auth::get_right_name($p_right);

        return $this->generic_right(
            $p_right,
            'swap_config',
            self::EMPTY_ID_PARAM,
            new isys_exception_auth(_L('LC__AUTH__SWAPCI_EXCEPTION__MISSING_RIGHT_FOR_SWAP_CONFIG', [$l_right_name]))
        );
    }

    /**
     * Method for checking, if the user is allowed to see/use the swap-button in the navbar.
     *
     * @param   integer $p_right
     *
     * @return  boolean
     * @throws  isys_exception_auth
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function swap_in_list($p_right)
    {
        if (!$this->is_auth_active()) {
            return true;
        }

        return $this->generic_right(
            $p_right,
            'swap_in_list',
            self::EMPTY_ID_PARAM,
            new isys_exception_auth(_L('LC__AUTH__SWAPCI_EXCEPTION__MISSING_RIGHT_FOR_SWAP_BUTTON'))
        );
    }

    /**
     * Method for checking, if the user is allowed to see/use the swap-button in the navbar.
     *
     * @param   integer $p_right
     *
     * @return  boolean
     * @throws  isys_exception_auth
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function swap_in_obj($p_right)
    {
        if (!$this->is_auth_active()) {
            return true;
        }

        return $this->generic_right(
            $p_right,
            'swap_in_obj',
            self::EMPTY_ID_PARAM,
            new isys_exception_auth(_L('LC__AUTH__SWAPCI_EXCEPTION__MISSING_RIGHT_FOR_SWAP_BUTTON'))
        );
    }

    /**
     * Retrieve singleton instance of authorization class
     *
     * @return isys_auth_swapci
     * @author Selcuk Kekec <skekec@i-doit.com>
     */
    public static function instance()
    {
        // If the DAO has not been loaded yet, we initialize it now.
        if (self::$m_dao === null) {
            self::$m_dao = new isys_auth_dao(isys_application::instance()->container->get('database'));
        }

        if (self::$m_instance === null) {
            self::$m_instance = new self;
        }

        return self::$m_instance;
    }

    /**
     * Get ID of related module.
     *
     * @todo should be part of the interfaces as of version 1.6
     *
     * @return  integer
     */
    public function get_module_id()
    {
        return C__MODULE__SWAPCI;
    }

    /**
     * Get title of related module.
     *
     * @todo should be part of the interfaces as of version 1.6
     *
     * @return  string
     */
    public function get_module_title()
    {
        return 'LC__MODULE__SWAPCI';
    }
}
