<?php

/**
 * AJAX
 *
 * @package     i-doit
 * @subpackage  General
 * @author      Leonard Fischer <lfischer@i-doit.org>
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class isys_ajax_handler_swapci extends isys_ajax_handler
{
    /**
     * Init method, which gets called from the framework.
     *
     * @author  Leonard Fischer <lfischer@i-doit.org>
     */
    public function init()
    {
        // We set the header information because we don't accept anything than JSON.
        header('Content-Type: application/json');

        $l_return = [
            'success' => true,
            'message' => null,
            'data'    => null
        ];

        try {
            switch ($_GET['func']) {
                case 'retrieve_selected_categories':
                    $l_return['data'] = $this->retrieve_selected_categories($_POST['obj_type'], $_POST['with_data_from'] ?: false);
                    break;

                case 'get_second_obj_browser':
                    $l_return['data'] = $this->get_second_obj_browser($_POST['swap_object']);
                    break;

                case 'match_object_types':
                    $l_return['data'] = $this->match_object_types($_POST['obj_a'], $_POST['obj_b']);
                    break;

                case 'swap':
                    $l_return['data'] = $this->swap([
                        'old_obj'             => $_POST['old_obj'],
                        'new_obj'             => $_POST['new_obj'],
                        'obj_type'            => $_POST['obj_type'],
                        'categories'          => $_POST['categories'],
                        'consider_status'     => $_POST['consider_status'],
                        'archive_swapped_obj' => $_POST['archive_swapped_obj'],
                        'swap_sysid'          => $_POST['swap_sysid']
                    ]);
                    break;

                case 'get_workplaces_by_obj_type':
                    $l_return['data'] = $this->get_workplaces_by_obj_type($_POST['obj_type']);
                    break;
            }
        } catch (Exception $e) {
            $l_return['success'] = false;
            $l_return['message'] = $e->getMessage();
        }

        echo isys_format_json::encode($l_return);

        $this->_die();
    }

    /**
     * This method defines, if the hypergate needs to be included for this request.
     *
     * @static
     * @return  boolean
     * @author  Leonard Fischer <lfischer@i-doit.org>
     */
    public static function needs_hypergate()
    {
        return true;
    }

    /**
     * Method for retrieving all selected categories of a given object-type. This also includes "sub"-categories.
     *
     * @param   string $p_obj_type
     * @param   mixed  $p_with_data_from
     *
     * @return  array
     * @throws  isys_exception_general
     * @author  Leonard Fischer <lfischer@i-doit.org>
     */
    protected function retrieve_selected_categories($p_obj_type, $p_with_data_from = false)
    {
        if (!defined($p_obj_type)) {
            throw new isys_exception_general('The given constant ("' . $p_obj_type . '") could not be found!');
        }

        $l_return = [];
        $l_blacklist = isys_swapci_swap::CATEGORY_BLACKLIST;
        $l_obj_type_id = constant($p_obj_type);

        // This variable holds all selected categories by the given object-type.
        $l_selected_categories = isys_swapci_dao::instance($this->m_database_component)
            ->get_config('objtype_' . $p_obj_type);

        if (!is_array($l_selected_categories)) {
            $l_selected_categories = [];
        }

        $l_cmdb_dao = isys_cmdb_dao::instance($this->m_database_component);

        $l_catg_res = $l_cmdb_dao->gui_get_catg_by_objtype_id($l_obj_type_id);

        if (count($l_catg_res) > 0) {
            while ($l_row = $l_catg_res->get_row()) {
                $l_subcat_res = $l_cmdb_dao->catg_get_subcats($l_row['isysgui_catg__id']);

                if (count($l_subcat_res) > 0) {
                    while ($l_subcat_row = $l_subcat_res->get_row()) {
                        $l_subcat = $l_cmdb_dao->get_all_catg($l_subcat_row['isysgui_catg__id'])
                            ->get_row();

                        if (in_array($l_subcat['isysgui_catg__const'], $l_blacklist, true)) {
                            continue;
                        }

                        if ($p_with_data_from !== false && $this->category_rows_by_obj($l_subcat['isysgui_catg__source_table'] . '_list', $p_with_data_from) === 0) {
                            continue;
                        }

                        $l_return[$l_subcat['isysgui_catg__const']] = in_array($l_subcat['isysgui_catg__const'], $l_selected_categories, true);
                    }
                }

                if ($p_with_data_from !== false && $this->category_rows_by_obj($l_row['isysgui_catg__source_table'] . '_list', $p_with_data_from) === 0) {
                    continue;
                }

                if (in_array($l_row['isysgui_catg__const'], $l_blacklist, true)) {
                    continue;
                }

                $l_return[$l_row['isysgui_catg__const']] = in_array($l_row['isysgui_catg__const'], $l_selected_categories, true);
            }
        }

        $l_catc_res = $l_cmdb_dao->gui_get_catg_custom_by_objtype_id($l_obj_type_id);

        if (count($l_catc_res) > 0) {
            while ($l_row = $l_catc_res->get_row()) {
                if ($p_with_data_from !== false && $this->category_rows_by_obj('isys_catg_custom_fields_list', $p_with_data_from) === 0) {
                    continue;
                }

                $l_return[$l_row['isysgui_catg_custom__const']] = in_array($l_row['isysgui_catg_custom__const'], $l_selected_categories, true);
            }
        }

        $l_cats_res = $l_cmdb_dao->gui_get_cats_by_objtype_id($l_obj_type_id);

        if (count($l_cats_res) > 0) {
            while ($l_row = $l_cats_res->get_row()) {
                $l_subcat_res = $l_cmdb_dao->cats_get_subcats($l_row['isysgui_cats__id']);

                if (count($l_subcat_res) > 0) {
                    while ($l_subcat_row = $l_subcat_res->get_row()) {
                        $l_subcat = $l_cmdb_dao->get_all_cats($l_subcat_row['isysgui_cats__id'])
                            ->get_row();

                        if (in_array($l_subcat['isysgui_cats__const'], $l_blacklist, true)) {
                            continue;
                        }

                        if ($p_with_data_from !== false && $this->category_rows_by_obj($l_subcat['isysgui_cats__source_table'], $p_with_data_from) === 0) {
                            continue;
                        }

                        $l_return[$l_subcat['isysgui_cats__const']] = in_array($l_subcat['isysgui_cats__const'], $l_selected_categories, true);
                    }
                }

                if ($p_with_data_from !== false && $this->category_rows_by_obj($l_row['isysgui_cats__source_table'], $p_with_data_from) === 0) {
                    continue;
                }

                if (in_array($l_row['isysgui_cats__const'], $l_blacklist, true)) {
                    continue;
                }

                $l_return[$l_row['isysgui_cats__const']] = in_array($l_row['isysgui_cats__const'], $l_selected_categories, true);
            }
        }

        return $l_return;
    }

    protected function get_second_obj_browser($p_first_obj)
    {
        global $g_comp_template, $g_comp_database;

        $l_cmdb_status = isys_factory::get_instance('isys_swapci_dao', $g_comp_database)
            ->get_config('swap_obj_status');

        $l_obj_type = isys_factory::get_instance('isys_cmdb_dao', $this->m_database_component)
            ->get_type_by_object_id($p_first_obj)
            ->get_row();

        $l_popup = new isys_smarty_plugin_f_popup();
        $l_params = [
            'name'                                       => 'C__MODULE__SWAPCI__STORED_OBJECT',
            'p_strPopupType'                             => 'browser_object_ng',
            isys_popup_browser_object_ng::C__EDIT_MODE   => true,
            isys_popup_browser_object_ng::C__TYPE_FILTER => $l_obj_type['isys_obj_type__const'],
            isys_popup_browser_object_ng::C__CMDB_FILTER => implode(';', $l_cmdb_status),
            'callback_accept'                            => 'window.obj_browser_callback2();'
        ];

        return $l_popup->navigation_edit($g_comp_template, $l_params);
    }

    /**
     * Method for retrieving the amount of category-rows, by a given object ID.
     *
     * @param   string  $p_category_table
     * @param   integer $p_obj_id
     *
     * @return  integer
     * @author  Leonard Fischer <lfischer@i-doit.org>
     */
    protected function category_rows_by_obj($p_category_table, $p_obj_id)
    {
        $l_dao = isys_cmdb_dao::instance($this->m_database_component);

        try {
            // First check, if the table exists.
            if (count($l_dao->retrieve('SHOW TABLES LIKE "' . $p_category_table . '";')) === 0) {
                return 0;
            }

            // Then check, if the column exists.
            if (count($l_dao->retrieve('SHOW COLUMNS FROM ' . $p_category_table . ' LIKE "' . $p_category_table . '__isys_obj__id";')) === 0) {
                return 0;
            }

            return (int)$l_dao
                ->retrieve('SELECT COUNT(*) AS cnt FROM ' . $p_category_table . ' WHERE ' . $p_category_table . '__isys_obj__id = ' . $l_dao->convert_sql_id($p_obj_id) . ';')
                ->get_row_value('cnt');
        } catch (Exception $e) {
            // @todo  Should we display the exception message?
        }

        return 0;
    }

    /**
     * Method to check, if the two given objects are of the same type.
     *
     * @param   integer $p_obj_a
     * @param   integer $p_obj_b
     *
     * @return  string
     * @throws  isys_exception_general
     * @author  Leonard Fischer <lfischer@i-doit.org>
     */
    protected function match_object_types($p_obj_a, $p_obj_b)
    {
        if ($p_obj_a == $p_obj_b) {
            throw new isys_exception_general(_L('LC__MODULE__SWAPCI__GUI_ERROR_OBJECTS_ARE_IDENTICAL'));
        }

        $l_obj_a = isys_factory::get_instance('isys_cmdb_dao', $this->m_database_component)
            ->get_object($p_obj_a, true)
            ->get_row();
        $l_obj_b = isys_factory::get_instance('isys_cmdb_dao', $this->m_database_component)
            ->get_object($p_obj_b, true)
            ->get_row();

        if ($l_obj_a['isys_obj_type__id'] != $l_obj_b['isys_obj_type__id']) {
            throw new isys_exception_general(_L('LC__MODULE__SWAPCI__GUI_ERROR_OBJ_TYPE_MISMATCH'));
        }

        // Final check: Is the user allowed to edit the objects?
        isys_auth_cmdb::instance()
            ->check(isys_auth::EDIT, 'OBJ_ID/' . $p_obj_a);
        isys_auth_cmdb::instance()
            ->check(isys_auth::EDIT, 'OBJ_ID/' . $p_obj_b);

        return $l_obj_a['isys_obj_type__const'];
    }

    /**
     * AJAX method for starting the swap-process.
     *
     * @param   array $p_options
     *
     * @return  array
     * @author  Leonard Fischer <lfischer@i-doit.org>
     */
    protected function swap(array $p_options)
    {
        if (is_string($p_options['categories'])) {
            $p_options['categories'] = isys_format_json::decode($p_options['categories'], true);
        }

        $l_results = (new isys_swapci_swap())->init($p_options)
            ->swap()
            ->get_results();

        // Because this is an ajax request, we need to encode all strings (because of possible umlaute).
        $l_results['old']['title'] = $l_results['old']['title'];
        $l_results['new']['title'] = $l_results['new']['title'];

        foreach ($l_results['log'] as &$l_log) {
            $l_log['message'] = $l_log['message'];
        }

        return $l_results;
    }

    /**
     * Method for retrieving all workplaces, which inherit an object of the given object type.
     *
     * @param   integer $p_obj_type
     *
     * @return  array
     * @author  Leonard Fischer <lfischer@i-doit.org>
     */
    protected function get_workplaces_by_obj_type($p_obj_type)
    {
        global $g_comp_database, $g_dirs;

        $l_return = [];
        $l_dao = isys_factory::get_instance('isys_cmdb_dao_category_g_logical_unit', $g_comp_database);
        $l_allowed_obj_types = isys_factory::get_instance('isys_swapci_dao', $g_comp_database)
            ->get_config('available_obj_types');
        $l_workplace_cache = [];

        $l_sql = 'SELECT workplace.*, workplace_type.* FROM isys_catg_logical_unit_list
			INNER JOIN isys_obj AS workplace ON workplace.isys_obj__id = isys_catg_logical_unit_list__isys_obj__id__parent
			INNER JOIN isys_obj_type AS workplace_type ON workplace_type.isys_obj_type__id = workplace.isys_obj__isys_obj_type__id
			INNER JOIN isys_obj AS object ON object.isys_obj__id = isys_catg_logical_unit_list__isys_obj__id
			WHERE workplace.isys_obj__isys_obj_type__id = ' . $l_dao->convert_sql_id(C__OBJTYPE__WORKSTATION) . '
			AND object.isys_obj__isys_obj_type__id = ' . $l_dao->convert_sql_id($p_obj_type) . ';';

        $l_res = $l_dao->retrieve($l_sql);

        if (count($l_res) > 0) {
            while ($l_row = $l_res->get_row()) {
                if (in_array($l_row['isys_obj__id'], $l_workplace_cache)) {
                    continue;
                }

                $l_workplace_cache[] = $l_row['isys_obj__id'];

                $l_workplace = [
                    'id'        => $l_row['isys_obj__id'],
                    'name'      => $l_row['isys_obj__title'],
                    'type_name' => _L($l_row['isys_obj_type__title']),
                    'icon'      => (file_exists($l_row['isys_obj_type__icon']) ? $l_row['isys_obj_type__icon'] : $g_dirs['images'] . 'tree/' . $l_row['isys_obj_type__icon']),
                    'children'  => []
                ];

                $l_child_res = $l_dao->get_data_by_parent($l_row['isys_obj__id']);

                if (count($l_child_res) > 0) {
                    while ($l_child_row = $l_child_res->get_row()) {
                        $l_workplace['children'][$l_child_row['isys_obj__title']] = [
                            'id'              => $l_child_row['isys_obj__id'],
                            'name'            => $l_child_row['isys_obj__title'],
                            'type_name'       => _L($l_child_row['isys_obj_type__title']),
                            'icon'            => (file_exists($l_child_row['isys_obj_type__icon']) ? $l_child_row['isys_obj_type__icon'] : $g_dirs['images'] . 'tree/' .
                                $l_child_row['isys_obj_type__icon']),
                            'allowed_to_swap' => (in_array($l_child_row['isys_obj_type__const'], $l_allowed_obj_types))
                        ];
                    }

                    ksort($l_workplace['children']);

                    $l_workplace['children'] = array_values($l_workplace['children']);
                }

                $l_return[$l_row['isys_obj__title']] = $l_workplace;
            }
        }

        ksort($l_return);

        return array_values($l_return);
    }
}
