<?php

/**
 * i-doit
 *
 * "Swap CI" module class - a.k.a "cold standby".
 *
 * @package     i-doit
 * @subpackage  Modules
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @version     1.2.0
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       i-doit 1.2.3
 */
class isys_module_swapci extends isys_module implements isys_module_authable
{
    /**
     * Define, if this module shall be displayed in the extras menu.
     *
     * @var  boolean
     */
    const DISPLAY_IN_MAIN_MENU = true;

    /**
     * Defines, if this module shall be displayed in the system menu
     *
     * @var  boolean
     */
    const DISPLAY_IN_SYSTEM_MENU = true;

    /**
     * Instance of "isys_swapci_dao".
     *
     * @var  isys_swapci_dao
     */
    protected $m_dao;

    /**
     * Variable which holds the template component.
     *
     * @var  isys_component_template
     */
    protected $template = null;

    /**
     * Variable which holds the database component.
     *
     * @var  isys_component_database
     */
    protected $database = null;

    /**
     * Method for initialization.
     *
     * @param   isys_module_request $p_req
     *
     * @return  boolean
     */
    public function init(isys_module_request &$p_req)
    {
        $this->database = isys_application::instance()->container->get('database');
        $this->template = isys_application::instance()->container->get('template');
        $this->m_dao = isys_swapci_dao::instance($this->database);

        return true;
    }

    /**
     * This method builds the tree for the menu.
     *
     * @param   isys_component_tree $p_tree
     * @param   boolean             $p_system_module
     * @param   integer             $p_parent
     *
     * @author  Leonard Fischer <lfischer@i-doit.org>
     * @see     isys_module::build_tree()
     */
    public function build_tree(isys_component_tree $p_tree, $p_system_module = true, $p_parent = null)
    {
        $l_parent = -1;

        if ($p_system_module) {
            $l_parent = $p_tree->find_id_by_title(_L('LC__AUTH_GUI__GLOBALSETTINGS_CONDITION'));
        }

        if ($p_parent !== null) {
            $l_root = (int)$p_parent;
        } else {
            $l_root = $p_tree->add_node(
                C__MODULE__SWAPCI . '0',
                $l_parent,
                _L('LC__MODULE__SWAPCI'),
                '',
                '',
                '',
                0,
                '',
                '',
                isys_auth_swapci::instance()->has_any_rights_in_module()
            );
        }

        if ($p_system_module) {
            $p_tree->add_node(
                C__MODULE__SWAPCI . '1',
                $l_root,
                _L('LC__MODULE__SWAPCI__CONFIG'),
                isys_helper_link::create_url([
                    C__GET__MODULE_ID     => C__MODULE__SYSTEM,
                    C__GET__MODULE_SUB_ID => C__MODULE__SWAPCI,
                    C__GET__TREE_NODE     => C__MODULE__SWAPCI . '1',
                    C__GET__SETTINGS_PAGE => 'swap_config'
                ]),
                '',
                '',
                0,
                '',
                '',
                isys_auth_swapci::instance()->is_allowed_to(isys_auth::EDIT, 'SWAP_CONFIG')
            );
        } else {
            $p_tree->add_node(
                C__MODULE__SWAPCI . '2',
                $l_root,
                _L('LC__MODULE__SWAPCI__GUI'),
                isys_helper_link::create_url([
                    C__GET__MODULE_ID     => C__MODULE__SWAPCI,
                    C__GET__TREE_NODE     => C__MODULE__SWAPCI . '2',
                    C__GET__SETTINGS_PAGE => 'swap_gui'
                ]),
                '',
                '',
                0,
                '',
                '',
                isys_auth_swapci::instance()->is_allowed_to(isys_auth::EXECUTE, 'SWAP_GUI')
            );

            $p_tree->add_node(
                C__MODULE__SWAPCI . '3',
                $l_root,
                _L('LC__MODULE__SWAPCI__GUI_BY_WORKPLACE'),
                isys_helper_link::create_url([
                    C__GET__MODULE_ID     => C__MODULE__SWAPCI,
                    C__GET__TREE_NODE     => C__MODULE__SWAPCI . '3',
                    C__GET__SETTINGS_PAGE => 'swap_gui_by_workplace'
                ]),
                '',
                '',
                0,
                '',
                '',
                isys_auth_swapci::instance()->is_allowed_to(isys_auth::EXECUTE, 'SWAP_GUI_BY_WORKPLACE')
            );

            $p_tree->add_node(
                C__MODULE__SWAPCI . '4',
                $l_root,
                _L('LC__MODULE__SWAPCI__REPORT'),
                isys_helper_link::create_url([
                    C__GET__MODULE_ID        => C__MODULE__REPORT,
                    C__GET__REPORT_PAGE      => C__REPORT_PAGE__VIEWS,
                    C__GET__REPORT_REPORT_ID => 'isys_swapci_reportview_show_objects'
                ]),
                '',
                '',
                0,
                '',
                ''
            );
        }
    }

    /**
     * @return isys_module|void
     * @throws isys_exception_auth
     */
    public function start()
    {
        $l_modreq = isys_module_request::get_instance();

        // Build the menutree, if we are inside the module itself instead of the system.
        if ($_GET[C__GET__MODULE_ID] != C__MODULE__SYSTEM) {
            $l_tree = $l_modreq->get_menutree();

            $this->build_tree($l_tree, false);

            $l_modreq->get_template()
                ->assign("menu_tree", $l_tree->process($_GET[C__GET__TREE_NODE]));
        }

        switch ($_GET[C__GET__SETTINGS_PAGE]) {
            case 'swap_gui':
                isys_auth_swapci::instance()
                    ->check(isys_auth::EXECUTE, 'SWAP_GUI');

                $this->process_gui();
                break;

            case 'swap_gui_by_workplace':
                isys_auth_swapci::instance()
                    ->check(isys_auth::EXECUTE, 'SWAP_GUI_BY_WORKPLACE');

                $this->process_gui_by_workplace();
                break;

            case 'swap_config':
                isys_auth_swapci::instance()
                    ->check(isys_auth::EDIT, 'SWAP_CONFIG');

                $this->process_config();
                break;
        }

        $l_modreq->get_template()
            ->assign('module_www_dir', self::getWwwPath() . 'templates/');
    }

    /**
     * Method for preparing the "swap" GUI.
     *
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    private function process_gui()
    {
        $l_rules = [];

        // Load the "Swap CI" object type configuration.
        $l_config = $this->m_dao->get_config('available_obj_types');

        if (!is_array($l_config)) {
            $l_config = [];
        }

        $l_yes_no = get_smarty_arr_YES_NO();

        // Create some rules for the different form-fields.
        $l_rules['C__MODULE__SWAPCI__CONFIG_CONSIDER_STATUS']['p_arData'] = $l_yes_no;
        $l_rules['C__MODULE__SWAPCI__CONFIG_SWAP_SYSID']['p_arData'] = $l_yes_no;
        $l_rules['C__MODULE__SWAPCI__CONFIG_ARCHIVE_SWAPPED_OBJ']['p_arData'] = $l_yes_no;

        // And add values from the configuration.
        $l_rules['C__MODULE__SWAPCI__OBJECT'][isys_popup_browser_object_ng::C__TYPE_FILTER] = implode(';', $l_config);
        $l_rules['C__MODULE__SWAPCI__CONFIG_CONSIDER_STATUS']['p_strSelectedID'] = $this->m_dao->get_config('consider_status');
        $l_rules['C__MODULE__SWAPCI__CONFIG_ARCHIVE_SWAPPED_OBJ']['p_strSelectedID'] = $this->m_dao->get_config('archive_swapped_obj');
        $l_rules['C__MODULE__SWAPCI__CONFIG_SWAP_SYSID']['p_strSelectedID'] = $this->m_dao->get_config('swap_sysid');

        if ($_GET[C__CMDB__GET__OBJECT]) {
            list($l_old, $l_new) = explode(',', $_GET[C__CMDB__GET__OBJECT]);

            $l_rules['C__MODULE__SWAPCI__OBJECT']['p_strValue'] = $l_old;
            $l_rules['C__MODULE__SWAPCI__STORED_OBJECT']['p_strValue'] = $l_new;

            $this->template->assign('old_is_filled', ($l_old > 0))
                ->assign('new_is_filled', ($l_new > 0))
                ->assign('from_cmdb_module', true);
        }

        $this->template->activate_editmode()
            ->assign('log_debug_level', isys_log::C__DEBUG)
            ->assign('log_icons', isys_format_json::encode(isys_log::get_log_icons()))
            ->assign('ajax_url', isys_helper_link::create_url([C__GET__AJAX_CALL => 'swapci', C__GET__AJAX => 1]))
            ->assign('categories', isys_format_json::encode(array_values($this->get_all_categories())))
            ->smarty_tom_add_rules('tom.content.bottom.content', $l_rules)
            ->include_template('contentbottomcontent', self::getPath() . 'templates/swap_gui.tpl');
    }

    /**
     *
     */
    private function process_gui_by_workplace()
    {
        $l_cmdb_dao = isys_factory::get_instance('isys_cmdb_dao', $this->database);
        $l_config = $this->m_dao->get_config();

        if (!is_array($l_config)) {
            $l_config = [];
        }

        $l_obj_types = [];

        if (is_array($l_config['available_obj_types'])) {
            foreach ($l_config['available_obj_types'] as $l_obj_type) {
                if (defined($l_obj_type)) {
                    $l_obj_types[constant($l_obj_type)] = $l_cmdb_dao->get_objtype_name_by_id_as_string(constant($l_obj_type));
                }
            }
        }

        $this->template->activate_editmode()
            ->assign('ajax_url', isys_helper_link::create_url([C__GET__AJAX_CALL => 'swapci', C__GET__AJAX => 1]))
            ->assign('swap_gui_url', isys_helper_link::create_url([C__GET__MODULE_ID => C__MODULE__SWAPCI, C__GET__SETTINGS_PAGE => 'swap_gui'], true))
            ->smarty_tom_add_rules('tom.content.bottom.content', ['object_type_selection' => ['p_arData' => $l_obj_types]])
            ->include_template('contentbottomcontent', self::getPath() . 'templates/swap_gui_by_workplace.tpl');
    }

    /**
     * Method for preparing the "configuration" GUI.
     *
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    private function process_config()
    {
        if ($_POST[C__GET__NAVMODE] == C__NAVMODE__NEW || $_POST[C__GET__NAVMODE] == C__NAVMODE__EDIT) {
            isys_component_template_navbar::getInstance()
                ->set_active(true, C__NAVBAR_BUTTON__SAVE)
                ->set_active(true, C__NAVBAR_BUTTON__CANCEL);
        } else {
            isys_component_template_navbar::getInstance()
                ->set_active(true, C__NAVBAR_BUTTON__EDIT);
        }

        if ($_POST[C__GET__NAVMODE] == C__NAVMODE__SAVE) {
            $l_obj_type_config = [];
            $l_config = [
                'available_obj_types' => ['type' => C__TYPE__JSON, 'value' => $_POST['C__MODULE__SWAPCI__AVAILABLE_OBJECT_TYPES']],
                'consider_status'     => ['type' => C__TYPE__INT, 'value' => $_POST['C__MODULE__SWAPCI__CONFIG_CONSIDER_STATUS']],
                'archive_swapped_obj' => ['type' => C__TYPE__INT, 'value' => $_POST['C__MODULE__SWAPCI__CONFIG_ARCHIVE_SWAPPED_OBJ']],
                'swap_sysid'          => ['type' => C__TYPE__INT, 'value' => $_POST['C__MODULE__SWAPCI__CONFIG_SWAP_SYSID']],
                'swap_obj_status'     => [
                    'type'  => C__TYPE__JSON,
                    'value' => isys_format_json::encode(explode(',', $_POST['C__MODULE__SWAPCI__CONFIG_STORE_STATUS__selected_values']))
                ]
            ];

            // Here we select all POST data, with keys starting with "objtype_".
            foreach ($_POST as $l_key => $l_value) {
                if (strpos($l_key, 'objtype_') === 0) {
                    $l_obj_type_config[$l_key] = $l_value;
                }
            }

            $this->m_dao->save_config($l_config)
                ->save_obj_type_config($l_obj_type_config)
                ->reload_config();
        }

        $l_rules = $l_tpl_obj_types = [];

        // Load the "Swap CI" object type configuration.
        $l_config = $this->m_dao->get_config('available_obj_types');
        $l_cmdb_filter = $this->m_dao->get_config('swap_obj_status');

        if (!is_array($l_config)) {
            $l_config = [];
        }

        $l_cmdb_status = [];

        $l_cmdb_res = isys_factory::get_instance('isys_cmdb_dao_status', $this->database)
            ->get_cmdb_status();

        if (count($l_cmdb_res) > 0) {
            while ($l_cmdb_row = $l_cmdb_res->get_row()) {
                $l_cmdb_status[] = [
                    'id'  => $l_cmdb_row['isys_cmdb_status__const'],
                    'val' => _L($l_cmdb_row['isys_cmdb_status__title']),
                    'sel' => (in_array($l_cmdb_row['isys_cmdb_status__const'], $l_cmdb_filter, true))
                ];
            }
        }

        $l_yes_no = get_smarty_arr_YES_NO();

        // Create some rules for the different form-fields.
        $l_rules['C__MODULE__SWAPCI__CONFIG_CONSIDER_STATUS']['p_arData'] = $l_yes_no;
        $l_rules['C__MODULE__SWAPCI__CONFIG_SWAP_SYSID']['p_arData'] = $l_yes_no;
        $l_rules['C__MODULE__SWAPCI__CONFIG_ARCHIVE_SWAPPED_OBJ']['p_arData'] = $l_yes_no;

        // And add values from the configuration.
        $l_rules['C__MODULE__SWAPCI__AVAILABLE_OBJECT_TYPES']['p_strValue'] = isys_format_json::encode($l_config);
        $l_rules['C__MODULE__SWAPCI__CONFIG_STORE_STATUS']['p_arData'] = $l_cmdb_status;
        $l_rules['C__MODULE__SWAPCI__CONFIG_CONSIDER_STATUS']['p_strSelectedID'] = $this->m_dao->get_config('consider_status');
        $l_rules['C__MODULE__SWAPCI__CONFIG_ARCHIVE_SWAPPED_OBJ']['p_strSelectedID'] = $this->m_dao->get_config('archive_swapped_obj');
        $l_rules['C__MODULE__SWAPCI__CONFIG_SWAP_SYSID']['p_strSelectedID'] = $this->m_dao->get_config('swap_sysid');

        // Load all object-types.
        $l_obj_types = isys_factory::get_instance('isys_cmdb_dao', $this->database)
            ->get_object_type();

        foreach ($l_obj_types as $l_obj_type) {
            $l_tpl_obj_types[$l_obj_type['LC_isys_obj_type__title']] = [
                'id'       => $l_obj_type['isys_obj_type__id'],
                'const'    => $l_obj_type['isys_obj_type__const'],
                'selected' => (in_array($l_obj_type['isys_obj_type__const'], $l_config)),
                'title'    => $l_obj_type['LC_isys_obj_type__title']
            ];
        }

        // For easy sorting by name, but still providing a "normal" array (instead of associative).
        ksort($l_tpl_obj_types);
        $l_tpl_obj_types = array_values($l_tpl_obj_types);

        $this->template->assign('ajax_url', isys_helper_link::create_url([C__GET__AJAX_CALL => 'swapci', C__GET__AJAX => 1]))
            ->assign('obj_types', $l_tpl_obj_types)
            ->assign('obj_types_json', isys_format_json::encode($l_tpl_obj_types))
            ->assign('categories', isys_format_json::encode(array_values($this->get_all_categories())))
            ->smarty_tom_add_rules('tom.content.bottom.content', $l_rules)
            ->include_template('contentbottomcontent', self::getPath() . 'templates/swap_config.tpl');
    }

    /**
     * Retrieves all categories in a certain format, we need for this module.
     *
     * @return  array
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    protected function get_all_categories()
    {
        $l_categories = [];

        // Retrieve all categories for loading them into the GUI.
        $l_dao = isys_cmdb_dao::instance($this->database);
        $l_catg_res = $l_dao->get_all_catg();
        $l_cats_res = $l_dao->get_all_cats();
        $l_catc_res = $l_dao->get_all_catg_custom();

        if (count($l_catg_res) > 0) {
            while ($l_row = $l_catg_res->get_row()) {
                $l_categories['g_' . _L($l_row['isysgui_catg__title'])] = [
                    'id'       => $l_row['isysgui_catg__id'],
                    'const'    => $l_row['isysgui_catg__const'],
                    'selected' => false,
                    'title'    => _L($l_row['isysgui_catg__title'])
                ];
            }
        }

        if (count($l_cats_res) > 0) {
            while ($l_row = $l_cats_res->get_row()) {
                $l_categories['s_' . _L($l_row['isysgui_cats__title'])] = [
                    'id'       => $l_row['isysgui_cats__id'],
                    'const'    => $l_row['isysgui_cats__const'],
                    'selected' => false,
                    'title'    => _L($l_row['isysgui_cats__title'])
                ];
            }
        }

        if (count($l_catc_res) > 0) {
            while ($l_row = $l_catc_res->get_row()) {
                $l_categories['s_' . $l_row['isysgui_catg_custom__title']] = [
                    'id'       => $l_row['isysgui_catg_custom__id'],
                    'const'    => $l_row['isysgui_catg_custom__const'],
                    'selected' => false,
                    'title'    => $l_row['isysgui_catg_custom__title']
                ];
            }
        }

        ksort($l_categories);

        return $l_categories;
    }

    /**
     * Callback function for adding the "swap" button to the navbar.
     *
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public static function add_swap_button()
    {
        $l_obj_type = isys_cmdb_dao::instance(isys_application::instance()->container->get('database'))->get_object_type($_GET[C__CMDB__GET__OBJECTTYPE]);
        $l_obj_types = isys_swapci_dao::instance(isys_application::instance()->container->get('database'))->get_config('available_obj_types') ?: [];

        if (!in_array($l_obj_type['isys_obj_type__const'], $l_obj_types, true)) {
            return;
        }

        // Preparing the buttons default options.
        $l_options = [
            'active'        => false,
            'icon'          => 'icons/silk/page_refresh.png',
            'icon_inactive' => 'icons/silk/page_refresh.png',
            'navmode'       => 'swapci'
        ];

        $l_url = isys_helper_link::create_url([C__GET__MODULE_ID => C__MODULE__SWAPCI, C__GET__SETTINGS_PAGE => 'swap_gui'], true);

        // Depending on the CMDB context, we need to set different options.
        switch ($_GET[C__CMDB__GET__VIEWMODE]) {
            case C__CMDB__VIEW__LIST_OBJECT:
                // Inside an object list.
                $l_options['active'] = isys_auth_swapci::instance()
                    ->is_allowed_to(isys_auth::VIEW, 'SWAP_IN_LIST');
                $l_options['tooltip'] = _L('LC__MODULE__SWAPCI__NAVBAR__SWAP_OBJECTS_TOOLTIP');
                $l_options['js_onclick'] = "if ($$('.mainTableCheckbox input:checked').length && $$('.mainTableCheckbox input:checked').length <= 2) {" .
                    "window.location.href='" . $l_url . "&" . C__CMDB__GET__OBJECT . "='+$$('.mainTableCheckbox input:checked').invoke('getValue').join();" . "}";
                break;

            case C__CMDB__VIEW__CATEGORY:
                // Inside an object.
                $l_options['active'] = isys_auth_swapci::instance()
                    ->is_allowed_to(isys_auth::VIEW, 'SWAP_IN_OBJ');
                $l_options['tooltip'] = _L('LC__MODULE__SWAPCI__NAVBAR__SWAP_OBJECT_TOOLTIP');
                $l_options['js_onclick'] = "window.location.href='" . isys_helper_link::add_params_to_url($l_url, [C__CMDB__GET__OBJECT => $_GET[C__CMDB__GET__OBJECT]]) .
                    "';";
                break;

            default:
                $l_options['active'] = false;
                $l_options['visible'] = false;
        }

        isys_component_template_navbar::getInstance()
            ->append_button(_L('LC__MODULE__SWAPCI__NAVBAR__SWAP'), 'swapci', $l_options);
    }

    /**
     * Get related auth class for module
     *
     * @author Selcuk Kekec <skekec@i-doit.com>
     * @return isys_auth_swapci
     */
    public static function get_auth()
    {
        return isys_auth_swapci::instance();
    }

    /**
     * Method that is called after clicking "activate" in admin center. Override this methode for special handling.
     *
     * @param  string $p_identifier
     *
     * @return void
     * @throws isys_exception_dao
     */
    public function activate($p_identifier)
    {
        $l_dao = isys_cmdb_dao::instance(isys_application::instance()->container->get('database'));

        $l_dao->update('UPDATE isys_cmdb_status
            SET isys_cmdb_status__status = ' . $l_dao->convert_sql_int(C__RECORD_STATUS__NORMAL) . '
            WHERE isys_cmdb_status__const = "C__CMDB_STATUS__SWAPPED";');
    }

    /**
     * Method that is called after clicking "deactivate" in admin center. Override this methode for special handling.
     *
     * @param  string $p_identifier
     *
     * @return void
     * @throws isys_exception_dao
     * @author Leonard Fischer <lfischer@i-doit.com>
     */
    public function deactivate($p_identifier)
    {
        $l_dao = isys_cmdb_dao::instance(isys_application::instance()->container->get('database'));

        $l_dao->update('UPDATE isys_cmdb_status
            SET isys_cmdb_status__status = ' . $l_dao->convert_sql_int(C__RECORD_STATUS__DELETED) . '
            WHERE isys_cmdb_status__const = "C__CMDB_STATUS__SWAPPED";');
    }

    /**
     * Method for uninstalling the module.
     *
     * @author Leonard Fischer <lfischer@i-doit.com>
     */
    public function uninstall()
    {
        $l_dao = isys_cmdb_dao::instance(isys_application::instance()->container->get('database'));

        $l_dao->update('UPDATE isys_cmdb_status
            SET isys_cmdb_status__status = ' . $l_dao->convert_sql_int(C__RECORD_STATUS__DELETED) . '
            WHERE isys_cmdb_status__const = "C__CMDB_STATUS__SWAPPED";');
    }
}
