<?php

use idoit\Module\Cmdb\Search\Index\Signals;

/**
 * i-doit
 *
 * Core class for the "swap" process of the "Swap CI" module.
 *
 * @package     i-doit
 * @subpackage  Modules
 * @author      Leonard Fischer <lfischer@i-doit.com>
 * @version     1.2.2
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 * @since       i-doit 1.3.0
 */
class isys_swapci_swap
{
    /**
     * List of blacklisted categories.
     * @var  array
     */
    const CATEGORY_BLACKLIST = [
        'C__CATG__GLOBAL',
        'C__CATG__OBJECT_VITALITY',
        'C__CATG__LOGBOOK',
        'C__CATG__NETWORK_PORT_OVERVIEW',
        'C__CATG__BACKUP__ASSIGNED_OBJECTS',
        'C__CATG__GUEST_SYSTEMS',
        'C__CATG__NAGIOS_REFS_SERVICES_BACKWARDS',
        'C__CATG__NAGIOS_REFS_SERVICES',
        'C__CATG__NAGIOS_APPLICATION_FOLDER',
        'C__CATG__NAGIOS_APPLICATION_REFS_NAGIOS_SERVICE'
    ];

    /**
     * List of tables and fields, which can not be handled generically.
     * @var  array
     */
    const MANUAL_PROPERTIES = [
        'isys_auth'                   => ['isys_auth__isys_obj__id'],
        'isys_cable_connection'       => ['isys_cable_connection__isys_obj__id'],
        'isys_contact_2_isys_obj'     => ['isys_contact_2_isys_obj__isys_obj__id'],
        'isys_container'              => ['isys_container__isys_obj__id__child', 'isys_container__isys_obj__id', 'isys_container__isys_obj__id__parent'],
        'isys_netp_ifacel_2_isys_obj' => ['isys_obj__id'],
        'isys_obj_type'               => ['isys_obj_type__default_template'],
        'isys_workflow'               => ['isys_workflow__isys_obj__id'],
        'isys_workflow_2_isys_obj'    => ['isys_workflow_2_isys_obj__isys_obj__id']
    ];

    /**
     * List of additional category tables and fields, which can not be handled generically.
     * @var  array
     */
    const ADDITIONAL_PROPERTIES = [
        'isys_catg_assigned_cards_list'      => ['isys_catg_assigned_cards_list__isys_obj__id__card'],
        'isys_catg_cluster_list_2_isys_obj'  => ['isys_obj__id'],
        'isys_catg_log_port_list_2_isys_obj' => ['isys_obj__id'],
        'isys_catg_logical_unit_list'        => ['isys_catg_logical_unit_list__isys_obj__id__parent'],
        'isys_catg_nagios_service_dep_list'  => ['isys_catg_nagios_service_dep_list__service_dep_connection'],
        'isys_catg_virtual_machine_list'     => ['isys_catg_virtual_machine_list__primary']
    ];

    /**
     * Member variable which holds some info of the "old" (to be replaced-) object.
     * @var  array
     */
    protected $m_old_obj = [];

    /**
     * Member variable which holds some info of the "new" (stored-) object.
     * @var  array
     */
    protected $m_new_obj = [];

    /**
     * Member variable, which holds the module DAO.
     * @var  isys_component_database
     */
    protected $m_db = null;

    /**
     * Member variable, which holds the logbook component.
     * @var  isys_component_dao_logbook
     */
    protected $m_logbook = null;

    /**
     * Member variable, which holds the module DAO.
     * @var  isys_swapci_dao
     */
    protected $m_dao = null;

    /**
     * Member variable, which holds the CMDB DAO.
     * @var  isys_cmdb_dao
     */
    protected $m_cmdb_dao = null;

    /**
     * Logger instance.
     * @var  isys_log
     */
    protected $m_log = null;

    /**
     * Options array, which holds necessary information.
     * @var  array
     */
    protected $m_options = [];

    /**
     * Results array for returning to the caller (sort of "human readable" info and debug stuff).
     * @var  array
     */
    protected $m_results = [];


    /**
     * Initialize the swap process.
     *
     * @param   array $p_options
     * @return  isys_swapci_swap
     * @throws  isys_exception_general
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function init($p_options = [])
    {
        $this->m_db = isys_application::instance()->container->get('database');
        $this->m_cmdb_dao = isys_cmdb_dao::instance($this->m_db);

        // Check, if two objects were provided.
        if (empty($p_options['old_obj']) || empty($p_options['new_obj'])) {
            throw new isys_exception_general(_L('LC__MODULE__SWAPCI_EXCEPTION__MISSING_OBJ'));
        }

        // If no object-type is given or we only got an ID, we load the constant.
        if (empty($p_options['obj_type']) || is_numeric($p_options['obj_type'])) {
            $l_row = $this->m_cmdb_dao->get_object($p_options['new_obj'], true)->get_row();
            $p_options['obj_type'] = $l_row['isys_obj_type__const'];
        }

        if (!defined($p_options['obj_type'])) {
            throw new isys_exception_general(_L('LC__MODULE__SWAPCI_EXCEPTION__UNKNOWN_OBJ_TYPE', $p_options['obj_type']));
        }

        $this->m_old_obj = [
            'id'    => $p_options['old_obj'],
            'title' => $this->m_cmdb_dao->get_obj_name_by_id_as_string($p_options['old_obj']),
            'sysid' => $this->m_cmdb_dao->get_sysid_by_obj_id($p_options['old_obj'])
        ];
        $this->m_new_obj = [
            'id'    => $p_options['new_obj'],
            'title' => $this->m_cmdb_dao->get_obj_name_by_id_as_string($p_options['new_obj']),
            'sysid' => $this->m_cmdb_dao->get_sysid_by_obj_id($p_options['new_obj'])
        ];

        $this->m_logbook = isys_factory::get_instance('isys_component_dao_logbook', $this->m_db);
        $this->m_dao = isys_factory::get_instance('isys_swapci_dao', $this->m_db);
        $this->m_log = isys_log::get_instance('swap')
            ->set_log_file(isys_glob_get_temp_dir() . 'swap_log__' . date('Y-m-d_H-i-s') . '.log')
            ->set_log_level(isys_log::C__ALL);

        // Retrieve the configuration.
        $l_config = $this->m_dao->get_config();

        // Build the "default" settings from our config...
        $l_default = [
            'categories'          => $l_config['objtype_' . $p_options['objtype']],
            'consider_status'     => $l_config['consider_status'],
            'archive_swapped_obj' => $l_config['archive_old_obj'],
            'swap_sysid'          => $l_config['swap_sysid']
        ];

        unset($p_options['old_obj'], $p_options['new_obj']);

        // And merge them together.
        $this->m_options = array_merge($l_default, $p_options);
        $this->m_options['obj_type_title'] = $this->m_cmdb_dao->get_objtype_name_by_id_as_string($this->m_options['obj_type']);

        $this->m_log->info('Finished initialization!');

        return $this;
    }


    /**
     * The main "swap" method. Please note, that "init()" needs to be called, before this method.
     *
     * @return  isys_swapci_swap
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function swap()
    {
        try {
            $this
                ->prepare_category_data()
                ->purge_old_category_data()
                ->update_category_object_ids()
                ->update_relation_object_ids()
                ->update_remaining_object_ids()
                ->update_sysid()
                ->update_status()
                ->archive_swapped_object()
                ->empty_object_related_cache();

            $this->m_log->info('Writing a logbook note that the objects have been changed!');

            // Inserting "object swap notice" into the logbook.
            $this->m_logbook->set_entry(
                'C__LOGBOOK_EVENT__OBJECT_CHANGED',
                '',
                null,
                C__LOGBOOK__ALERT_LEVEL__0,
                $this->m_old_obj['id'],
                $this->m_old_obj['title'],
                $this->m_options['obj_type_title'],
                'LC__CMDB__CATG__GLOBAL',
                C__LOGBOOK_SOURCE__USER,
                null,
                _L('LC__MODULE__SWAPCI_LOGBOOK__OBJECT_SWAPPED_WITH_OBJECT', [$this->m_old_obj['title'], $this->m_new_obj['title']])
            );

            $this->m_logbook->set_entry(
                'C__LOGBOOK_EVENT__OBJECT_CHANGED',
                '',
                null,
                C__LOGBOOK__ALERT_LEVEL__0,
                $this->m_new_obj['id'],
                $this->m_new_obj['title'],
                $this->m_options['obj_type_title'],
                'LC__CMDB__CATG__GLOBAL',
                C__LOGBOOK_SOURCE__USER,
                null,
                _L('LC__MODULE__SWAPCI_LOGBOOK__OBJECT_SWAPPED_WITH_OBJECT', [$this->m_old_obj['title'], $this->m_new_obj['title']])
            );

            $categories = [];

            if (isset($this->m_options['categories']) && is_array($this->m_options['categories'])) {
                foreach ($this->m_options['categories'] as $l_category) {
                    $categories[] = $l_category['const'];
                }
            }

            Signals::instance()->onPostImport(time(), $categories);

            $this->m_log->info('Object Swap was successful!');
        } catch (Exception $e) {
            $this->m_log->fatal($e->getMessage());
        }

        $this->m_results['old'] = $this->m_old_obj;
        $this->m_results['new'] = $this->m_new_obj;
        $this->m_results['log'] = $this->m_log->get_log(true);

        // Finally we save the whole "swapping" process to our "archive".
        $this->save_to_archive();

        return $this;
    }


    /**
     * Variable for retrieving the swap results.
     *
     * @return  array
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    public function get_results()
    {
        return $this->m_results;
    }


    /**
     * This method will retrieve (database) table- and class names for the selected categories.
     *
     * @return  isys_swapci_swap
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    private function prepare_category_data()
    {
        $this->m_log->info('Collecting categories to swap!');

        $l_categories = [];

        foreach ($this->m_options['categories'] as $l_category) {
            $l_categories[] = $this->m_cmdb_dao->convert_sql_text($l_category);
        }

        // We reset the array here and fill it up with more information a few lines further.
        $this->m_options['categories'] = [];

        // Load all selected global categories...
        $l_result = $this->m_cmdb_dao->retrieve('SELECT isysgui_catg__title, isysgui_catg__const, isysgui_catg__source_table, isysgui_catg__class_name FROM isysgui_catg
			WHERE isysgui_catg__const IN (' . implode(', ', $l_categories) . ');');

        if (count($l_result) > 0) {
            while ($l_row = $l_result->get_row()) {
                $this->m_log->debug('> Global category "' . $l_row['isysgui_catg__const'] . '".');
                $this->m_options['categories'][] = [
                    'title' => $l_row['isysgui_catg__title'],
                    'const' => $l_row['isysgui_catg__const'],
                    'table' => $l_row['isysgui_catg__source_table'] . '_list',
                    'class' => $l_row['isysgui_catg__class_name']
                ];
            }
        }

        // Load all selected specific categories...
        $l_result = $this->m_cmdb_dao->retrieve('SELECT isysgui_cats__title, isysgui_cats__const, isysgui_cats__source_table, isysgui_cats__class_name FROM isysgui_cats
			WHERE isysgui_cats__const IN (' . implode(', ', $l_categories) . ');');

        if (count($l_result) > 0) {
            while ($l_row = $l_result->get_row()) {
                $this->m_log->debug('> Specific category "' . $l_row['isysgui_cats__const'] . '".');
                $this->m_options['categories'][] = [
                    'title' => $l_row['isysgui_cats__title'],
                    'const' => $l_row['isysgui_cats__const'],
                    'table' => $l_row['isysgui_cats__source_table'],
                    'class' => $l_row['isysgui_cats__class_name']
                ];
            }
        }

        // Load all selected custom categories...
        $l_result = $this->m_cmdb_dao->retrieve('SELECT isysgui_catg_custom__title, isysgui_catg_custom__const, isysgui_catg_custom__source_table, isysgui_catg_custom__class_name FROM isysgui_catg_custom
			WHERE isysgui_catg_custom__const IN (' . implode(', ', $l_categories) . ');');

        if (count($l_result) > 0) {
            while ($l_row = $l_result->get_row()) {
                $this->m_log->debug('> Global category "' . $l_row['isysgui_catg_custom__const'] . '".');
                $this->m_options['categories'][] = [
                    'title' => $l_row['isysgui_catg_custom__title'],
                    'const' => $l_row['isysgui_catg_custom__const'],
                    'table' => $l_row['isysgui_catg_custom__source_table'],
                    'class' => $l_row['isysgui_catg_custom__class_name']
                ];
            }
        }

        // This is a special case, because the "logical location" relation uses "C__CATG__LOGICAL_UNIT" instead of "C__CATG__ASSIGNED_WORKSTATION" category.
        if (in_array($this->m_cmdb_dao->convert_sql_text('C__CATG__ASSIGNED_WORKSTATION'), $l_categories)) {
            $l_categories[] = $this->m_cmdb_dao->convert_sql_text('C__CATG__LOGICAL_UNIT');
        }

        // Now something special: We load all relation types, which are related to the selected categories.
        $l_swappable_relation_types = [];
        $l_res = $this->m_cmdb_dao->retrieve('SELECT isys_relation_type__id FROM isys_relation_type WHERE isys_relation_type__category IS NULL
			OR isys_relation_type__category IN (' . implode(', ', $l_categories) . ');');

        if (count($l_res) > 0) {
            while ($l_row = $l_res->get_row()) {
                $l_swappable_relation_types[] = (int)$l_row['isys_relation_type__id'];
            }
        }

        $this->m_options['swappable_relation_types'] = $l_swappable_relation_types;

        return $this;
    }


    /**
     * This method will remove all data of every selected category (inside the "new" object).
     *
     * @return  isys_swapci_swap
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    private function purge_old_category_data()
    {
        $this->m_log->info('Purging data inside stored object, which would be overwritten by the "swapped" data.');

        foreach ($this->m_options['categories'] as $l_category) {
            $this->m_log->info('> Category "' . $l_category['const'] . '".');

            try {
                $l_sql = 'DELETE FROM ' . $l_category['table'] . ' WHERE ' . $l_category['table'] . '__isys_obj__id = ' . $this->m_cmdb_dao->convert_sql_id($this->m_new_obj['id']) . ';';

                $this->m_cmdb_dao->update($l_sql);
                $l_affected_rows = $this->m_cmdb_dao->affected_after_update();

                if ($l_affected_rows > 0) {
                    $this->m_log->debug('> SQL: "' . str_replace(["\n", "\t"], [' ', ''], $l_sql) . '", affected ' . $l_affected_rows . ' rows.');

                    // Noting the "purge" process inside the new objects logbook.
                    $this->m_logbook->set_entry(
                        'C__LOGBOOK_EVENT__CATEGORY_PURGED',
                        $l_sql,
                        null,
                        C__LOGBOOK__ALERT_LEVEL__3,
                        $this->m_new_obj['id'],
                        $this->m_new_obj['title'],
                        $this->m_options['obj_type_title'],
                        $l_category['title'],
                        C__LOGBOOK_SOURCE__USER,
                        null,
                        _L('LC__MODULE__SWAPCI_LOGBOOK__PURGED_BY_SWAPCI', $l_affected_rows),
                        null);
                } else {
                    $this->m_log->debug('> No data to purge.');
                }
            } catch (isys_exception_database $e) {
                $this->m_log->warning('> ' . $e->getMessage());
            }
        }

        $this->m_log->info('Now purging related relations in stored object.');

        $l_sql = 'SELECT isys_obj__title, isys_catg_relation_list__isys_obj__id FROM isys_catg_relation_list
			LEFT JOIN isys_obj ON isys_obj__id = isys_catg_relation_list__isys_obj__id
			WHERE (isys_catg_relation_list__isys_obj__id__master = ' . $this->m_cmdb_dao->convert_sql_id($this->m_new_obj['id']) . '
			OR isys_catg_relation_list__isys_obj__id__slave = ' . $this->m_cmdb_dao->convert_sql_id($this->m_new_obj['id']) . ')
			AND isys_catg_relation_list__isys_relation_type__id IN (' . implode(', ', $this->m_options['swappable_relation_types']) . ');';

        $l_res = $this->m_cmdb_dao->retrieve($l_sql);

        if (count($l_res) > 0) {
            while ($l_row = $l_res->get_row()) {
                $this->m_log->info('> Deleting relation object "' . $l_row['isys_obj__title'] . '".');
                $this->m_cmdb_dao->delete_object($l_row['isys_catg_relation_list__isys_obj__id']);
            }
        }

        $this->m_cmdb_dao->apply_update();

        return $this;
    }


    /**
     * This method will try to set all "old" object id's to the new ones, for every selected category.
     *
     * @return  isys_swapci_swap
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    private function update_category_object_ids()
    {
        $this->m_log->info('Swapping category data to the stored object.');

        foreach ($this->m_options['categories'] as $l_category) {
            $this->m_log->info('> Category "' . $l_category['const'] . '".');

            try {
                // @see SWAPCI-23 Check if the tables and fields existm before trying any action.
                if (!$this->m_cmdb_dao->table_exists($l_category['table'])) {
                    continue;
                }

                $l_sql = 'UPDATE ' . $l_category['table'] . '
					SET ' . $l_category['table'] . '__isys_obj__id = ' . $this->m_cmdb_dao->convert_sql_id($this->m_new_obj['id']) . '
					WHERE ' . $l_category['table'] . '__isys_obj__id = ' . $this->m_cmdb_dao->convert_sql_id($this->m_old_obj['id']) . ' ';

                // Extra sausage for "connector" category.
                if ($l_category['const'] == 'C__CATG__CONNECTOR') {
                    $l_sql .= ' AND isys_catg_connector_list__assigned_category = "C__CATG__CONNECTOR"';
                }

                $this->m_cmdb_dao->update($l_sql . ';');
                $l_affected_rows = $this->m_cmdb_dao->affected_after_update();

                if ($l_affected_rows > 0) {
                    $this->m_log->debug('> SQL: "' . str_replace(["\n", "\t"], [' ', ''], $l_sql) . '", affected ' . $l_affected_rows . ' rows.');

                    // Noting the "swap" process inside the old and new objects logbook.
                    $this->m_logbook->set_entry(
                        'C__LOGBOOK_EVENT__CATEGORY_CHANGED',
                        $l_sql,
                        null,
                        C__LOGBOOK__ALERT_LEVEL__0,
                        $this->m_new_obj['id'],
                        $this->m_new_obj['title'],
                        $this->m_options['obj_type_title'],
                        $l_category['title'],
                        C__LOGBOOK_SOURCE__USER,
                        null,
                        _L('LC__MODULE__SWAPCI_LOGBOOK__SWAPPED_BY_SWAPCI', [$l_affected_rows, $this->m_old_obj['title'], $this->m_old_obj['id']]),
                        null);

                    $this->m_logbook->set_entry(
                        'C__LOGBOOK_EVENT__CATEGORY_CHANGED',
                        $l_sql,
                        null,
                        C__LOGBOOK__ALERT_LEVEL__0,
                        $this->m_old_obj['id'],
                        $this->m_old_obj['title'],
                        $this->m_options['obj_type_title'],
                        $l_category['title'],
                        C__LOGBOOK_SOURCE__USER,
                        null,
                        _L('LC__MODULE__SWAPCI_LOGBOOK__SWAPPED_AWAY_BY_SWAPCI', [$l_affected_rows, $this->m_new_obj['title'], $this->m_new_obj['id']]),
                        null);
                } else {
                    $this->m_log->debug('> No data to swap.');
                }

                // Here we swap "remaining" category data, which may refer to the swap-objects.
                if (array_key_exists($l_category['table'], self::ADDITIONAL_PROPERTIES)) {
                    $this->update_additional_category_data($l_category['table']);
                }
            } catch (isys_exception_database $e) {
                $this->m_log->warning('> ' . $e->getMessage());
            }

            // This extra part is necessary for certain categories:
            if (in_array($l_category['const'], ['C__CATG__UNIVERSAL_INTERFACE', 'C__CMDB__SUBCAT__NETWORK_PORT', 'C__CATG__NETWORK_PORT', 'C__CATG__CONTROLLER_FC_PORT', 'C__CATG__POWER_CONSUMER'])) {
                $this->m_log->info('> Swapping related connectors.');

                try {
                    $l_sql = 'UPDATE isys_catg_connector_list
						SET isys_catg_connector_list__isys_obj__id = ' . $this->m_cmdb_dao->convert_sql_id($this->m_new_obj['id']) . '
						WHERE isys_catg_connector_list__isys_obj__id = ' . $this->m_cmdb_dao->convert_sql_id($this->m_old_obj['id']) . '
						AND isys_catg_connector_list__assigned_category = ' . $this->m_cmdb_dao->convert_sql_text($l_category['const']) . ';';

                    $this->m_cmdb_dao->update($l_sql);
                    $l_affected_rows = $this->m_cmdb_dao->affected_after_update();

                    if ($l_affected_rows > 0) {
                        $this->m_log->debug('> SQL: "' . str_replace(["\n", "\t"], [' ', ''], $l_sql) . '", affected ' . $l_affected_rows . ' rows.');
                    } else {
                        $this->m_log->debug('> No data to swap.');
                    }
                } catch (isys_exception_database $e) {
                    $this->m_log->warning('> ' . $e->getMessage());
                }
            }
        }

        $this->m_cmdb_dao->apply_update();

        return $this;
    }


    /**
     * This method will try to set all relations from the old object to the new one.
     *
     * @return  isys_swapci_swap
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    private function update_relation_object_ids()
    {
        // Now we select all relations, where we need to update the object ID.
        $this->m_log->info('Swapping all category related relations to the stored object.');

        $l_sql = 'SELECT isys_catg_relation_list__id, isys_obj__title, isys_catg_relation_list__isys_obj__id, isys_catg_relation_list__isys_obj__id__master, isys_catg_relation_list__isys_obj__id__slave
			FROM isys_catg_relation_list
			LEFT JOIN isys_obj ON isys_obj__id = isys_catg_relation_list__isys_obj__id
			WHERE (isys_catg_relation_list__isys_obj__id__master = ' . $this->m_cmdb_dao->convert_sql_id($this->m_old_obj['id']) . '
			OR isys_catg_relation_list__isys_obj__id__slave = ' . $this->m_cmdb_dao->convert_sql_id($this->m_old_obj['id']) . ')
			AND isys_catg_relation_list__isys_relation_type__id IN (' . implode(', ', $this->m_options['swappable_relation_types']) . ');';

        $l_res = $this->m_cmdb_dao->retrieve($l_sql);

        if (count($l_res) > 0) {
            $l_relations = [];

            while ($l_row = $l_res->get_row()) {
                $this->m_log->info('> Swapping relation object "' . $l_row['isys_obj__title'] . '".');
                $l_message = _L('LC__MODULE__SWAPCI_RELATION__SWAPPED_BY_SWAPCI', [$this->m_old_obj['title'], $this->m_old_obj['id'], $this->m_new_obj['title'], $this->m_new_obj['id']]);

                $l_query = [
                    'isys_catg_relation_list__description = ' . $this->m_cmdb_dao->convert_sql_text($l_message)
                ];

                if ($l_row['isys_catg_relation_list__isys_obj__id__master'] == $this->m_old_obj['id']) {
                    $l_query[] = 'isys_catg_relation_list__isys_obj__id__master = ' . $this->m_cmdb_dao->convert_sql_id($this->m_new_obj['id']);
                }

                if ($l_row['isys_catg_relation_list__isys_obj__id__slave'] == $this->m_old_obj['id']) {
                    $l_query[] = 'isys_catg_relation_list__isys_obj__id__slave = ' . $this->m_cmdb_dao->convert_sql_id($this->m_new_obj['id']);
                }

                $l_relations[] = (int)$l_row['isys_catg_relation_list__id'];
                $l_sql = 'UPDATE isys_catg_relation_list SET ' . implode(', ',
                        $l_query) . ' WHERE isys_catg_relation_list__id = ' . $this->m_cmdb_dao->convert_sql_id($l_row['isys_catg_relation_list__id']) . ';';

                $this->m_cmdb_dao->update($l_sql);
            }

            // Now we renew the relation titles.
            $this->m_log->info('> Updating relation titles.');
            $this->renew_relation_titles($l_relations);
        } else {
            $this->m_log->debug('> No relations to swap.');
        }

        $this->m_cmdb_dao->apply_update();

        return $this;
    }


    /**
     * This method will try to set all "old" object id's to the new ones, for every remaining table which could not be handled generically.
     *
     * @return  isys_swapci_swap
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    private function update_remaining_object_ids()
    {
        $this->m_log->info('Updating remaining object IDs in several other categories.');

        foreach (self::MANUAL_PROPERTIES as $l_table => $l_fields) {
            foreach ($l_fields as $l_field) {
                // @see SWAPCI-23 Check if the tables and fields existm before trying any action.
                if (!$this->m_cmdb_dao->table_exists($l_table) || !$this->m_cmdb_dao->column_exists_in_table($l_table, $l_field)) {
                    continue;
                }

                $l_sql = 'UPDATE ' . $l_table .
                    ' SET ' . $l_field . ' = ' . $this->m_cmdb_dao->convert_sql_id($this->m_new_obj['id']) .
                    ' WHERE ' . $l_field . ' = ' . $this->m_cmdb_dao->convert_sql_id($this->m_old_obj['id']) . ';';

                try {
                    $this->m_cmdb_dao->update($l_sql);
                    $this->m_log->debug('> Updating table "' . $l_table . '", field "' . $l_field . '" - ' . $this->m_cmdb_dao->affected_after_update() . ' rows affected.');
                } catch (isys_exception_database $e) {
                    $this->m_log->warning($e->getMessage());
                }
            }
        }

        $this->m_log->info('Also updating Authorization data, which relates to the swapped object.');
        $l_sql = 'UPDATE isys_auth
			SET isys_auth__path = "OBJ_ID/' . $this->m_cmdb_dao->convert_sql_id($this->m_new_obj['id']) . '"
			WHERE isys_auth__path LIKE "OBJ_ID/' . $this->m_cmdb_dao->convert_sql_id($this->m_old_obj['id']) . '"
			AND isys_auth__isys_module__id = ' . $this->m_cmdb_dao->convert_sql_id(C__MODULE__CMDB) . ';';

        $this->m_cmdb_dao->update($l_sql);
        $this->m_log->debug('> Updated "OBJ_ID/' . $this->m_old_obj['id'] . '" paths... ' . $this->m_cmdb_dao->affected_after_update() . ' rows affected.');

        $l_sql = 'UPDATE isys_auth
			SET isys_auth__path = "LOCATION/' . $this->m_cmdb_dao->convert_sql_id($this->m_new_obj['id']) . '"
			WHERE isys_auth__path LIKE "LOCATION/' . $this->m_cmdb_dao->convert_sql_id($this->m_old_obj['id']) . '"
			AND isys_auth__isys_module__id = ' . $this->m_cmdb_dao->convert_sql_id(C__MODULE__CMDB) . ';';

        $this->m_cmdb_dao->update($l_sql);
        $this->m_log->debug('> Updated "LOCATION/' . $this->m_old_obj['id'] . '" paths... ' . $this->m_cmdb_dao->affected_after_update() . ' rows affected.');

        $this->m_cmdb_dao->apply_update();

        return $this;
    }


    /**
     * This method will try to set all "additional" category data which could not be handled generically.
     *
     * @param   string $p_table
     * @return  isys_swapci_swap
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    private function update_additional_category_data($p_table)
    {
        foreach (self::ADDITIONAL_PROPERTIES[$p_table] as $l_field) {
            // @see SWAPCI-23 Check if the tables and fields existm before trying any action.
            if (!$this->m_cmdb_dao->table_exists($p_table) || !$this->m_cmdb_dao->column_exists_in_table($p_table, $l_field)) {
                continue;
            }

            $l_sql = 'UPDATE ' . $p_table .
                ' SET ' . $l_field . ' = ' . $this->m_cmdb_dao->convert_sql_id($this->m_new_obj['id']) .
                ' WHERE ' . $l_field . ' = ' . $this->m_cmdb_dao->convert_sql_id($this->m_old_obj['id']) . ';';

            try {
                $this->m_cmdb_dao->update($l_sql);
                $this->m_log->debug('> Updating additional field "' . $l_field . '" in table "' . $p_table . '" - ' . $this->m_cmdb_dao->affected_after_update() . ' rows affected.');
            } catch (isys_exception_database $e) {
                $this->m_log->warning($e->getMessage());
            }
        }

        $this->m_cmdb_dao->apply_update();

        return $this;
    }


    /**
     * This method will update the SYS-ID from the old object to the new one, if configured.
     *
     * @return  isys_swapci_swap
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    private function update_sysid()
    {
        if ($this->m_options['swap_sysid']) {
            $this->m_log->info('Swapping the SYS-ID ("' . $this->m_old_obj['sysid'] . '") and creating a new one for the swapped object.');
            $l_catg_global = $this->m_cmdb_dao->get_catg_by_const('C__CATG__GLOBAL')->get_row();

            $l_sql = 'UPDATE isys_obj
				SET isys_obj__sysid = ' . $this->m_cmdb_dao->convert_sql_text($this->m_old_obj['sysid']) . '
				WHERE isys_obj__id = ' . $this->m_cmdb_dao->convert_sql_id($this->m_new_obj['id']) . ';';

            $this->m_cmdb_dao->update($l_sql);

            $this->m_logbook->set_entry(
                'C__LOGBOOK_EVENT__CATEGORY_CHANGED',
                $l_sql,
                null,
                C__LOGBOOK__ALERT_LEVEL__0,
                $this->m_new_obj['id'],
                $this->m_new_obj['title'],
                $this->m_options['obj_type_title'],
                $l_catg_global['isysgui_catg__title'],
                C__LOGBOOK_SOURCE__USER,
                null,
                _L('LC__MODULE__SWAPCI_LOGBOOK__SWAPPED_SYSID_BY_SWAPCI', [$this->m_old_obj['title'], $this->m_old_obj['id']])
            );

            // And now changing the SYS-ID to the other object.
            $l_new_sysid = $this->m_cmdb_dao->generate_sysid($this->m_options['obj_type'], $this->m_old_obj['id']);

            $l_sql = 'UPDATE isys_obj
				SET isys_obj__sysid = ' . $this->m_cmdb_dao->convert_sql_text($l_new_sysid) . '
				WHERE isys_obj__id = ' . $this->m_cmdb_dao->convert_sql_id($this->m_old_obj['id']) . ';';

            $this->m_cmdb_dao->update($l_sql);

            $this->m_log->debug('Created and saved new SYS-ID ("' . $l_new_sysid . '") for old object.');

            $this->m_logbook->set_entry(
                'C__LOGBOOK_EVENT__CATEGORY_CHANGED',
                $l_sql,
                null,
                C__LOGBOOK__ALERT_LEVEL__0,
                $this->m_old_obj['id'],
                $this->m_old_obj['title'],
                $this->m_options['obj_type_title'],
                $l_catg_global['isysgui_catg__title'],
                C__LOGBOOK_SOURCE__USER,
                null,
                _L('LC__MODULE__SWAPCI_LOGBOOK__NEW_SYSID_BY_SWAPCI')
            );

            // Also switching the SYS-IDs in our member-variables.
            $this->m_new_obj['sysid'] = $this->m_old_obj['sysid'];
            $this->m_old_obj['sysid'] = $l_new_sysid;

            $this->m_cmdb_dao->apply_update();
        }

        return $this;
    }


    /**
     * This method will update the CMDB status of the old object to "swapped", if configured.
     *
     * @return  isys_swapci_swap
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    private function update_status()
    {
        if ($this->m_options['consider_status'] && defined('C__CMDB_STATUS__SWAPPED')) {
            $this->m_log->info('Setting CMDB status of the swapped object to "' . _L('LC__CMDB_STATUS__SWAPPED') . '".');

            $this->m_cmdb_dao->set_object_cmdb_status($this->m_old_obj['id'], C__CMDB_STATUS__SWAPPED);

            // Inserting the CMDB status change into the logbook.
            $this->m_logbook->set_entry(
                'C__LOGBOOK_EVENT__CATEGORY_CHANGED',
                $this->m_cmdb_dao->get_last_query(),
                null,
                C__LOGBOOK__ALERT_LEVEL__0,
                $this->m_old_obj['id'],
                $this->m_old_obj['title'],
                $this->m_options['obj_type_title'],
                'LC__CMDB__CATG__GLOBAL',
                C__LOGBOOK_SOURCE__USER,
                null,
                _L('LC__MODULE__SWAPCI_LOGBOOK__CMDB_STATUS_SWAPPED_BY_SWAPCI', [_L('LC__CMDB_STATUS__SWAPPED')]),
                null);

            $this->m_log->info('Setting CMDB status of the new object to "' . _L('LC__CMDB_STATUS__IN_OPERATION') . '".');

            $this->m_cmdb_dao->set_object_cmdb_status($this->m_new_obj['id'], C__CMDB_STATUS__IN_OPERATION);

            // Inserting the CMDB status change into the logbook.
            $this->m_logbook->set_entry(
                'C__LOGBOOK_EVENT__CATEGORY_CHANGED',
                $this->m_cmdb_dao->get_last_query(),
                null,
                C__LOGBOOK__ALERT_LEVEL__0,
                $this->m_new_obj['id'],
                $this->m_new_obj['title'],
                $this->m_options['obj_type_title'],
                'LC__CMDB__CATG__GLOBAL',
                C__LOGBOOK_SOURCE__USER,
                null,
                _L('LC__MODULE__SWAPCI_LOGBOOK__CMDB_STATUS_SWAPPED_BY_SWAPCI', [_L('LC__CMDB_STATUS__IN_OPERATION')]),
                null);
        }

        // @since  v1.2.2
        $this->m_cmdb_dao->object_changed($this->m_old_obj['id'], null);
        $this->m_cmdb_dao->object_changed($this->m_new_obj['id'], null);

        return $this;
    }


    /**
     * This method will archive the swapped object, if configured.
     *
     * @return  isys_swapci_swap
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    private function archive_swapped_object()
    {
        if ($this->m_options['archive_swapped_obj']) {
            $this->m_log->info('Archiving old object.');

            // We can only "archive" it, when it is "normal".
            if ($this->m_cmdb_dao->get_object_status_by_id($this->m_old_obj['id']) == C__RECORD_STATUS__NORMAL) {
                $this->m_cmdb_dao->rank_record($this->m_old_obj['id'], C__CMDB__RANK__DIRECTION_DELETE, 'isys_obj');
                $this->m_log->debug('Success!');
            } else {
                $this->m_log->debug('Did not archive the old object - Maybe it already is archived?');
            }
        }

        return $this;
    }


    /**
     * This method will try to remove all "object related" cache-data.
     *
     * @return  isys_swapci_swap
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    private function empty_object_related_cache()
    {
        $l_new_id = $this->m_cmdb_dao->convert_sql_id($this->m_new_obj['id']);
        $l_old_id = $this->m_cmdb_dao->convert_sql_id($this->m_old_obj['id']);

        $this->m_log->info('Removing all object related cache-data');

        // Removing cache data for the "quick info" popups.
        $this->m_cmdb_dao->update('DELETE FROM isys_cache_qinfo WHERE isys_cache_qinfo__isys_obj__id IN (' . $l_new_id . ', ' . $l_old_id . ');');
        $this->m_log->debug('> "isys_cache_qinfo" ' . $this->m_cmdb_dao->affected_after_update() . ' rows affected.');

        // Removing "lock" data.
        $this->m_cmdb_dao->update('DELETE FROM isys_lock WHERE isys_lock__isys_obj__id IN (' . $l_new_id . ', ' . $l_old_id . ');');
        $this->m_log->debug('> "isys_lock" ' . $this->m_cmdb_dao->affected_after_update() . ' rows affected.');

        $this->m_cmdb_dao->apply_update();

        return $this;
    }


    /**
     * Method for saving the whole process inside the "Swap CI" archive.
     *
     * @return  isys_swapci_swap
     * @author  Leonard Fischer <lfischer@i-doit.com>
     */
    private function save_to_archive()
    {
        global $g_comp_session;

        $this->m_dao->save_to_archive(
            $g_comp_session->get_user_id(),
            null,
            $this->m_old_obj['id'],
            $this->m_new_obj['id'],
            $this->m_results);

        return $this;
    }


    /**
     * Method for renewing the affected relation objects.
     *
     * @param  array $p_relation_ids
     * @author Van Quyen Hoang <qhoang@i-doit.org>
     */
    private function renew_relation_titles(array $p_relation_ids = [])
    {
        if (count($p_relation_ids) > 0) {
            $l_dao = isys_factory::get_instance('isys_cmdb_dao_category_g_relation', $this->m_db);

            $l_sql = 'SELECT * FROM isys_catg_relation_list
				INNER JOIN isys_obj ON isys_obj__id = isys_catg_relation_list__isys_obj__id
				WHERE isys_catg_relation_list__id ' . $this->m_cmdb_dao->prepare_in_condition($p_relation_ids) . ';';

            $l_res = $l_dao->retrieve($l_sql);

            if (count($l_res) > 0) {
                while ($l_row = $l_res->get_row()) {
                    $l_obj = $l_row['isys_catg_relation_list__isys_obj__id'];
                    $l_master = $l_row['isys_catg_relation_list__isys_obj__id__master'];
                    $l_slave = $l_row['isys_catg_relation_list__isys_obj__id__slave'];
                    $l_relation_type = $l_row['isys_catg_relation_list__isys_relation_type__id'];
                    $l_dao->update_relation_object($l_obj, $l_master, $l_slave, $l_relation_type);

                    $l_old_name = $l_row['isys_obj__title'];
                    $l_new_name = $l_dao->get_obj_name_by_id_as_string($l_row['isys_obj__id']);

                    if ($l_old_name != $l_new_name) {
                        $this->m_log->debug('> Updated old title "' . $l_row['isys_obj__title'] . '" to "' . $l_new_name . '".');
                    }
                }
            }
        } else {
            $this->m_log->debug('> No titles to update...');
        }
    }
}
