/**
 * This "ListSelector" class is used for the "Swap CI" GUI to select Object-types and their categories.
 * But it is possible to feed this class with all sorts of information (just like the "dialog-list").
 *
 * "items" should look like this:
 * [{id:123, const:'C__OBJTYPE__RANDOM', title:'Some random Object type', selected:false}, {...}]
 *
 * @package     i-doit
 * @subpackage  modules
 * @author      Leonard Fischer <lfischer@i-doit.org>
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
var ListSelector = Class.create({
    /**
     * Constructor method, initializes everything necessary.
     * @param  element
     * @param  options
     */
    initialize: function (element, options) {
        // Setting the "root" element for this GUI element.
        this.element = $(element).update();
        // Setting the "receiver" element, this is mandatory!
        this.receiver = $(options.receiver);
        
        this.options = Object.extend({
            title:              '',
            items:              [],
            selection_callback: Prototype.emptyFunction
        }, options || {});
        
        this
            // Building the GUI.
            .render()
            // Check the selected items.
            .check_selected_items()
            // The observer only need to be called once.
            .set_observer();
    },
    
    /**
     * Method for rendering the list.
     * @return  ListSelector
     */
    render: function () {
        var i,
            cnt  = this.options.items.length,
            item,
            list = new Element('ul', {className: 'obj-type-list editmode'}),
            headline;
        
        if (!this.options.title.blank()) {
            headline = new Element('h3', {className: 'p5 border-bottom bg-neutral-200'}).update(this.options.title);
        }
        
        for (i = 0; i < cnt; i++) {
            item = this.options.items[i];
            
            // Somehow "item['const']" works, but item.const causes IE errors.
            list.insert(new Element('li', {'data-id': item.id, 'data-const': item['const']}).update(
                new Element('label')
                    .update(new Element('span').update(item.title))
                    .insert(new Element('input', {type: 'checkbox', className: 'fr'}))
            ));
        }
        
        this.element
            .insert(headline)
            .insert(list)
            .removeClassName('hide')
            .show();
        
        return this;
    },
    
    /**
     * Method for checking all selected items (more dev-friendly and easier to debug in IE).
     * @return  ListSelector
     */
    check_selected_items: function () {
        var i,
            cnt = this.options.items.length,
            item,
            checkbox;
        
        for (i = 0; i < cnt; i++) {
            item = this.options.items[i];
            
            if (item.selected) {
                // Somehow "item['const']" works, but item.const causes IE errors.
                this.element.select('li[data-const="' + item['const'] + '"] input')[0].checked = true;
            }
        }
        
        return this;
    },
    
    /**
     * Method for resetting all observers.
     * @return  ListSelector
     */
    set_observer: function () {
        this.element.stopObserving();
        
        this.element.on('change', 'input', this.change_value.bindAsEventListener(this));
        
        return this;
    },
    
    /**
     * Method which gets called, when clicking an item inside the list.
     * @param  ev
     */
    change_value: function (ev) {
        var selection = this.element.select('input:checked').invoke('up', 'li').invoke('readAttribute', 'data-const');
        
        // Write the selected objects in to our receiver (as JSON string).
        this.receiver.setValue(JSON.stringify(selection));
        
        // Call the callback.
        this.options.selection_callback(ev.findElement());
    }
});
