<?php

/**
 * Event Manager
 *
 * Executes actions triggered by CMDB-Events like creating objects.
 * Currently only generates logbook entries for the defined events, but more
 * actions are possible.
 *
 * SINGLETON
 *
 * @package     i-doit
 * @subpackage  Events
 * @author      Dennis Bluemer <dbluemer@i-doit.org>
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class isys_workflow_event_manager extends isys_event_manager
{
    /**
     * Variable, holding the singleton instance.
     *
     * @static
     * @var  isys_event_manager
     */
    private static $m_instance = null;

    /**
     * isys_import__id
     *
     * @var int
     */
    private $m_import_id = null; // function

    /**
     * Method for retrieving the singleton instance.
     *
     * @static
     * @return  isys_event_manager
     */
    public static function getInstance()
    {
        if (self::$m_instance === null) {
            self::$m_instance = new self();
        }

        return self::$m_instance;
    }

    /**
     * Alert level mapping.
     *
     * @var  array
     */
    protected static $m_alertLevels = [
        'C__LOGBOOK_EVENT__CATEGORY_ARCHIVED'              => C__LOGBOOK__ALERT_LEVEL__1,
        'C__LOGBOOK_EVENT__CATEGORY_ARCHIVED__NOT'         => C__LOGBOOK__ALERT_LEVEL__1,
        'C__LOGBOOK_EVENT__CATEGORY_DELETED'               => C__LOGBOOK__ALERT_LEVEL__2,
        'C__LOGBOOK_EVENT__CATEGORY_DELETED__NOT'          => C__LOGBOOK__ALERT_LEVEL__1,
        'C__LOGBOOK_EVENT__CATEGORY_PURGED'                => C__LOGBOOK__ALERT_LEVEL__3,
        'C__LOGBOOK_EVENT__CATEGORY_PURGED__NOT'           => C__LOGBOOK__ALERT_LEVEL__1,
        'C__LOGBOOK_EVENT__CATEGORY_CHANGED'               => C__LOGBOOK__ALERT_LEVEL__0,
        'C__LOGBOOK_EVENT__CATEGORY_CHANGED__NOT'          => C__LOGBOOK__ALERT_LEVEL__1,
        'C__LOGBOOK_EVENT__CATEGORY_RECYCLED'              => C__LOGBOOK__ALERT_LEVEL__1,
        'C__LOGBOOK_EVENT__CATEGORY_RECYCLED__NOT'         => C__LOGBOOK__ALERT_LEVEL__1,
        'C__LOGBOOK_EVENT__OBJECT_CREATED'                 => C__LOGBOOK__ALERT_LEVEL__0,
        'C__LOGBOOK_EVENT__OBJECT_CREATED__NOT'            => C__LOGBOOK__ALERT_LEVEL__1,
        'C__LOGBOOK_EVENT__OBJECT_CHANGED'                 => C__LOGBOOK__ALERT_LEVEL__0,
        'C__LOGBOOK_EVENT__OBJECT_CHANGED__NOT'            => C__LOGBOOK__ALERT_LEVEL__1,
        'C__LOGBOOK_EVENT__OBJECT_ARCHIVED'                => C__LOGBOOK__ALERT_LEVEL__1,
        'C__LOGBOOK_EVENT__OBJECT_ARCHIVED__NOT'           => C__LOGBOOK__ALERT_LEVEL__1,
        'C__LOGBOOK_EVENT__OBJECT_DELETED'                 => C__LOGBOOK__ALERT_LEVEL__2,
        'C__LOGBOOK_EVENT__OBJECT_DELETED__NOT'            => C__LOGBOOK__ALERT_LEVEL__1,
        'C__LOGBOOK_EVENT__OBJECT_PURGED'                  => C__LOGBOOK__ALERT_LEVEL__3,
        'C__LOGBOOK_EVENT__OBJECT_PURGED__NOT'             => C__LOGBOOK__ALERT_LEVEL__1,
        'C__LOGBOOK_EVENT__OBJECT_RECYCLED'                => C__LOGBOOK__ALERT_LEVEL__1,
        'C__LOGBOOK_EVENT__OBJECT_RECYCLED__NOT'           => C__LOGBOOK__ALERT_LEVEL__1,
        'C__LOGBOOK_EVENT__POBJECT_MALE_PLUG_CREATED__NOT' => C__LOGBOOK__ALERT_LEVEL__1,
        'C__LOGBOOK_EVENT__OBJECTTYPE_CREATED'             => C__LOGBOOK__ALERT_LEVEL__1,
        'C__LOGBOOK_EVENT__OBJECTTYPE_CREATED__NOT'        => C__LOGBOOK__ALERT_LEVEL__1,
        'C__LOGBOOK_EVENT__OBJECTTYPE_CHANGED'             => C__LOGBOOK__ALERT_LEVEL__1,
        'C__LOGBOOK_EVENT__OBJECTTYPE_CHANGED__NOT'        => C__LOGBOOK__ALERT_LEVEL__1,
        'C__LOGBOOK_EVENT__OBJECTTYPE_ARCHIVED'            => C__LOGBOOK__ALERT_LEVEL__1,
        'C__LOGBOOK_EVENT__OBJECTTYPE_ARCHIVED__NOT'       => C__LOGBOOK__ALERT_LEVEL__1,
        'C__LOGBOOK_EVENT__OBJECTTYPE_DELETED'             => C__LOGBOOK__ALERT_LEVEL__2,
        'C__LOGBOOK_EVENT__OBJECTTYPE_DELETED__NOT'        => C__LOGBOOK__ALERT_LEVEL__1,
        'C__LOGBOOK_EVENT__OBJECTTYPE_PURGED'              => C__LOGBOOK__ALERT_LEVEL__3,
        'C__LOGBOOK_EVENT__OBJECTTYPE_PURGED__NOT'         => C__LOGBOOK__ALERT_LEVEL__1,
        'C__LOGBOOK_EVENT__OBJECTTYPE_RECYCLED'            => C__LOGBOOK__ALERT_LEVEL__1,
        'C__LOGBOOK_EVENT__OBJECTTYPE_RECYCLED__NOT'       => C__LOGBOOK__ALERT_LEVEL__1,
        'C__LOGBOOK_ENTRY__WORKFLOW_CREATED'               => C__LOGBOOK__ALERT_LEVEL__0,
        'C__LOGBOOK_EVENT__WORKFLOW_ACCEPTED'              => C__LOGBOOK__ALERT_LEVEL__0,
        'C__LOGBOOK_EVENT__WORKFLOW_CANCELLED'             => C__LOGBOOK__ALERT_LEVEL__0,
        'C__LOGBOOK_EVENT__WORKFLOW_COMPLETED'             => C__LOGBOOK__ALERT_LEVEL__0,
        'C__LOGBOOK_ENTRY__TEMPLATE_APPLIED'               => C__LOGBOOK__ALERT_LEVEL__1
    ];

    /**
     * Manages an event affecting the Workflows by creating an entry in the logbook.
     *
     * @param   string $p_strConstEvent   The event constant
     * @param   string $p_strDesc         The description of the logbook entry to create
     * @param   int $p_wID             The ID of the affected workflow, if applicable
     * @param   int $p_wTypeID         The type ID of the affected workflow, if applicable
     * @param   string $p_title           The title of the event
     * @param   null $p_changes
     * @param   null $p_comment
     *
     * @return  boolean
     */
    public function triggerWorkflowEvent($p_strConstEvent, $p_strDesc, $p_wID = null, $p_wTypeID = null, $p_title = null, $p_changes = null, $p_comment = null)
    {
        $database = isys_application::instance()->container->get('database');
        $l_workflow_dao = new isys_workflow_dao($database);
        $l_workflow_type_dao = new isys_workflow_dao_type($database);
        $l_wTitle = $l_workflow_dao->get_title_by_id($p_wID);
        $l_workflow_type__id = $l_workflow_dao->get_workflow_type_by_id($p_wID);
        $l_wTypeTitle = $l_workflow_type_dao->get_title_by_id($l_workflow_type__id);

        $l_daoLogbook = new isys_component_dao_logbook($database);

        // Set entry in the logbook.
        return $l_daoLogbook->set_entry($p_strConstEvent, $p_strDesc, isys_glob_datetime(), self::$m_alertLevels[$p_strConstEvent], null, $l_wTitle, $l_wTypeTitle, null,
            C__LOGBOOK_SOURCE__INTERNAL, $p_changes, $p_comment);
    }

    /**
     * Method for translating the current event.
     *
     * @param   string $p_strEvent
     * @param   string $p_name
     * @param   string $p_category
     * @param   string $p_objType
     * @param   string $p_entry_identifier
     * @param    int   $p_changed_entries
     *
     * @return  string
     */
    public function translateEvent($p_strEvent, $p_name, $p_category, $p_objType, $p_entry_identifier = null, $p_changed_entries = 0)
    {
        $languageManager = isys_application::instance()->container->get('language');

        $l_entry_lc = 'LC__LOGBOOK__CATEGORY_ENTRY';
        if ($p_changed_entries > 1) {
            $l_entry_lc = sprintf(isys_application::instance()->container->get('language')
                ->get('LC__LOGBOOK__CATEGORY_ENTRIES'), $p_changed_entries);
        }
        if (isset($p_entry_identifier) && !empty($p_entry_identifier)) {
            $l_entry_lc = 'LC__LOGBOOK__SPECIFIC_CATEGORY_ENTRY';
        }
        $message = null;
        switch ($p_strEvent) {
            case 'C__LOGBOOK_ENTRY__WORKFLOW_CREATED':
                $message = $languageManager->get($p_objType) . ' "' . $p_name . '" ' . $languageManager->get('LC__LOGBOOK__OBJECT_CREATED');
                break;
            case 'C__LOGBOOK_EVENT__WORKFLOW_ACCEPTED':
                $message = $languageManager->get($p_objType) . ' "' . $p_name . '" ' . $languageManager->get('LC__LOGBOOK__WORKFLOW_HAS_BEEN_ACCEPTED');
                break;
            case 'C__LOGBOOK_EVENT__WORKFLOW_CANCELLED':
                $message = $languageManager->get($p_objType) . ' "' . $p_name . '" ' . $languageManager->get('LC__LOGBOOK__WORKFLOW_HAS_BEEN_CANCELLED');
                break;
            case 'C__LOGBOOK_EVENT__WORKFLOW_COMPLETED':
                $message = $languageManager->get($p_objType) . ' "' . $p_name . '" ' . $languageManager->get('LC__LOGBOOK__WORKFLOW_HAS_BEEN_CONDUCTED');
                break;
            default:
                $message = parent::translateEvent($p_strEvent, $p_name, $p_category, $p_objType, $p_entry_identifier = null, $p_changed_entries = 0);
                break;
        }
        return $message;
    }
}
